/********************************************************************************
*                                                                               *
*                     Database field handling                                   *
*                                                                               *
*********************************************************************************
* Copyright (C) 2003 by Mathew Robertson.       All Rights Reserved.            *
* Copyright (C) 2003 by Giancarlo Formicuccia.  All Rights Reserved.            *
*********************************************************************************
* This library is free software; you can redistribute it and/or                 *
* modify it under the terms of the GNU Lesser General Public                    *
* License as published by the Free Software Foundation; either                  *
* version 2.1 of the License, or (at your option) any later version.            *
*                                                                               *
* This library is distributed in the hope that it will be useful,               *
* but WITHOUT ANY WARRANTY; without even the implied warranty of                *
* MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU             *
* Lesser General Public License for more details.                               *
*                                                                               *
* You should have received a copy of the GNU Lesser General Public              *
* License along with this library; if not, write to the Free Software           *
* Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA.    *
*********************************************************************************/
#include <config.h>
#include <fox/fxver.h>
#include <fox/xincs.h>
#include <fox/fxdefs.h>
#include <fox/FXStream.h>
#include <fox/FXString.h>
#include <fox/FXWString.h>
#include <fox/FXCharset.h>
#include <fox/FXSize.h>
#include <fox/FXPoint.h>
#include <fox/FXRectangle.h>
#include <fox/FXRegistry.h>
#include <fox/FXApp.h>
using namespace FX;
#include "fxexdefs.h"
#include "FXArray.h"
#include "FXDatabaseField.h"
#include "FXDatabaseQuery.h"
#include "FXDatabaseInterface.h"
using namespace FXEX;
namespace FXEX {

FXDEFMAP(FXDatabaseField) FXDatabaseFieldMap[] = {
  FXMAPFUNC(SEL_EVENT,        FXDatabaseField::ID_EVENT,          FXDatabaseField::onDispatch),
  FXMAPFUNC(SEL_EVENT,        FXDatabaseField::ID_SETFIELDTYPE,   FXDatabaseField::onSetFieldType),
  FXMAPFUNC(SEL_EVENT,        FXDatabaseField::ID_SETDATA,        FXDatabaseField::onSetData),
  FXMAPFUNC(SEL_EVENT,        FXDatabaseField::ID_GETDATA,        FXDatabaseField::onGetData),
  FXMAPFUNC(SEL_EVENT,        FXDatabaseField::ID_REFRESHOLD,     FXDatabaseField::onRefreshOld)
};

FXIMPLEMENT(FXDatabaseField, FXBaseObject, FXDatabaseFieldMap, ARRAYNUMBER(FXDatabaseFieldMap));

FXDatabaseField::FXDatabaseField(FXDatabaseQuery *qry): FXBaseObject(qry->getApp(), NULL, 0) {
  this->qry = qry;
  fType.fxType = FXVariant::DT_UNASSIGNED;
  fType.nullable = FALSE;
  fType.name = "";
}

void FXDatabaseField::checkBind() const {
  if(!qry || !qry->isOpen()) FXDatabaseInterface::dbThrow("Field not bound", -1);
}

long FXDatabaseField::onDispatch(FXObject *, FXSelector, void *data) {
  FXdbEvent *ev = (FXdbEvent *) data;
  ev->dbf = this;
  switch(ev->type) {
    case dbCancelUpdate:
      value = oldValue;
      oldValue.setType(FXVariant::DT_UNASSIGNED);
      ev->data = value;
      break;
    case dbUpdate:
      oldValue.setType(FXVariant::DT_UNASSIGNED);
      break;
    case dbAddNew:
      oldValue = value;
      value.setType(FXVariant::DT_UNASSIGNED);
      if(fType.readOnly) ev->data = FXString("<read only>");
      else ev->data.setType(FXVariant::DT_UNASSIGNED);
      break;
    case dbEdit:
      oldValue = value;
      break;
    case dbDelete:
      value.setType(FXVariant::DT_UNASSIGNED);
      break;
    case dbRefresh:
      ev->data = value;
      break;
    case dbAskData: /* Special care for this one */
      if(target && !fType.readOnly) {
        ev->data.setType(FXVariant::DT_UNASSIGNED);
        target->handle(this, MKUINT(message, SEL_EVENT), ev);
        if(!ev->data.isUnassigned()) {
          if(!ev->data.isNull()) ev->data.convertTo(fType.fxType);
          if((ev->state!=rsAddNew) && (ev->data==oldValue))
            ev->data.setType(FXVariant::DT_UNASSIGNED);
          /* Warning - may throw exceptions (size owfl etc) */
          else value = convertData(ev->data);
        }
      } else ev->data.setType(FXVariant::DT_UNASSIGNED);
      return 1;
    default: break;
  }
  if(target) target->handle(this, MKUINT(message, SEL_EVENT), ev);
  return 1;
}

long FXDatabaseField::onRefreshOld(FXObject *, FXSelector, void *data) {
  FXdbEvent *ev = (FXdbEvent *) data;
  if(target) {
    ev->dbf = this;
    ev->data = oldValue;
    ev->type = dbRefresh;
    target->handle(this, MKUINT(message, SEL_EVENT), ev);
  }
  return 1;
}

void FXDatabaseField::save(FXStream& store) const {
  FXBaseObject::save(store);
  oldValue.save(store);
  value.save(store);
  store << qry;
  store << fType.fxType;
  store << fType.nullable;
  store << fType.name;
  store << fType.maxSize;
  store << fType.varSize;
}

void FXDatabaseField::load(FXStream& store) {
  FXBaseObject::load(store);
  oldValue.load(store);
  value.load(store);
  store >> qry;
  store >> fType.fxType;
  store >> fType.nullable;
  store >> fType.name;
  store >> fType.maxSize;
  store >> fType.varSize;
}

FXDatabaseField::operator FXVariant() {
  return getValue();
}

const FXVariant &FXDatabaseField::getValue() const {
  checkBind();
  return value;
}

FXVariant FXDatabaseField::convertData(const FXVariant &src) {
  FXASSERT(qry);
  FXASSERT(!src.isUnassigned());
  if(src.isNull()) {
    if(!fType.nullable) FXDatabaseInterface::dbThrow("Cannot set field to NULL", -1);
    return src;
  } else {
    FXVariant tmp(src);
    tmp.convertTo(fType.fxType);
    if(fType.maxSize && tmp.getSize()>fType.maxSize) FXDatabaseInterface::dbThrow("Field size overflow", -1);
    return tmp;
  }
}

FXDatabaseField &FXDatabaseField::operator=(const FXVariant &v) {
  checkBind();
  if(!qry->isEditable()) FXDatabaseInterface::dbThrow("Query not editable", -1);
  if(fType.readOnly) FXDatabaseInterface::dbThrow("Field is read-only", -1);
  value = convertData(v);
  return *this;
}

long FXDatabaseField::onSetFieldType(FXObject *, FXSelector, void *data) {
  fType = *(FXFieldType *) data;
  return 1;
}

long FXDatabaseField::onSetData(FXObject *, FXSelector, void *data) {
  if(data) value = *(FXVariant *) data;
  else value.setType(FXVariant::DT_UNASSIGNED);
  return 1;
}

long FXDatabaseField::onGetData(FXObject *, FXSelector, void *data) {
  *(FXVariant *) data = value;
  return 1;
}

}
