/*

    WiRouter KeyRec - Salvatore Fresta

    Contact: salvatorefresta@gmail.com

    Copyright 2010-2011 Salvatore Fresta aka Drosophila

    This program is free software; you can redistribute it and/or
    modify it under the terms of  the  GNU General Public License
    as published by the  Free Software Foundation; either version
    2 of the License, or (at your option) any later version.

    This program  is  distributed  in the hope  that  it  will be
    useful, but WITHOUT ANY WARRANTY;  without  even the  implied
    warranty  of  MERCHANTABILITY  or  FITNESS  FOR  A PARTICULAR
    PURPOSE. See the GNU General Public License for more details.

    You should have  received a copy  of  the  GNU General Public
    License along  with  this program;  if not, write to the Free
    Software Foundation,Inc., 59 Temple Place, Suite 330, Boston,
    MA 02111-1307 USA

    http://www.gnu.org/licenses/gpl-2.0.txt

*/

#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <errno.h>
#include "teletu.h"
#include "socket.h"
#include "sha256.h"
#include "wirouterkeyrec.h"

extern int errno; 


/*
    Update the Teletu configuration file
*/

int teletu_update_config_file(unsigned char *filename, unsigned char *host, int port, unsigned char *url, char *delone, char *deltwo, int debug) {
  
    FILE *fp = NULL;
    
    int sd,
        pkglen;
    unsigned char *pkg     = NULL,
                  *rec     = NULL,
                  *payload = NULL;

  
    printf("\nUpdating Teletu configuration file\nConnecting to %s...", host);

    if((sd = socket_connect(host, port)) < 0) {
      if(debug) PRINT_DEBUG("Connection failed.\n\n");
      return -1;
    }

    printf("\nConnection estabilished.");
  
    pkglen = vspr((char **)&pkg, "GET %s HTTP/1.1\r\n"
                                 "Host: %s\r\n\r\n", url, host);
  
    if(send(sd, (char *)pkg, pkglen, 0) < 0) {
      if(debug) PRINT_DEBUG("Sending failed.\n\n");
      close(sd);
      free(pkg);
      return -1;
    }
  
    free(pkg);

    if(!(rec = socket_receive(sd, 4))) {
      if(debug) PRINT_DEBUG("Receive failed.\n\n");
      close(sd);
      return -1;
    }
  
    if(delone && deltwo) {
      payload = (unsigned char *) grab((char *)rec, delone, deltwo, 0, 0);
      if(!payload) {
	if(debug) PRINT_DEBUG("Unable to grab the payload.\n\n");
        return -1;
      }
    }
    else {
      payload = (unsigned char *) strstr((char *)rec, "\r\n\r\n");
      if(!payload) {
	if(debug) PRINT_DEBUG("Unable to get the payload.\n\n");
        return -1;
      }
    }

    close(sd);

    printf("\nConnection closed.");

    fp = fopen((char *)filename, "w+");
    if(!fp) {
      printf("\nUnable to open the configuration file [Errno: %s].\n\n", strerror(errno));
      return -1;
    }
  
    fprintf(fp, "%s", (char *)payload);
    fclose(fp);
  
    free(rec);
    
    printf("\nUpdate complete.\nCheck the content of %s\n\n", filename);
  
    return 0;
  
}



Teletu_config *teletu_get_config(unsigned char *mac_in, unsigned char *error, unsigned char *config_file, int debug) {

   Teletu_config *config_list = NULL;

    unsigned int mac_in_end = 0,
                 mac_start  = 0,
                 mac_end    = 0,
                 base       = 0,
                 base_old   = 0,
                 sn         = 0,
                 sn_old     = 0,
                 incr       = 0,
                 incr_old   = 0,
                 step       = 0,
                 t          = 0,
                 found      = 0;

    unsigned char *filename  = NULL,
                  *buffer    = NULL,
                  *substring = NULL,
                  wpa_key[15];

    FILE *fp = NULL;


    /*
        Set the configuration file
    */
    if(strlen((char *)config_file) > 0)
      filename = config_file;
    else {

      if(!wr_check_file_exist((unsigned char *) TELETU_DEFAULT_CONFIG_FILE))
        filename = (unsigned char *) TELETU_DEFAULT_CONFIG_FILE;
      else
        filename = (unsigned char *) TELETU_DEFAULT_CONFIG_FILE_DIR;

    }

    if(debug) PRINT_DEBUG("[TeleTu-%s] Configuration file %s.", mac_in, filename);

    fp = fopen((char *)filename, "rb");
    if(!fp) {
      PRINT_ERROR("Unable to open the TeleTu configuration file [Errno: %s]", strerror(errno));
      return NULL;
    }
    
    sscanf((char *)mac_in+9, "%02x:", &t);
    mac_in_end = t << 16;
      
    sscanf((char *)mac_in+12, "%02x:", &t);
    mac_in_end |= t << 8;
      
    sscanf((char *)mac_in+15, "%02x:", &t);
    mac_in_end |= t;

readfile:

    while((buffer = wr_myfgets(fp))) {

      if(buffer[0] == 0x23) continue; // On comment read the next line
	
      for(step=0; step<=5; step++) {

          if(!step)
              substring = (unsigned char *) strtok((char *)buffer, " ");
          else
              substring = (unsigned char *) strtok(NULL, " ");

          if(!substring) break;

          switch(step) {	    

              /*
                  Checking the first three bytes of the input mac address
              */
              case 0:
                  if(strncmp((char *)substring, (char *)mac_in, 8)) goto readfile;
              break;

              /*
                  Extraction of the start mac range
              */
              case 1:
		
		if(strlen((char *)substring) < 8) goto readfile;
                
                sscanf((char *)substring, "%02x:", &t);
                mac_start = t << 16;
                
		sscanf((char *)substring+3, "%02x:", &t);
		mac_start |= t << 8;
		
		sscanf((char *)substring+6, "%02x:", &t);
		mac_start |= t;
		
              break;

              /*
                  Extraction of the end mac range
              */
              case 2:

                if(strlen((char *)substring) < 8) goto readfile;
                
                sscanf((char *)substring, "%02x:", &t);
                mac_end = t << 16;
                
		sscanf((char *)substring+3, "%02x:", &t);
		mac_end |= t << 8;
		
		sscanf((char *)substring+6, "%02x:", &t);
		mac_end |= t;

              break;

              /*
                  Extraction of the serial number
              */
              case 3:
                  sn = (unsigned int) atoi((char *)substring);
              break;

              /*
                  Extaction of the base
              */
              case 4:

                if(strlen((char *)substring) < 6) goto readfile;

                sscanf((char *)substring, "%02x", &t);
                base = t << 16;
                
		sscanf((char *)substring+2, "%02x", &t);
		base |= t << 8;
		
		sscanf((char *)substring+4, "%02x", &t);
		base |= t;   

              break;
	      
	      /*
                  Extraction of the increment value
              */
              case 5:
                  incr = (unsigned int) atoi((char *)substring);
              break;

              default: break;

          }


      }
      
      
      /*
	  Checking if the last three bytes of mac address are between two consecutive rows
      */
      
      if(mac_in_end >= mac_end) {
	base_old = base;
        sn_old = sn;
	incr_old = incr;
	continue;
      }
      
      if(mac_in_end <= mac_start && sn_old != 0 && base_old != 0 && ((mac_in_end-base_old) % incr_old == 0) && ((base-mac_in_end) % incr == 0)) {
	
	snprintf((char *)wpa_key, sizeof(wpa_key), "%dY%07d", sn_old, (mac_in_end-base_old)/incr_old);
        config_list = teletu_config_add(config_list, wpa_key, base_old, sn_old, mac_in);
	
	snprintf((char *)wpa_key, sizeof(wpa_key), "%dY%07d", sn, (base-mac_in_end)/incr);
        config_list = teletu_config_add(config_list, wpa_key, base, sn, mac_in);
	
	found = 1;
	
      }
      
      
      /*
	  Checking if the last three bytes of mac address are between the range (reliability: 100%)
      */
      
      if(mac_in_end >= mac_start && mac_in_end <= mac_end && ((mac_in_end-base) % incr == 0)) {
	
	snprintf((char *)wpa_key, sizeof(wpa_key), "%dY%07d", sn, (mac_in_end-base)/incr);
	config_list = teletu_config_add(config_list, wpa_key, base, sn, mac_in);
	    
	found = 1;
	    
      }
      
      sn_old   = 0;
      base_old = 0;
      incr_old = 0;

    }

    if(!found) strncpy((char *)error, "Router not present in the config file.", ERROR_LENGTH-1);
    
     /*
          WPA Key Not Found
      */
      if(!found) {
	config_list = teletu_config_add(config_list, (unsigned char *)"key not found!", 0, 0, mac_in);
        if(!config_list)
          PRINT_FATAL_ERROR("Unable to allocate the memory for the configuration of %s [Errno: %s].", mac_in, strerror(errno));
      }


    fclose(fp);

    return config_list;

}



/*
    Deletes the specified Teletu_config list
*/

Teletu_config *teletu_config_free(Teletu_config *first) {

  Teletu_config *tmp = NULL;

  while(first) {
      tmp = first;
      first = first->next;
      free(tmp);
    }

    return first;

}



/*
    Creates a new record of Teletu_config type and attach
    it at the end of the specified Teletu_config list
*/

Teletu_config *teletu_config_add(Teletu_config *first, unsigned char wpa_key[15], unsigned int base, unsigned int sn, unsigned char mac[18]) {

    Teletu_config *new = NULL,
                  *tmp = NULL;

    if(first) for(tmp=first; tmp->next; tmp=tmp->next);

    new = (Teletu_config *) calloc(1, sizeof(Teletu_config));
    if(!new) return first;

    strncpy((char *)new->wpa_key, (char *)wpa_key, 14);
    strncpy((char *)new->mac, (char *)mac, 17);
    new->base = base;
    new->sn   = sn;
    new->next = NULL;

    if(first)
      tmp->next = new;
    else
      first = new;

    return first;

}


