/*
 * Copyright (c) 2024 NVIDIA CORPORATION & AFFILIATES. All rights reserved.
 *
 * This software is available to you under a choice of one of two
 * licenses.  You may choose to be licensed under the terms of the GNU
 * General Public License (GPL) Version 2, available from the file
 * COPYING in the main directory of this source tree, or the
 * OpenIB.org BSD license below:
 *
 *     Redistribution and use in source and binary forms, with or
 *     without modification, are permitted provided that the following
 *     conditions are met:
 *
 *      - Redistributions of source code must retain the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer.
 *
 *      - Redistributions in binary form must reproduce the above
 *        copyright notice, this list of conditions and the following
 *        disclaimer in the documentation and/or other materials
 *        provided with the distribution.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS
 * BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
 * ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN
 * CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 *
 */



#pragma once

// std
#include <chrono>
#include <cstdint>
#include <limits>
#include <vector>
// 3rd party
#include "nlohmann/json.hpp"
// this project
#include "common/json_utils.h"

/**
 * Histogram, literetly
 */
template <typename BucketType = std::int64_t, typename CounterType = BucketType>
struct Histogram
{
    using Bucket = BucketType;
    using Counter = CounterType;

    using Buckets = std::vector<Bucket>;
    using Counters = std::vector<Counter>;

    // when we want the histogram to take into account all possible counters,
    // we put this value as the last backet, it is a big-enough-value for our histograms
    static BucketType bucket_infinity;

    Histogram() = default;

    //! ctor.
    explicit Histogram(Buckets a_buckets) : buckets{std::move(a_buckets)}
    {
        // create as much counters as we have buckets
        counters.resize(buckets.size());
    }

    //! Add N times a value, to the histogram
    void Add(Bucket value, Counter n = 1) { Update(value, n); }

    //! Remove N times a value, from the histogram
    void Remove(Bucket value, Counter n = 1) { Update(value, -n); }

    //! Update histogrm with N tims a value (could be negative times)
    void Update(Bucket value, Counter n)
    {
        auto bucket_it = std::lower_bound(std::begin(buckets), std::end(buckets), value);
        if (bucket_it != std::end(buckets)) {
            auto bucket_index = std::distance(std::begin(buckets), bucket_it);
            counters[bucket_index] += n;
        }
    }

    Buckets buckets;
    Counters counters;
};

// define bucket_infinity (static member)
template <typename BucketType, typename CounterType>
BucketType Histogram<BucketType, CounterType>::bucket_infinity{1000000000};   // over 30 years worth of seconds

// we will work with these histograms
using IntegralHistogram = Histogram<std::int64_t, std::int64_t>;
using HoursHistogram = Histogram<std::chrono::duration<float, std::ratio<3600>>, std::int64_t>;

// supprt JSON serialization
template <typename BucketType, typename CounterType>
void to_json(nlohmann::ordered_json& j, const Histogram<BucketType, CounterType>& h);
template <typename BucketType, typename CounterType>
void from_json(const nlohmann::ordered_json& j, Histogram<BucketType, CounterType>& h);

// NOTE:
//   uncomment this code, and remove the defintions above,
//   when we serialize histograms as buckets + counters arrays
//
// template <typename BucketType, typename CounterType>
// void to_json(nlohmann::json& j, const Histogram<BucketType, CounterType>& h)
// {
//     j = nlohmann::json{{"buckets", h.buckets}, {"counters", h.counters}};
// }
// template <typename BucketType, typename CounterType>
// void from_json(const nlohmann::json& j, Histogram<BucketType, CounterType>& h)
// {
//     j.at("buckets").get_to(h.buckets);
//     j.at("counters").get_to(h.counters);
// }

/**
 * telemetry mertics configuratiopn
 */
struct MetricsConfig
{
    HoursHistogram::Buckets ended_jobs_duration_in_hours_buckets;
    IntegralHistogram::Buckets active_jobs_num_hcas_buckets;
    IntegralHistogram::Buckets trees_level_buckets;
};

void to_json(nlohmann::ordered_json& j, const MetricsConfig& m);
void from_json(const nlohmann::ordered_json& j, MetricsConfig& m);

// use this when no we use non-orderd JSON
// NLOHMANN_DEFINE_TYPE_NON_INTRUSIVE(MetricsConfig, ended_jobs_duration_in_hours_buckets, active_jobs_num_hcas_buckets,
// trees_level_buckets);
