/* police.c
 *
 * This file is part of httpd.
 *
 * 02/17/96 			Michael Temari, <temari@ix.netcom.com>
 * 07/07/96 Initial Release	Michael Temari, <temari@ix.netcom.com>
 *
 */
#include <sys/types.h>
#include <sys/stat.h>
#include <ctype.h>
#include <stdio.h>
#include <string.h>
#include <stdlib.h>
#include <time.h>
#include <fcntl.h>
#include <unistd.h>
#include <errno.h>

#include "http.h"
#include "utility.h"
#include "config.h"
#include "pass.h"

_PROTOTYPE(static int authaccess, (struct http_request *rq, struct http_reply *rp));
_PROTOTYPE(static void purl, (struct http_request *rq, struct http_reply *rp));

static int authaccess(rq, rp)
struct http_request *rq;
struct http_reply *rp;
{
struct authuser *pu;

   /* no authorization so no access to anyone */
   if(rp->auth == NULL) {
   	rp->status = HTTP_STATUS_FORBIDDEN;
   	strcpy(rp->statusmsg, "No Access to file");
   	return(-1);
   }

   /* no password file so it is a free for all */
   if(rp->auth->passwdfile == NULL)
	return(0);

   /* they did not give us an authorized user */
   if(rq->authuser[0] == '\0') {
   	rp->status = HTTP_STATUS_UNAUTHORIZED;
   	strcpy(rp->statusmsg, "No Access to file");
   	return(-1);
   }

   /* check if user okay */
   pu = rp->auth->users;
   if(pu == NULL)
	;	/* no user list we allow anyone in file */
   else {
	while(pu != NULL) {
		if(!strcmp(pu->user, rq->authuser))
			break;
		pu = pu->next;
	}
	/* user is not in list so no access */
	if(pu == NULL) {
		rp->status = HTTP_STATUS_UNAUTHORIZED;
		strcpy(rp->statusmsg, "Forbidden User not authorized");
		return(-1);
	}
   }

   /* check if password file exists, if not no access */
   if(passfile(rp->auth->passwdfile)) {
   	rp->status = HTTP_STATUS_UNAUTHORIZED;
   	strcpy(rp->statusmsg, "Invalid passwd file");
   	return(-1);
   }

   /* check if user in password file, if not no access */
   if(passuser(rp->auth->passwdfile, rq->authuser)) {
	rp->status = HTTP_STATUS_UNAUTHORIZED;
	strcpy(rp->statusmsg, "Forbidden Bad User");
	return(-1);
   }

   /* check if a password exists, if not no access */
   if(passnone(rp->auth->passwdfile, rq->authuser)) {
	rp->status = HTTP_STATUS_UNAUTHORIZED;
	strcpy(rp->statusmsg, "Forbidden no password");
	return(-1);
   }


   /* check if password matches, if not no access */
   if(passpass(rp->auth->passwdfile, rq->authuser, rq->authpass)) {
	rp->status = HTTP_STATUS_UNAUTHORIZED;
	strcpy(rp->statusmsg, "Forbidden bad password");
	return(-1);
   }

   /* whew, all the checks passed so I guess we let them have it */
   return(0);
}

int police(rq, rp)
struct http_request *rq;
struct http_reply *rp;
{
int size;
struct stat st;
struct dirsend *ds;

   purl(rq, rp);

   rp->mtype = "text/html";

#ifdef DEBUG
   fprintf(stderr, "httpd: Trying %s\n", rp->realurl);
#endif

   if(stat(rp->realurl, &st)) {
   	if(errno == EACCES)
   		rp->status = HTTP_STATUS_FORBIDDEN;
   	else
		rp->status = HTTP_STATUS_NOT_FOUND;
	strcpy(rp->statusmsg, strerror(errno));
	if(rq->method != HTTP_METHOD_PUT || rp->status != HTTP_STATUS_NOT_FOUND)
		return(0);
   }

   /* If it is a directory do the appropriate thang! */
   if(rq->method == HTTP_METHOD_GET || rq->method == HTTP_METHOD_HEAD)
   if((st.st_mode & S_IFMT) == S_IFDIR) {
	if(rq->url[strlen(rq->url) - 1] != '/') {
		strncat(rq->url, "/", sizeof(rq->url) - strlen(rq->url));
		rp->status = HTTP_STATUS_MOVED_TEMP;
		sprintf(rp->statusmsg, "Moved to %s", rq->url);
		return(0);
	}
	size = strlen(rq->url);
	ds = dirsend;
	while(ds != NULL) {
		strncpy(rq->url+size, ds->file, sizeof(rq->url)-size);
		purl(rq, rp);
		if(stat(rp->realurl, &st)) {
			if(errno == EACCES)
				rp->status = HTTP_STATUS_FORBIDDEN;
			else
			if(errno != ENOENT)
				rp->status = HTTP_STATUS_NOT_FOUND;
		} else
			break;
		if(rp->status != HTTP_STATUS_OK) {
			strcpy(rp->statusmsg, strerror(errno));
			return(0);
		}
		ds = ds->next;
	}
	if(ds == NULL) {
		rq->url[size] = '\0';
		purl(rq, rp);
		if(stat(rp->realurl, &st)) {
	   		if(errno == EACCES)
   				rp->status = HTTP_STATUS_FORBIDDEN;
   			else
				rp->status = HTTP_STATUS_NOT_FOUND;
			strcpy(rp->statusmsg, strerror(errno));
			return(0);
		}
	}
   }

   /* now check authorizations */
   if(authaccess(rq, rp)) {
#if 0
   	rp->status = HTTP_STATUS_UNAUTHORIZED;
   	strcpy(rp->statusmsg, "No Access to file");
#endif
	return(0);
   }

   if(rq->method == HTTP_METHOD_PUT && !(rp->urlaccess & URLA_WRITE)) {
	rp->status = HTTP_STATUS_METHOD_NOT_ALLOWED;
	strcpy(rp->statusmsg, "Method not allowed");
	return(0);
   }

   if(rp->status == HTTP_STATUS_OK) {
	/* Here is where we check if it is a program or script to run */
	if(cgiexec(rq, rp))
   		return(0);

	if((st.st_mode & S_IFMT) == S_IFDIR) {
		rp->status = HTTP_STATUS_NOT_FOUND;
		strcpy(rp->statusmsg, "Directory listing not available");
		return(0);
	}

	if((st.st_mode & S_IFMT) != S_IFREG) {
		rp->status = HTTP_STATUS_NOT_FOUND;
		strcpy(rp->statusmsg, "Not a regular file");
		return(0);
	}
   }

   /* open the URL for updating */
   if(rq->method == HTTP_METHOD_PUT) {
	rp->status = HTTP_STATUS_OK;
	strcpy(rp->statusmsg, "OK");
	rp->ofd = open(rp->realurl, O_WRONLY | O_CREAT | O_TRUNC);
	if(rp->ofd < 0) {
   		if(errno == EACCES)
   			rp->status = HTTP_STATUS_FORBIDDEN;
   		else
			rp->status = HTTP_STATUS_NOT_FOUND;
		strcpy(rp->statusmsg, strerror(errno));
		return(0);
	}
	return(0);
   }

   if(!(rp->urlaccess & URLA_READ)) {
   	rp->status = HTTP_STATUS_FORBIDDEN;
   	strcpy(rp->statusmsg, "No way...");
	return(0);
   }

   rp->mtype = mimetype(rp->realurl);

   rp->size = st.st_size;
   rp->modtime = st.st_mtime;

   /* open the url if it is a file */
   rp->fd = open(rp->realurl, O_RDONLY);
   if(rp->fd < 0) {
	if(errno == EACCES)
   		rp->status = HTTP_STATUS_FORBIDDEN;
   	else
		rp->status = HTTP_STATUS_NOT_FOUND;
	strcpy(rp->statusmsg, strerror(errno));
	return(0);
   }

   return(0);
}

static void purl(rq, rp)
struct http_request *rq;
struct http_reply *rp;
{
struct vpath *pv;
int gotreal, gotperm;
char *p;
int match;
int len;

   gotreal = 0; gotperm = 0;

#ifdef DEBUG
   fprintf(stderr, "httpd: Processing url = \"%s\"\n", rq->url);
#endif
   for(pv = vpath; pv != NULL; pv = pv->next) {
   	len = strlen(pv->from) - 1;
   	if(pv->from[len] == '*')
   		if(len == 0)
   			match = 1;
   		else
   			match = !strncmp(rq->url, pv->from, len);
   	else
   		if(!strcmp(rq->url, pv->from))
   			match = 2;
   		else
   			match = 0;
#ifdef DEBUG
   	fprintf(stderr, "httpd: Trying \"%s\" %d %d %d %s\n",
		pv->from, match, gotreal, gotperm, pv->auth->name);
#endif
   	if(match) {
		gotperm = 1;
		rp->auth = pv->auth;
		if(pv->urlaccess == -1 && rp->auth != NULL)
			rp->urlaccess = rp->auth->urlaccess;
		else
			rp->urlaccess = pv->urlaccess;
   		if(strcmp(pv->to, ".")) {
   			gotreal = 1;
   			strncpy(rp->realurl, pv->to, sizeof(rp->realurl));
			rp->realurl[sizeof(rp->realurl)-1] = '\0';
			if(match != 2) {
   				strncat(rp->realurl, rq->url+len, sizeof(rp->realurl) - strlen(rp->realurl));
				rp->realurl[sizeof(rp->realurl)-1] = '\0';
			}
   		}
   	}
   	if(match == 2) break;
   }

   if(rp->urlaccess == -1) rp->urlaccess = mkurlaccess("");

   if(!gotreal) {
	strncpy(rp->realurl, rq->url, sizeof(rp->realurl));
	rp->realurl[sizeof(rp->realurl)-1] = '\0';
   }

   if(!gotperm)
   	rp->auth = NULL;

   /* remove any .. references */
   p = rp->realurl;
   while(*p) {
	while(*p && *p != '/') p++;
	if(*p != '/') continue;
	p++;
	if(*p != '.') continue;
	p++;
	if(*p != '.') continue;
	p++;
	strcpy(p - 3, p);
	p = p - 3;
   }

#ifdef DEBUG
   fprintf(stderr, "DEBUG: url = \"%s\"  realurl = \"%s\"  auth = \"%s\"\n",
	rq->url, rp->realurl, ((rp->auth == NULL) ? "No Access" : rp->auth->name));
   fprintf(stderr, "DEBUG: query = %s\n", rq->query);
#endif

   return;
}
