/* netdate.c
 *
 * netdate	A client using the TIMEP protocol (see RFC 868) for
 *		fetching the current date and time off a network server.
 *
 * Usage:	netdate [-dhsuvw] [server_name]
 *
 * 07/02/92 Tnet Release	Michael Temari, <temari@temari.ae.ge.com>
 *				Fred N. van Kempen, <waltje@uwalt.nl.mugnet.org>
 *
 * 07/02/96 1.00		Michael Temari, <temari@ix.netcom.com>
 *
 */

#include <sys/types.h>
#include <sys/ioctl.h>
#include <sys/wait.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <errno.h>
#include <fcntl.h>
#include <signal.h>
#include <unistd.h>
#include <time.h>
#include <net/netlib.h>
#include <net/hton.h>
#include <net/gen/in.h>
#include <net/gen/inet.h>
#include <net/gen/tcp.h>
#include <net/gen/tcp_io.h>
#include <net/gen/socket.h>
#include <net/gen/netdb.h>

#define	HOSTFILE	"/etc/netdate.hst"
#define	NETDATE_LOG	"/usr/adm/netdate.log"

int opt_d = 0;	/* show difference between our time and net time */
int opt_h = 0;	/* use host specified as default */
int opt_s = 0;	/* set our system time to nettime */
int opt_u = 0;	/* display time as GMT */
int opt_v = 0;	/* show what is happening */
int opt_w = 0;	/* write host to file */

static char *days[] = {
  "Sun", "Mon", "Tue", "Wed", "Thu", "Fri", "Sat"
};
static char *months[] = {
  "Jan", "Feb", "Mar", "Apr", "May", "Jun",
  "Jul", "Aug", "Sep", "Oct", "Nov", "Dec"
};

char defhost[256];

_PROTOTYPE(static int timep, (char *host, time_t *nettime));
_PROTOTYPE(static void usage, (void));
_PROTOTYPE(int main, (int argc, char *argv[]));

/*
 * Send a TIMEP request to a server.
 */
static int timep(host, nettime)
char *host;
time_t *nettime;
{
nwio_tcpconf_t tcpconf;
nwio_tcpcl_t tcpcopt;
nwio_tcpcl_t tcplopt;
char *tcp_device;
int netfd;
ipaddr_t nethost;
tcpport_t netport;

struct hostent *hp;
struct servent *sp;
u32_t val;
char buff[128];
time_t hosttime, now;
int s;
int tries;

   if(opt_v)
	fprintf(stderr, "Resolving %s...", host);

   if((hp = gethostbyname(host)) == (struct hostent *)NULL) {
	fprintf(stderr, "Unknown host %s!\n", host);  
	return(-1);
   } else
	memcpy((char *) &nethost, (char *) hp->h_addr, hp->h_length);

   /* Now, to which port must we connect? */
   if((sp = getservbyname("time", "tcp")) == (struct servent *)NULL) {
	fprintf(stderr, "TIME port is unknown????\n");
	return(-1);
   } else
	netport = sp->s_port;

   if(opt_v)
	fprintf(stderr, " trying %s:%d...\n", inet_ntoa(nethost), netport);


   /* Connect to time host */
   if((tcp_device = getenv("TCP_DEVICE")) == NULL)
	tcp_device = TCP_DEVICE;

   if((netfd = open(tcp_device, O_RDWR)) < 0) {
	perror("netdate: opening tcp");
	return(-1);
   }

   tcpconf.nwtc_flags = NWTC_LP_SEL | NWTC_SET_RA | NWTC_SET_RP;
   tcpconf.nwtc_remaddr = nethost;
   tcpconf.nwtc_remport = netport;

   s = ioctl(netfd, NWIOSTCPCONF, &tcpconf);
   if(s < 0) {
	perror("netdate: NWIOSTCPCONF");
	close(netfd);
	return(-1);
   }

   s = ioctl(netfd, NWIOGTCPCONF, &tcpconf);
   if(s < 0) {
	perror("netdate: NWIOGTCPCONF");
	close(netfd);
	return(-1);
   }

   tcpcopt.nwtcl_flags = 0;

   tries = 0;
   do {
	s = ioctl(netfd, NWIOTCPCONN, &tcpcopt);
	if(s == -1 && errno == EAGAIN) {
		if(tries++ >= 10)
			break;
		if(opt_v)
			fprintf(stderr, "Connect got error EAGAIN, sleeping 1 and retrying\n");
		sleep(1);
	} else
		break;
   } while(1);

   if(s < 0) {
	perror("netdate: NWIOTCPCONN");
	close(netfd);
	return(-1);
   }

   s = ioctl(netfd, NWIOGTCPCONF, &tcpconf);
   if(s < 0) {
	perror("netdate: NWIOGTCPCONF");
	close(netfd);
	return(-1);
   }

  /* Wait for the TIMEP server to send us its time. */
  s = read(netfd, (char *) &val, sizeof(u32_t));
  if (s != sizeof(u32_t)) {
	fprintf(stderr, "netdate: bad time reply.\n");
	close(netfd);
	return(-1);
  }
  hosttime = (time_t) ntohl(val);
  hosttime = hosttime - (time_t) 2208988800L;	/* adjust from Network Time */
  *nettime = hosttime;

  (void) close(netfd);

  return(0);
}

static void usage()
{
   fprintf(stderr, "Usage: netdate [-dhsuvw] [server]\n");
   exit(-1);
}

int main(argc, argv)
int argc;
char *argv[];
{
int c;
time_t ourtime;
time_t nettime;
time_t difftime;
struct tm *tm;
int rmttime;
char *rmthost;
FILE *logfile;
int logit = 0;
FILE *fp;

   opterr = 0;
   while((c = getopt(argc, argv, "dhsuvw")) != EOF)
	switch(c) {
		case 'd': opt_d = 1; break;
		case 'h': opt_h = 1; break;
		case 's': opt_s = 1; break;
		case 'u': opt_u = 1; break;
		case 'v': opt_v = 1; break;
		case 'w': opt_w = 1; break;
		default:
			usage();
	}

   if(opt_h) {
	fp = fopen(HOSTFILE, "r");
	if(fp == (FILE *)NULL) {
		fprintf(stderr, "netdate: Could not open default host file %s\n", HOSTFILE);
		return(-1);
	};
	if(fgets(defhost, sizeof(defhost), fp) == (char *)NULL) {
		fprintf(stderr, "netdate: Could not read default host file %s\n", HOSTFILE);
		fclose(fp);
		return(-1);
	}
	fclose(fp);
	if(defhost[strlen(defhost)-1] == '\n')
		defhost[strlen(defhost)-1] = '\0';
   }

   /* At most one more argument allowed. */
   if(optind != argc && optind != (argc - 1))
	usage();

   /* Get time from network host or just use our own */
   rmttime = 0;
   if(optind != argc)
	if(timep(argv[optind], &nettime))
		return(-1);
	else {
		rmttime = 1; rmthost = argv[optind];
	}
   else
	if(opt_h)
		if(timep(defhost, &nettime))
			return(-1);
		else {
			rmttime = 1; rmthost = defhost;
		}
	else
		(void) time(&nettime);

   (void) time(&ourtime);

   /* Do we need to set this system's clock? */
   if(rmttime && opt_s) {
	if(opt_v)
		fprintf(stderr, "Setting system time\n");
	if(stime(&nettime))
		perror("netdate: error setting time");
	else
		logit = 1;
   }

   if(opt_u)
	tm = gmtime(&nettime);
   else
	tm = localtime(&nettime);

   printf("%s, %2d %s %d %02d:%02d:%02d %s%s%s\n",
		days[tm->tm_wday],
		tm->tm_mday, months[tm->tm_mon], 1900+tm->tm_year,
		tm->tm_hour, tm->tm_min, tm->tm_sec,
		tzname[tm->tm_isdst],
		rmttime ? " from " : "",
		rmttime ? rmthost : "");

   /* Do we need to show the difference in time? */
   if(rmttime && (opt_d || opt_s)) {
	printf("Network Time: %s", ctime(&nettime));
	printf("Our Time    : %s", ctime(&ourtime));
	difftime = labs(ourtime - nettime);
	printf("Difference  : %s%d:%02d\n", (ourtime >= nettime ? "": "-"), difftime / 60, difftime % 60);
   }

   /* open log file if it exists */
   if(logit) {
	if((logfile = fopen(NETDATE_LOG, "r")) != (FILE *)NULL) {
		fclose(logfile);
		logfile = fopen(NETDATE_LOG, "a");
	}
	if(logfile != (FILE *)NULL) {
		difftime = labs(ourtime - nettime);
		tm = localtime(&nettime);
		fprintf(logfile, "%s, %2d %s %d %02d:%02d:%02d %s %s%02d:%02d%s%s\n",
			days[tm->tm_wday],
			tm->tm_mday, months[tm->tm_mon], 1900+tm->tm_year,
			tm->tm_hour, tm->tm_min, tm->tm_sec,
			tzname[tm->tm_isdst],
			(ourtime >= nettime ? " ": "-"), difftime / 60, difftime % 60,
			rmttime ? " from " : "",
			rmttime ? rmthost : "");
		fclose(logfile);
	}
   }

   if(opt_w && rmttime) {
	fp = fopen(HOSTFILE, "w");
	if(fp == (FILE *)NULL) {
		fprintf(stderr, "netdate: Could not open default host file %s\n", HOSTFILE);
		return(-1);
	};
	fprintf(fp, "%s\n", rmthost);
	fclose(fp);
   }

   return(0);
}
