static char rcsid[] = "$Id: hsearch.c,v 1.1 1996/03/20 21:38:36 paul Exp $";

#include <stdlib.h>

#include "hsearch.h"

/*
 * hsearch.c
 * ========
 *
 * This file contains a library of symbol table manipulation
 * functions.  The parameters and return values of the (externally
 * visible) functions are given in hsearch.h.  A
 * symbol table contains <id, value> pairs, where each id is an unsigned
 * long, and each value is a void *.  Each symbol table is stored as
 * an extensible hash table, with the symbol_table structure
 * containing a pointer to the table.
 *
 * This file (and hsearch.h) has been added to INMON because Amoeba lacks
 * the hsearch (and associated) library functions.
 */

/*
 * Number of <sym_name, sym_data> pairs that are stored in a bucket.
 */

#define ENTRIES_PER_BUCKET 5


/*
 * Buckets contain pairs of <hash_val, pointer to user
 * data>, with an unused entry indicated by a 0 value for used.
 */
struct bucket_entry {
    unsigned long hash_val;
    void *data;
    unsigned int used:1;
};

/*
 * An entry in the hash table---the number of bits significant
 * in accessing this bucket, and a pointer to the bucket itself.
 */
struct entry {
    int num_bits;             /* Number of bits used in accessing this entry */
    struct bucket_entry *bp;  /* Bucket associated with this entry           */
};

/*
 * We have a pointer to the hash table and num_bits.  num_bits gives
 * the number of bits in the hash value used in indexing.  Table
 * has, therefore, 2**num_bits entries.
 */
typedef struct symbol_table {
    int num_bits;         /* table contains 2**num_bits entries */
    struct entry *table;  /* pointer to array of table entries  */
} *symptr;


/*
 * Macro to determine which table entry points to the bucket 
 * associated with hash value "hval" given that "num_bits"
 * are currently in use.
 */

#define HASH_BITS 16

#define calc_index(hval, num_bits) \
  ((0xffff & (hval)) >> (HASH_BITS - (num_bits))) 


/*
 * Local functions not declared in sym.h
 */
static symptr hcreate_table(void);
static struct bucket_entry *hlocate(symptr sym_table, unsigned long key);
static int    hincrease_table(symptr sym_table);
static int    hbucket_create(symptr sym_table, int array_index);
static struct bucket_entry *hnew_bucket(void);


/*
 * Function: hsearch
 * Parameters: key - value to add/lookup in the hash table
 *             tp - pointer to hash table to look in
 *             user_info - user data for key
 * Returns: Existing or new user_info value, 0 if an error
 *          (such as a malloc error) occurs.
 *
 * If tp points to a null pointer then a new hash table is created.
 */

void *
hsearch(unsigned long key, void **tp, void *user_data)
{
    symptr table = *tp;
    int count;
    unsigned array_index;
    struct bucket_entry *bp;
	
    if (table == 0) {
	if ((table = hcreate_table()) == 0) {
	    return 0;
	}
	*tp = table;
    }
    bp = hlocate(table, key);
    if (bp) return bp->data;

    while (1) {
	if (table->table == 0) {
	    if (hincrease_table(table) == -1) {
		return 0;
	    }
	} else {
	    array_index = calc_index(key, table->num_bits);
	    bp = table->table[array_index].bp;
	    for (count = 0; count < ENTRIES_PER_BUCKET; count++, bp++) {
		if (bp->used == 0) {
		    bp->hash_val = key;
		    bp->data = user_data;
		    bp->used = 1;
		    return user_data;
		}
	    }
	    if (table->table[array_index].num_bits == table->num_bits) {
		if (table->num_bits == HASH_BITS) {
		    return 0;
		}
		if (hincrease_table(table) == -1) {
		    return 0;
		}
	    } else {
		hbucket_create(table, array_index);
	    }
	}
    }
}


/*
 * Function: hfind
 * Parameters: key - key value of entry to look up.
 *             tp - hash table to look in
 * Returns: user data key, NULL if key not found.
 */

void *
hfind(unsigned long key, void **tp)
{
    symptr table = *tp;
    struct bucket_entry *bp = hlocate(table, key);

    return bp ? bp->data : 0;
}


/*
 * Function: hdelete
 * Parameters: key - key value of entry to delete
 *             tp - hash table to look in
 * Returns: user data key, NULL if key not found.
 */

void *
hdelete(unsigned long key, void **tp)
{
    symptr table = *tp;
    struct bucket_entry *bp = hlocate(table, key);

    if (bp) {
	bp->used = 0;
	return bp->data;
    } else {
	return 0;
    }
}


/*
 * Function: hwalk
 * Parameters: tp - hash table to traverse
 *             call_me - function to call for each entry in the hash table.
 */

void
hwalk(void *tp, action_func call_me)
{
    symptr sym_table = tp;
    struct bucket_entry *bp = 0;
    int i, j;

    if (sym_table->table == 0) return;

    for (i = 0; i < (1 << sym_table->num_bits); i++) {
	if (sym_table->table[i].bp == bp) continue;

	bp = sym_table->table[i].bp;
	for (j = 0; j < ENTRIES_PER_BUCKET; j++) {
	    if (bp[j].used) {
		(*call_me)(bp[j].data);
	    }
	}
    }
}


/*
 * Function: hcreate_table
 * Parameters: none
 * Returns: pointer to new symbol_table struct if space for the struct 
 *          is allocated sucessfully, 0 if the malloc fails.
 */

static symptr
hcreate_table(void)
{
    symptr new_table;

    if ((new_table = malloc(sizeof(*new_table))) == 0) {
	return 0;
    }
    
    new_table->table = 0;
    return new_table;
}


/*
 * Function: hlocate
 * Parameters: sym_table - table to search in
 *             sym_name - name to search for
 * Returns: Pointer to bucket entry if found; null if not.
 */

static struct bucket_entry *
hlocate(symptr sym_table, unsigned long key)
{
    struct bucket_entry *bp;
    int i;

    if (sym_table == 0 || sym_table->table == 0) {
	return 0;
    }

    bp = sym_table->table[calc_index(key, sym_table->num_bits)].bp;
    for (i = 0; i < ENTRIES_PER_BUCKET; i++, bp++) {
	if (bp->used && bp->hash_val == key) {
	    break;
	}
    }
    return i < ENTRIES_PER_BUCKET ? bp : 0;
}


/*
 * Function: hincrease_table
 * Parameter: sym_table - table whose size needs to be increased
 * Returns: 0 if succesful, -1 on malloc failure.
 *
 * If table is empty then create a table of size one and allocate a 
 * bucket to it.  Otherwise, double the table size, and copy each pointer
 * in the original table to two positions in the new table.
 */

    static int
hincrease_table(symptr sym_table)
{
    if (sym_table->table == 0) {
	struct bucket_entry *bp;

	if ((bp = hnew_bucket()) == 0 ||
	    (sym_table->table = malloc(sizeof(struct entry))) == 0) {
	    if (bp) {
		free(bp);
	    }
	    return -1;
	}
	sym_table->table->bp = bp;
	sym_table->num_bits = sym_table->table->num_bits = 0;
    } else {
	struct entry *new_table;
	int i;

	if ((new_table = malloc(sizeof(*new_table) *
				(1 << (sym_table->num_bits + 1)))) == 0) {
	    return -1;
	}
	for (i = 0; i < (1 << sym_table->num_bits); i++) {
	    new_table[2*i].num_bits = new_table[2*i + 1].num_bits =
	      sym_table->table[i].num_bits;
	    new_table[2*i].bp = new_table[2*i + 1].bp =
	      sym_table->table[i].bp;
	}
	sym_table->num_bits++;
	free(sym_table->table);
	sym_table->table = new_table;
    }
    return 0;
}


/*
 * Function: hbucket_create
 * Parameters: sym_table - table to create a new bucket in
 *             array_index - the index at which the new bucket is created.
 * Returns: 0 on sucess, -1 on malloc failure.
 *
 * 2n table entries point to the bucket pointed to by entry array_index.
 * a new bucket is allocated, and the first n entries set to point to it.
 * Any entries from the existing bucket that belong in the new one are
 * transferred to it.
 */

    static int
hbucket_create(symptr sym_table, int array_index)
{
    unsigned start, len, i;
    int bit_pos;
    struct bucket_entry *new, *existing;
    struct bucket_entry *bp;

    if ((new = hnew_bucket()) == 0) {
	return -1;
    }
    existing = sym_table->table[array_index].bp;
    len = 1 << (sym_table->num_bits - sym_table->table[array_index].num_bits);
    start = array_index - array_index % len;

    /*
     * Add links to new bucket.
     */
    for (i = start + len/2; i < start + len; i++) {
	sym_table->table[i].bp = new;
    }
    for (i = start; i < start + len; i++) {
	sym_table->table[i].num_bits++;
    }
    /*
     * Move appropriate entries into new bucket.
     */
    bit_pos = 1 << (HASH_BITS - sym_table->table[start].num_bits);
    for (bp = new, i=0; i < ENTRIES_PER_BUCKET; i++) {
	if (existing[i].hash_val & bit_pos) {
	    *bp = existing[i];
	    existing[i].used = 0;
	    bp++;
	}
    }
    return 0;
}


/*
 * Function: hnew_bucket
 * Returns: pointer to a newly allocated bucket on success; 0 on failure.
 */

    static struct bucket_entry *
hnew_bucket(void)
{
    struct bucket_entry *bp;
    int i;

    bp = malloc(sizeof *bp * ENTRIES_PER_BUCKET);
    if (bp) {
	for (i = 0; i < ENTRIES_PER_BUCKET; i++) {
	    bp[i].used = 0;
	}
    }
    return bp;
}
