static char rcsid[] = "$Id: insplit.c,v 1.2 1996/06/04 08:30:18 paul Exp $";

/*
 * Based on: insplit.c 1.2 94/09/07 Paul Ashton"
 */

/*
 * insplit
 * =======
 *
 * This program takes one or more interaction network logfiles, and produces
 * a separate logfile for each interaction contained in the logfiles.  The
 * command line for the program is:
 *
 * insplit [-c cutoff] [-o outdir] [-u undefeventfile] logfile [logfile ...]
 *
 * The cutoff value is an integer.  Any interactions which contain
 * fewer than cutoff events are not output.  The default cutoff is 2,
 * which supresses all trivial interactions that have only sink and
 * source events.  The value of dir is a directory to which the
 * interaction logfiles should be written ("."  by default).  After
 * these options comes a list of one or more logfiles which are to be
 * processed.  One output file is produced for each interaction
 * network containing more than cutoff events.  Each is a logfile with
 * the same format as a raw logfile, having the sub-task # of the of
 * the first sub-task in the interaction as the file name.
 *
 * The following data structures are used in grouping the logfiles' events
 * into interactions:
 *
 * event  - one per event record
 * stn_hd - one per sub-task  
 * in_hd  - one per interaction
 *
 * Also two hash tables, keyed on sub-task number, are maintained---one for
 * the in_hd structures (usually referred to as ipp), and one for the
 * stn_hd structures (usually referred to as spp), as direct
 * access to these structures is required.
 *
 * The events are grouped into interactions in the following way.  Files are
 * read in the order that they appear in the command line, and events are
 * read in the order that they apear in the file.  
 *
 * Once the event list is created for each file, events are incorporated
 * into the data structures desrcribed above.  Those that cannot yet
 * be inserted are appended to the free list.  Once all files have been
 * read in events from the free list are incorporated into the event data
 * structures until the free list is empty, or no further events can be
 * added from the free list.  Finally the interaction table is traversed,
 * and each interaction is written to a different file.
 *
 * The way in which events are added to the data structures depends on the
 * event type.  For source events, new in_hd and stn_hd entries are created
 * for the new interaction, and the event made the first event of the stn_hds
 * chain.  For other events, the event is only added if its stn is found
 * in the stn_hd table.  If it is, then the event is appended to the stn chain,
 * otherwise the event remains on the free_list.  When fork events are appended
 * a new stn_hd event is created, and linked onto the list of the appropriate
 * in_hd.
 *
 * Change history:
 *
 * Who  Date        What
 * ======================================================================
 * PJA  May 91     Initial coding
 *
 * PJA  14/6/91    Added MS, KBD, and CONS input begin and end events,
 *                 changed TERM_INPUT to make the input end the source,
 *                 and the begin a normal event.
 * PJA  15/7/91    Changed .h file inclusion to allow sun 4/4.1.1 compiles.
 * PJA  8/9/94     Ported to Amoeba.
 */

#include <stdlib.h>
#include <stdio.h>
#include <unistd.h>
#include <assert.h>

#include "../../kernel/logging.h"
#include "hsearch.h"
#include "logfiles.h"


typedef struct event_list event_list_t;
typedef struct stn_hd stn_hd_t;
typedef struct in_hd in_hd_t;

/*
 * Structure to hold information about an individual event.
 */
struct event_list {
    struct ev_record ev;
    void *rest;
    evtype_t ev_type;
    event_list_t *next;
};

/*
 * Sub-task header.  in should always have a value because the IN_HD
 * struct for a sub-task should exist before the stn_hd is created.
 */
struct stn_hd {
    int stn;    		/* number of this sub-task */
    in_hd_t *in;		/* interaction that this sub-task */
				/* belongs to */
    event_list_t *events_start;	/* first event in the sub-task list */
    event_list_t *events_end;	/* last event in the sub-task list */
    stn_hd_t *next;
};
	
/*
 * Interaction network header
 */
struct in_hd {
    int      evcount;       /* number of events in the interaction net */
    int      in_num;        /* stn of the first sub-task in the interaction */
    stn_hd_t *stns_start;   /* first sub-task in list*/
    stn_hd_t *stns_end;     /* last sub-task in list */
};


#define USAGE "Usage: %s [-c cutoff] [-d] [-o dir] [-u undef_file] logfile [logfile ...]\n"

static char *progname;
static int debug = 0;

/*
 * cutoff, rootp, and tot_drops need to be global.  They are used in
 * functions that are called by hwalk(), which means they cannot be
 * passed as a parameters.  Similar reasons exist for log_file_w and
 * name_walk.
 */
static int cutoff = 2;     /* set event cutoff at 2 - this means INs that */
                           /* are just a source and a sink are suppresed   */
                           /* by default */
static void *rootp;
FILE *log_file_w;

/*
 * Globals used in buidling up the overall host table
 */

char **hosts = 0;

/*
 * Functions declared in this file.
 */
static void usage(void);

static void read_files(int argc, char *argv[], void **ipp,
				void **spp, stn_hd_t *free_list);
static int read_named_file(char *filename, stn_hd_t *file_events);

static void add_to_list(stn_hd_t *new_events, void **ipp, void **spp,
			stn_hd_t *free_list);
static event_list_t *new_sink(struct ev_record *ep, int stn);
static int add_events(void **ipp, void **spp, stn_hd_t *event_list);
static int new_source_event(void **ipp, void **spp, event_list_t *ep);
static int new_other_event(void **spp, event_list_t *ep, stn_hd_t *chain);
static stn_hd_t *create_stnhd(void **spp, int stn, in_hd_t *ip);
static void *ehsearch(unsigned long key, void **rootp, void *user_data);

static void append_event(stn_hd_t *event_list, event_list_t *ep);
static void event_error(event_list_t *ep, char *msg);

static void write_files(in_hd_t *ip);
static void output_in(void *node);
static void write_in_to_file(char *filename, in_hd_t *ip);

static int events_remain(stn_hd_t *sp);

static void free_ins(void *rp);
static void delete_innode(void *node);
static void free_stnhds(void *rp);
static void delete_stnhd(void *node);
static void free_event_list(stn_hd_t *event_list);
static void free_event(event_list_t *ep);


/*
 * Function: main
 * Arguments: argc, argv - see USAGE for expected arguments
 * Returns: 0 if all OK, exits with a status of 1 if a fatal
 *          error is encountered.
 *
 * The options are processed, and the global cutoff and the local outputdir
 * are changed if necessary.  The output directory is then checked to ensure
 * that we can create files there before going any further.  Then read_files
 * is called to read the log files, and if no interactions are found in
 * any of the files then we exit.  We then continue to attempt to add free
 * events to the sub-task linked lists until either no events remain on the 
 * free event list, or until we make a pass of the event free list where no
 * events are removed.  The current directory is then changed to the
 * output directory, and write_files called to output the interaction network
 * logfiles.
 */

int
main(int argc, char *argv[])
{
    void    *ip = 0;             /* interaction table */
    void    *sp = 0;             /* sub-task table */
    in_hd_t  free_events;
    stn_hd_t free_stn;
    char    *outputdir = ".";
    char    *undef_name = 0;   /* file name to write undefined events to */

    progname = argv[0];

    free_events.evcount = 0;
    free_stn.stn = free_events.in_num = INVALID_STN;
    free_events.stns_start = free_events.stns_end = &free_stn;
    free_stn.in = &free_events;
    free_stn.events_start = free_stn.events_end = 0;
    free_stn.next = 0;
	
    /*
     * Check and process arguments.  See the man page and USAGE for
     * details of the arguments.  If -o or -c 
     * is specified more than once then the value is the final one
     * on the command line.
     */
    {
	int option;
	char *cp;
	extern char *optarg;
	extern int optind;
			      
	while ((option = getopt(argc, argv, "c:do:u:")) != -1)
	  switch(option) {
	    case 'c':
	      /*
	       * Specifies the cutoff for IN output -
	       * INs with this number of events
	       * or fewer are not written.
	       */
	      cutoff = strtol(optarg, &cp, 0);
	      if (*cp != '\0') usage();
	      break;

	    case 'd':
	      debug = 1;
	      break;

	    case 'o':
	      /*
	       * -o specifies an output directory
	       */
	      outputdir = optarg;
	      break;

	    case 'u':
	      /*
	       * -u specifies an output file for
	       * undefined events
	       */
	      undef_name = optarg;
	      break;

	    default:
	      /*
	       * unknown argument.
	       */
	      usage();
	      break;
	  }

	argc -= optind;       /* set to the number of args remaining */
	argv += optind;       /* step over already processed args */
	if (argc < 1) {
	    /*
	     * Whoops - no files to process
	     */
	    usage();
	}
    }

    /*
     * Check out the output directory now - before we get too far
     * down the track.  Note that using access is known to cause
     * problems for setuid programs, so perhaps an alternative test
     * should be made.
     */

    if (access(outputdir, X_OK|R_OK|W_OK) == -1) {
	(void) fprintf(stderr,
		       "%s: Output directory `%s' inacessible because: ",
		       progname, outputdir);
	perror("");
	exit(1);
    }

    /*
     * the remaining command line arguments are logfiles - read them in.
     */
    read_files(argc, argv, &ip, &sp, &free_stn);
    if (ip == 0) {
	fprintf(stderr, "%s: No interactions found\n", progname);
	exit(1);
    }

    /*
     * Change directory in preparation for writing the output files.
     */
    if (chdir(outputdir) == -1) {
	fprintf(stderr, "%s: cd to output dir failed because: ",
		       progname);
	perror("");
    }

    /*
     * Keep calling add_events to make passes thru free_events until
     * either all of the events have been removed from free_events,
     * or until we have a pass where no events are removed.
     */
    while (free_stn.events_start) {
	int num_added = add_events(&ip, &sp, &free_stn);

	if (debug) fprintf(stderr, "%d events added\n", num_added);

	if (!num_added) {
	    if (events_remain(&free_stn)) {
		if (debug) {
		    fprintf(stderr, "%s: Unattached events remain\n",
			    progname);
		}
		if (undef_name != 0) {
		    /*
		     * Output the unattached events.
		     */
		    write_in_to_file(undef_name, &free_events);
		}
	    }
	    free_event_list(&free_stn);
	    break;
	}
    }

    write_files(ip);

    free_ins((char *)ip);       /* mainly for mallocmap's benefit - */
    free_stnhds((char *)sp);    /* omit?            */
    return 0;
}


static void
usage(void)
{
    (void) fprintf(stderr, USAGE, progname);
    exit(1);
}


/*
 * Function: read_files
 * Parameters: argc - the number of file name arguments on the command line
 *             argv - the file name parameters
 *             ipp - pointer to in_hd table
 *             spp - pointer to stn_hd table
 *             free_list - pointer to free list head
 * Returns: the total number of event drops which occurred in all
 * successfully read files.
 *
 * Steps through the file names, attmpting to read each.  If no error
 * occurs reading all of the events in a log file, then add_to_list is
 * called to put the file events into the event data structures.
 */

static void
read_files(int argc, char *argv[], void **ipp, void **spp, stn_hd_t *free_list)
{
    stn_hd_t file_events;

    /*
     * The user has specified files for us to read - read
     * each file, and if it is read sucessfully incorporte
     * its events into the stn lists and the free list
     */
    while (argc > 0) {
	if (read_named_file(*argv, &file_events)) {
	    /*
	     * events in the file were read in successfully,
	     * so incorporate them into the sub-task
	     * and free lists.
	     */
	    add_to_list(&file_events, ipp, spp, free_list);
	}
	argv++, argc--;      /* move to next file name */
    }
}


/*
 * Function: read_named_file
 * Parameters: filename - name of file to read event records from
 *             file_events - event list header which is set to point
 *                           to the linked list of events read.
 * Returns: 1 if successful, 0 if error (and events_start and events_end
 *          pointers set to 0).  It is an error if the file cannot be
 *          opened for reading, or if a format error is found in reading
 *          the file.
 *
 * Events are read in one by one from the logfile.  An event structure is
 * created, and initialised for each event.  Event(s) remaining after the
 * initialisation function is complete are appended to the event list
 * being constructed for the file.  Both initial comments and comments in
 * EV_PSCOMMENT event records are added to the comment data structures.
 */

static int
read_named_file(char *filename, stn_hd_t *file_events)
{
    FILE *log_file;
    struct ev_record ev;
    void *rest;
    event_list_t *elistp;
    int seq_no = 0;

    file_events->events_start = file_events->events_end = 0;

    if ((log_file = fopen(filename, "r")) != 0) {
	/*
	 * file opened OK - read it in
	 */

	while (log_read_event(log_file, &ev, &rest)) {
	    ++seq_no;
	    elistp = malloc(sizeof(*elistp));
	    assert(elistp);
	    elistp->ev = ev;
	    elistp->rest = rest;
	    elistp->ev_type = log_event_type(ev.e_event);
	    elistp->next = 0;
	    
	    /*
	     * Create sink to represent the end of the sub-task that had been
	     * associated with the thread that read the input.
	     */
	    if (ev.e_event == EV_TTY_INPUT && ev.e_other_stn != INVALID_STN) {
		append_event(file_events, new_sink(&ev, ev.e_other_stn));
	    }

	    append_event(file_events, elistp);
	}

	fclose(log_file);
	return 1;
    } else {
	fprintf(stderr, "%s: Couldn't open '%s' for reading\n",
		progname, filename);
	return 0;
    }
}


/*
 * Function: add_to_list
 * Parameters: new_events - a linked list of events to add in to the sub-task
 *                          lists and the free list.
 *             ipp - pointer to the table of in_hd structs
 *             spp - pointer to the table of stn_hd structs
 *             free_list - the event free list.
 * Returns: nothing
 *
 * Takes events from new_events and incorporates all it can into the sub-task
 * lists (using add_events()).  Any remaining events are appended to the free
 * list.
 */

static void
add_to_list(stn_hd_t *new_events, void **ipp, void **spp, stn_hd_t *free_list)
{
    (void) add_events(ipp, spp, new_events);
    if (new_events->events_start) {
	/*
	 * There are events that are left over - add them to the
	 * free list.
	 */
	if (free_list->events_start) {
	    free_list->events_end->next = new_events->events_start;
	} else {
	    free_list->events_start = new_events->events_start;
	}
	free_list->events_end = new_events->events_end;
    }
}


/*
 * Function: new_sink
 * Parameters: ep - event giving rise to the new sink event
 *             stn - stn to give to the new event 
 * Returns: a pointer to a new sink event record.
 *
 * This function creates a new EV_SINK event which results from the
 * event passed as ep.  stn is the sub-task # for the EV_SINK event.
 * Associated with an EV_SINK event is a single parameter: the stn
 * of the event giving rise to the sink event.
 */

static event_list_t *
new_sink(struct ev_record *ep, int stn)
{
    event_list_t *elist;
	
    elist = malloc(sizeof(*elist));
    assert(elist);

    elist->ev = *ep;
    elist->ev.e_event = EV_SINK;
    elist->ev.e_stn = stn;
    elist->ev.e_other_stn = INVALID_STN;
    elist->ev.e_bufflen = 0;
    elist->rest = 0;
    elist->ev_type = sink;
    elist->next = 0;
    return elist;
}


/*
 * Function: add_events
 * Parameters: ipp - pointer to root of in_hd table
 *             spp - pointer root of stn_hd table
 *             event_list - a list of events to attempt to insert into their
 *                          sub-task list.
 * Returns: the number of the events from the event list that were
 *          successfully added.
 *
 * This function traverses the events in event_list trying to
 * add each one its sub-task list.  If the event can be added, it is
 * removed from event_list and added to an appropriate sub-task list.
 * If the event is found to be in error in some way then an error
 * message is printed, and the event is removed from event_list.
 *
 * The way in which an event is handled depends on its type.  Source events
 * cause a new interaction to be started---an stn_hd is created with its
 * in_num and stn the same, and the event is linked into the chain of
 * the new_stn.  All other event types (frk, join, normal and sink) are only
 * added to their list if their sub-task is already in the stn_hd table.  For
 * fork events the event is linked into the old sub-task, and a new stn_hd
 * (with no events linked to it) is created to link events for the new
 * sub-task to.
 */

static int
add_events(void **ipp, void **spp, stn_hd_t *event_list)
{
    int additions = 0;
    event_list_t *ep, *next, *tail=0;
    stn_hd_t *chain;

#define unlink_event(ep, tail) { \
    if (tail) \
        tail->next = ep->next; \
    else \
        event_list->events_start = ep->next; \
    ep->next = 0; \
}

    for (ep = event_list->events_start; ep; ep = next) {
	next = ep->next;

	switch(ep->ev_type) {
	  case source:
	    unlink_event(ep, tail);
	    if (new_source_event(ipp, spp, ep)) {
		additions++;
	    }
	    break;

	  case frk: case normal: case sink: case join:
	    chain = hfind(ep->ev.e_stn, spp);
	    if (chain != 0) {
		/*
		 * STN is in stn_hd table - record can be added.
		 */
		unlink_event(ep, tail);
		if (new_other_event(spp, ep, chain)) {
		    additions++;
		}
	    } else {
		/*
		 * event left in list so advance tail
		 */
		tail = ep;
	    }
	    break;

	  default:
	    fprintf(stderr, "%s: event type %d not handled by add_events\n",
			   progname, ep->ev_type);
	    break;
	} /*switch*/
    } /*for*/

    /*
     * The event_list->events_end pointer may now be out of date - 
     * recalculate it.
     */
    if (event_list->events_start == 0) {
	event_list->events_end = 0;
    } else {
	for (tail = event_list->events_start; tail->next; tail = tail->next) {
	    ;
	}
	event_list->events_end = tail;
    }

    return additions;
}


/*
 * Function: new_source_event
 * Parameters: ipp - interaction network table
 *             spp - sub-task table
 *             ep - source event to add
 * Returns: 1 if added successfully, 0 otherwise
 *
 * A brand new interaction network and a brand new
 * sub-task are beginning - create a new entry in the
 * interaction network table, and a new entry in
 * the sub-task table, and link everything together.
 */

static int
new_source_event(void **ipp, void **spp, event_list_t *ep)
{
    in_hd_t *new_in = malloc(sizeof(*new_in));
    stn_hd_t *new_chain;

    assert(new_in);
    new_in->in_num = ep->ev.e_stn;
    new_in->evcount = 1;
    new_in->stns_start = new_in->stns_end = 0;
    if (hfind(new_in->in_num, ipp) != 0) {
	/*
	 * URK - the interaction network is already in
	 * the table - this shouldn't happen!
	 */
	event_error(ep, "interaction # is already in the I.N. table");
	free_event(ep);
	free(new_in);
	return 0;
    }

    new_in = ehsearch(new_in->in_num, ipp, new_in);
	
    if ((new_chain = create_stnhd(spp, ep->ev.e_stn, new_in)) == 0) {
	/*
	 * URK - this sub-task is already in the
	 * table - this shouldn't happen!
	 */
	event_error(ep, "stn is already in the STN table");
	free_event(ep);
	free(new_in);
	/*
	 * Remove the node from the IN table because of the error.
	 */
	(void) hdelete(new_in->in_num, ipp);
	return 0;
    }
    new_chain->events_start = new_chain->events_end = ep;
    new_chain->in = new_in;
    return 1;
}


/*
 * Function: new_other_event
 * Parameters: spp - sub-task table
 *             ep - source event to add
 *             chain - sub-task chain that this event belongs to
 * Returns: 1 if added event successfully, 0 otherwise
 *
 * For fork events create a new stn_hd.  For all events append into the
 * stn chain sp.
 */ 

static int
new_other_event(void **spp, event_list_t *ep, stn_hd_t *chain)
{
    int addedok = 1;                 /* insert OK unless shown otherwise */
	
    assert(chain->stn != 0);

    if (ep->ev_type == frk) {
	/*
	 * Need to start a new empty stn chain for the new stn.
	 */
	if (create_stnhd(spp, ep->ev.e_other_stn, chain->in) == 0) {
	    /*
	     * The sub-task is already in the table
	     */
	    event_error(ep, "stn is already in the table");
	    free_event(ep);
	    addedok = 0;
	}
    }

    if (addedok) {
	append_event(chain, ep);
	chain->in->evcount++;
    }
    return addedok;
}


/*
 * Function: create_stnhd
 * Parameters: spp - the sub-task table
 *             stn - stn to give to the new stn_hd
 *             ip - in_hd to link the event into
 *
 * Allocates and initialises a new stn_hd struct, inserts it into the
 * stn_hd table, and links it into the interaction's stn chain.  If the 
 * stn already exists in the table then something is
 * wrong, so the stn_hd is freed, and null is returned.
 */

static stn_hd_t *
create_stnhd(void **spp, int stn, in_hd_t *ip)
{
    stn_hd_t *ret_val = malloc(sizeof(*ret_val));

    assert(ret_val && spp && ip);
    ret_val->stn = stn;
    ret_val->in = ip;
    ret_val->events_start = ret_val->events_end = 0;
    ret_val->next = 0;

    if (hfind(ret_val->stn, spp) != 0) {
	/*
	 * URK - this sub-task is already in the
	 * table - this shouldn't happen!
	 */
	free(ret_val);
	ret_val = 0;
    } else {
	ret_val = ehsearch(ret_val->stn, spp, ret_val);

	if (ip->stns_start == 0) {
	    ip->stns_start = ret_val;
	} else {
	    ip->stns_end->next = ret_val;
	}
	ip->stns_end = ret_val;
    }
    return ret_val;
}


/*
 * Function: ehsearch
 *
 * An error checking wrapper for hsearch
 */

static void *
ehsearch(unsigned long key, void **rootp, void *user_data)
{
    void *retval = hsearch(key, rootp, user_data);
    
    if (retval == 0) {
	fprintf(stderr, "%s: hsearch storage alloc failed\n", progname);
	exit(1);
    }
    return retval;
}


/*
 * Function: append_event
 * Parameters: event_list - list of events headed by an STN_HD
 *             ep - linked list of EVENTS.
 * Returns: nothing
 *
 * Appends ep to event_list
 */

static void
append_event(stn_hd_t *event_list, event_list_t *ep)
{
    event_list_t *tail;

    if (event_list->events_start == 0) {
	event_list->events_start = ep;
    } else {
	event_list->events_end->next = ep;
    }

    for (tail = ep ; tail->next ; tail = tail->next) {
	;
    }
    event_list->events_end = tail;
}


/*
 * Function: event_error
 * Parameter: ep - event pointer
 * Returns: nothing
 *
 * An error relating to an event record occurred.  Print out the supplied
 * message plus event details.  The timestamp is included to (nearly) uniquely
 * identify the event record.
 */

static void
event_error(event_list_t *ep, char *msg)
{
    fprintf(stderr, 
	    "%s: %s\n\tstn = %u\tnew_stn = %u\n\tevent = %d\ttime = %d/%d\n\n",
	    progname, msg, ep->ev.e_stn, ep->ev.e_other_stn,
	    ep->ev.e_event, ep->ev.e_secs, ep->ev.e_usecs);
}


/*
 * Function: write_files
 * Parameter: ip - the in_hd table
 * Returns: nothing.
 *
 * Uses hwalk to traverse the table.  It calls output_in to write each
 * interaction network.
 */

static void
write_files(in_hd_t *ip)
{
    hwalk(ip, output_in);
}


/*
 * Function: output_in
 * Paraemers: node - pointer to an IN_HD pointer stored in the in_hd table.
 * Returns: nothing
 *
 * Called by hwalk(3).  The # of the interaction network is used as
 * the name of the log file.  
 */

#define FILENAMELEN 8

static void
output_in(void *node)
{
    in_hd_t *ip = node;
    char filename[FILENAMELEN + 1];
	
    /*
     * Fewer events in the IN than cutoff - forget it.
     */
    if (ip->evcount <= cutoff) {
	return;
    }
    
    sprintf(filename, "%0*x", FILENAMELEN, ip->in_num);
    write_in_to_file(filename, ip);
}


/*
 * Function: write_in_to_file
 * Parameter: filename - name of file to write
 *            ip - interaction network to write.
 *
 * The IN's stn list is traversed, and for each
 * stn, all of its events are written.  If an error occurs then an error
 * message is written, and the output file unlinked.
 */

static void
write_in_to_file(char *filename, in_hd_t *ip)
{
    FILE *lp;
    stn_hd_t *sp;
    event_list_t *ep;
    int start_secs, start_usecs;
	
    if ((lp = fopen(filename, "w")) == 0) {
	fprintf(stderr, "%s: write open failed for log files %s\n",
		progname, filename);
	return;
    }

    /*
     * Now write the event records.  Adjust time so that they are 
     * relative to the source event which has time 0.
     */
    start_secs = ip->stns_start->events_start->ev.e_secs;
    start_usecs = ip->stns_start->events_start->ev.e_usecs;
    for (sp = ip->stns_start; sp; sp = sp->next) {
	for (ep = sp->events_start; ep; ep = ep->next) {
	    ep->ev.e_secs -= start_secs;
	    ep->ev.e_usecs -= start_usecs;
	    if (ep->ev.e_usecs < 0) {
		ep->ev.e_usecs += 1000000;
		ep->ev.e_secs--;
	    }
	    if (!log_write_event(lp, &ep->ev, ep->rest)) {
		fprintf(stderr, "%s: error writing to %s\n",
			progname, filename);
		fclose(lp);
		return;
	    }
		
	} /* for event list */
    } /* for stn list */

    fclose(lp);
}


/*
 * Function: events_remain
 * Parameter: sp - stn_hd pointer to a linked list of events
 * Returns: 1 if there events remaining in the list that should have been
 *          allocated to an interaction network;
 */

static int
events_remain(stn_hd_t *sp)
{
    event_list_t *ep;

    for (ep = sp->events_start ; ep ; ep = ep->next) {
	/*
	 * Extraneous EV_SINK events can arise - for instance
	 * where the first term_input event in a file is used to
	 * create a sink_event for the previous interacrtion.
	 * Maybe there should be further checking of EV_SINK
	 * events to see whether they are legal final free_list
	 * members or not.
	 */
	if (ep->ev.e_event != EV_SINK) {
	    return 1;
	}
    }
    return 0;
}


/*
 * Function: free_ins
 * Parameter: rp - pointer to the in_hd table.
 * Returns: nothing
 *
 * Initiates the freeing of all of the nodes in the in_hd table, and the 
 * in_hd structs that the nodes point to.  The global rootp is used to pass
 * the value of rp into delete_innode, the function which is passed to hwalk
 * for it to call.  The use of rootp is a bit of a hack.  It is also used in
 * the freeing up of the stn_hd table.  As long as one of these doesn't call
 * itself or the other function then there should be no problem.
 */

static void
free_ins(void *rp)
{
    rootp = rp;
    hwalk(rp, delete_innode);
}


/*
 * Function: delete_innode
 * Parameters: node - pointer to an IN_HD pointer stored in the in_hd table.
 * Returns: nothing
 *
 * Called by hwalk(3).
 */

static void
delete_innode(void *node)
{
    in_hd_t *ip = node;
	
    /*
     * safe to delete the node.
     */
    (void) hdelete(ip->in_num, &rootp);
    free(ip);
}


/*
 * Function: free_stnhds
 * Parameter: rp - pointer to the root of the stn_hd table.
 * Returns: nothing
 *
 * Initiates the freeing of all of the nodes in the stn_hd table, and the 
 * stn_hd structs and associated event lists that the nodes point to.
 * The global rootp is used to pass the value of rp into delete_stnhd,
 * the function which is passed to hwalk for it to call.  The use of rootp
 * is a bit of a hack.  It is also used in the freeing up of the in_hd table.
 * As long as one of these doesn't call itself or the other function then
 * there should be no problem.
 */

static void
free_stnhds(void *rp)
{
    rootp = rp;
    hwalk(rp, delete_stnhd);
}


/*
 * Function: delete_stnhd
 * Paraemers: node - pointer to an IN_HD pointer stored in the in_hd table.
 * Returns: nothing
 *
 * Called by hwalk(3).
 */

static void
delete_stnhd(void *node)
{
    stn_hd_t *sp = node;
	
    (void) hdelete(sp->stn, &rootp);
    free_event_list(sp);
    free(sp);
}


/*
 * Function: free_event_list
 * Parameter: event_list - an stn_hd pointer to a list of events to free.
 * Returns: nothing
 *
 * Frees all of the events in event_list and zeroes event_list's events_start
 * and events_end pointers.
 */

static void
free_event_list(stn_hd_t *event_list)
{
    event_list_t *ep, *savep;

    for (ep = event_list->events_start; ep; ep = savep) {
	savep = ep->next;
	free_event(ep);
    }
    event_list->events_start = event_list->events_end = 0;
}


/*
 * Function: free_event
 * Parameter: ep - an event pointer
 * Returns: nothing
 *
 * Frees the event record ep.
 */

static void
free_event(event_list_t *ep)
{
    free(ep->rest);
    free(ep);
}

