/* $Id: logdump.c,v 1.2 1996/06/04 08:30:20 paul Exp $ */

/*
 * logdump
 * =======
 *
 * Command line:
 *
 *     logdump logfile [...]
 *
 * logdump prints in text form the contents of each of the
 * logfiles specified on the command line.  The text form of a logfile
 * consists of:
 *
 *     - its name
 *     - the contents of its header
 *     - the process comments at the start of the file
 *     - details of all event records
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>

#include "../../kernel/logging.h"

#include "messages.h"
#include "logfiles.h"

static char *progname;

static void usage(void);
static void dump_file(char *filename);
static void dump_event(struct ev_record *ep, void *rest);


/*
 * Function: main
 * Command line arguments: logfile [logfile ...]
 *
 * Dumps all of the log files specified on the command line.
 */

void
main(int argc, char *argv[])
{
    int i;

    progname = argv[0];

    if (argc < 2) usage();

    for (i = 1; i < argc; i++) {
	dump_file(argv[i]);
    }

    exit(0);
}


static void
usage(void)
{
    fprintf(stderr, "Usage: %s logfile [...]\n", progname);
    exit(1);
}


/*
 * Function: dump_file
 * Parameter: filename - name of logfile whose contents are to be displayed.
 * Returns: nothing.
 *
 * Dumps the contents of the specified file.  Two passes are made through the
 * file.  In the first, all of the process comment information is recorded
 * (this information is needed to provide process names in the second pass).
 * In the second pass the log file information is written.
 */

static void
dump_file(char *filename)
{
    FILE *logfile;
    struct ev_record ev;
    void *rest;

    /*
     * First of all do a pass that collects all of the process name
     * information.
     */
   if ((logfile = fopen(filename, "r")) == 0) {
	fprintf(stderr, "%s: Failed to open %s\n", progname, filename);
	return;
    }

    while (log_read_event(logfile, &ev, &rest)) {
	dump_event(&ev, rest);
    }

    if (fclose(logfile) == EOF) {
	fprintf(stderr, "%s: Failed to close %s\n", progname,
		filename);
	return;
    }
}


static void 
dump_event(struct ev_record *ep, void *rest)
{
  char msg[80];
    printf("%s in %d: stn = %d, other stn = %d, time = %d %06d, bufflen = %d\n",
	   log_event_name(ep->e_event), ep->e_pnr, 
	   ep->e_stn, ep->e_other_stn, ep->e_secs, ep->e_usecs, ep->e_bufflen);
    if (ep->e_event == EV_TTY_INPUT) {
	printf("    Input = `");
	assert(rest);
	fwrite(rest, 1, ep->e_bufflen, stdout);
	printf("'\n");
    } else if (ep->e_event == EV_NEW_NAME) {
	printf("    New name = %s\n", (char *)rest);
    } else if (ep->e_event == EV_MSG_RECV) {
        message_is(ep->e_pnr, *(int *)rest, msg, sizeof(msg));
	printf("    Message = %s\n", msg);
    } else {
	assert(rest == 0);
    }
}
