static char rcsid[] = "$Id: logfiles.c,v 1.2 1996/06/04 08:30:21 paul Exp $";

/*
 * This file contains functions that manipulate (node and task) log files.
 * "logfiles.h" contains relevant definitions.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../../kernel/logging.h"
#include "logfiles.h"


/*
 * Function: log_read_event
 * Parameter: lp - logfile to read from
 * Returns: pointer to event record on success, 0 on failure
 *
 * An event record begins with an integer holding it's length.  This
 * is followed by some fields common to all event records, and then 
 * event-specific fields.
 */

int
log_read_event(FILE *lp, struct ev_record *ep, void **rest)
{
    void *ret_buff = 0;

    if (fread(ep, sizeof(*ep), 1, lp) == 0) {
	return 0;
    }

    if (ep->e_bufflen > 0) {
	ret_buff = malloc(ep->e_bufflen);
	assert(ret_buff);
	if ((int) fread(ret_buff, 1, ep->e_bufflen, lp) != ep->e_bufflen) {
	    return 0;
	}
    }
    if (rest) *rest = ret_buff;
    return 1;
}


/*
 * Function: log_write_event
 * Parameter: lp - logfile to write to
 *            ep - event to write
 * Returns: 0 on success, -1 on failure
 *
 * First of all the event record is constructed in a buffer, then it
 * is written.
 */

int
log_write_event(FILE *lp, struct ev_record *ep, void *rest)
{
    if (rest == 0) ep->e_bufflen = 0;
    if (fwrite(ep, sizeof(*ep), 1, lp) != 1) {
	return 0;
    }
    if (ep->e_bufflen > 0 &&
	(int) fwrite(rest, 1, ep->e_bufflen, lp) != ep->e_bufflen) {
	return 0;
    }

    return 1;
}


/**************************************************************************/

/*
 * Functions for returning event information---name and type.
 */

struct event_info {
    char *ev_name;
    evtype_t ev_type;
};

static struct event_info event_info[] = {
    {"EV_TTY_INPUT", source},
    {"EV_MSG_BLOCK", normal},
    {"EV_MSG_SEND", frk},
    {"EV_MSG_RECV", join},
    {"EV_FORK", frk},
    {"EV_EXIT", sink},
    {"EV_SINK", sink},
    {"EV_NEW_NAME", normal},
};

#define num_events (sizeof(event_info) / sizeof(*event_info))

/*
 * Function: log_event_name
 * Parameter: ev_num - event number whose name is to be returned
 * Returns: textual name for event ev_num
 */

char *
log_event_name(int ev_num)
{
    if (ev_num < 1 || ev_num > (int) num_events) {
	return "UNKNOWN EVENT TYPE";
    }

    return event_info[ev_num - 1].ev_name;
}


/*
 * Function: log_event_type
 * Parameter: ev_num - event number whose name is to be returned
 * Returns: textual name for event ev_num
 */

evtype_t
log_event_type(int ev_num)
{
    if (ev_num < 1 || ev_num > (int) num_events) {
	return notype;
    }

    return event_info[ev_num - 1].ev_type;
}
    
