static char rcsid[] = "$Id: symlib.c,v 1.1 1996/03/20 21:39:00 paul Exp $";
/*
 * Based on
 *
 * "@(#)symlib.c 1.2 94/09/12 Paul Ashton"
 */

#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <stdio.h>
#include <assert.h>

#include "symlib.h"

/*
 * Symbol table manipulation library.  This library provides various
 * symbol manipulation functions.  Each table stores <symbol name, user data>
 * pairs, where the user data is a void * supplied by the calling functions.
 * The table takes the form of a hash table with overflow handled using
 * chaining.
 *
 * sym_create_table creates a new symbol table by creating and initialising
 * a new symbol table structure.  A pointer to that structure is returned,
 * and must be passed to all of the other functions that act on symbol tables.
 *
 * XXX: Possible enhancement - make the table size a parameter 
 * of sym_create_table().
 *
 * Change history:
 *
 * Who  Date        What
 * ======================================================================
 * PJA  15/7/91    Changed .h file inclusion to allow sun 4/4.1.1 compiles.
 *
 * PJA  21/10/94   Ported to Amoeba.
 */

#define TABLE_SIZE 997

struct sym {
    char *sym_name;
    void *user_data;
    struct sym *sym_link;
} ;

struct symbol_table {
    struct sym *hash_table[TABLE_SIZE];
    int curr_entry;                 /* for traverse */
    struct sym *curr_sym;           /* for traverse */
};

static struct sym **sym_locate(symptr sym_table, char *sym_name);
static void sym_advance(symptr sym_table);
static int sym_hash(const char *symbol);
static int sym_check(const char *symbol);
static void sym_free(struct sym *sym_ptr, tidyfunc tidyup);


/*
 * For passing error numbers back to the caller (analogous to errno in
 * Unix system calls).
 */
int sym_errno;


/*
 * Function: sym_create_table
 * Parameters: none
 * Returns: pointer to new symbol_table struct (sym_errno = SERR_OK) if
 *          the struct is malloc'ed sucessfully, exits (via assert)
 *          if malloc fails.
 */

symptr
sym_create_table(void)
{
    symptr new_table;
    int i;

    sym_errno = SERR_OK;
    new_table = malloc(sizeof(*new_table));
    assert(new_table);

    for (i = 0 ; i < TABLE_SIZE ; i++) {
	new_table->hash_table[i] = 0;
    }

    new_table->curr_entry = -1;        /* traverse not in progress */
    new_table->curr_sym = 0;
    return new_table;
}


/*
 * Function: sym_find
 * Parameters: sym_table - symbol table to look in
 *             sym_name - symbol to look for
 * Returns: user data for sym_name (sym_errno = SERR_OK) if ok,
 *          NULL and sym_errno set appropriately if error.
 *
 * Note that NULL is a valid value of user_data, so sym_errno must be
 * used to determine whether an error ocurred or not.
 */

void *
sym_find(symptr sym_table, char *sym_name)
{
    struct sym **sym_ptr = sym_locate(sym_table, sym_name);

    return *sym_ptr ? (*sym_ptr)->user_data : NULL;
}


/*
 * Function: sym_add
 * Parameters: sym_table - symbol table to add symbol to
 *             sym_name - name of symbol to add
 *             user_info - user data for symbol
 * Returns: 0 (sym_errno = SERR_OK) if ok, -1 (sym_errno set to error code)
 *          if error.
 */

int
sym_add(symptr sym_table, char *sym_name, void *user_info)
{
    struct sym *new_sym;
    struct sym **sym_ptr = sym_locate(sym_table, sym_name);

    /*
     * ensure that the symbol is not already in the table
     * Note that sym_locate validates the symbol name.
     * Also note that sym_locate sets the value of sym_ptr such that
     * *sym_ptr should be assigned the address of a new sym struct for
     * sym_name.
     */
    if (sym_errno != SERR_NOT_FOUND) {
	if (sym_errno == SERR_OK) {
	    sym_errno = SERR_ALREADY_DEFINED; 
	}
	return -1;
    }

    sym_errno = SERR_OK;

    new_sym = malloc(sizeof(struct sym));
    assert(new_sym);
    new_sym->sym_name = malloc(strlen(sym_name) + 1);
    assert(new_sym->sym_name);
    strcpy(new_sym->sym_name, sym_name);
    new_sym->user_data = user_info;
    new_sym->sym_link = 0;
    *sym_ptr = new_sym;
    return 0;
}


/*
 * Function: sym_change_data
 * Parameters: sym_table - symbol table to use
 *             sym_name - name of symbol whose data is to be changed
 *             user_info - new user data for symbol.
 * Returns: old user data for sym_name (sym_errno = SERR_OK) if ok,
 *          NULL and sym_errno set appropriately if error.
 *
 * Note that NULL is a valid value of user_data, so sym_errno must be
 * used to determine whether an error ocurred or not.
 */

void *
sym_change_data(symptr sym_table, char *sym_name, void *user_info)
{
    void *old_user_info;
    struct sym **sym_ptr = sym_locate(sym_table, sym_name);

    if (sym_errno != SERR_OK) {
	return NULL;
    }
    
    old_user_info = (*sym_ptr)->user_data;
    (*sym_ptr)->user_data = user_info;
    return old_user_info;
}


/*
 * Function: sym_delete
 * Parameters: sym_table - table to delete symbol from
 *             sym_name - symbol to delete
 *             tidyup - function to call with the user_data of the
 *                      symbol as a parameter so that any tidyup action
 *                      required can be performed.  tidyup maybe 0.
 * Returns: 0 (and sym_errno = SERR_OK) on success, -1 (and sym_errno set
 *          appropriately) on failure.
 */
 
int
sym_delete(symptr sym_table, char *sym_name, tidyfunc tidyup)
{
    struct sym **sym_ptr = sym_locate(sym_table, sym_name);
    struct sym *del_sym;

    if (sym_errno != SERR_OK) {
	return -1;
    }

    /*
     * If a traverse is in progress, and the current symbol is the
     * one to be deleted then we must advance the traverse to the next
     * symbol.
     */
    del_sym = *sym_ptr;
    if (del_sym == sym_table->curr_sym) {
	sym_advance(sym_table);
    }
	
    *sym_ptr = del_sym->sym_link;
    sym_free(del_sym, tidyup);
    return 0;
}


/*
 * Function: sym_setup_traverse
 * Parameter: sym_table - a symbol table to setup a traverse for
 * Returns: nothing.
 *
 * Initialise the symbol table for a traverse.  curr_entry gets set to the
 * index of the first hash table entry that is occupied, and curr_sym is 
 * set to point to that entry.  If the table is empty, curr_entry will
 * end up as TABLE_SIZE, and the first call to sym_next will get an indication
 * that there are no further entries to traverse.
 */
 
void 
sym_setup_traverse(symptr sym_table)
{
    sym_errno = SERR_OK;
    sym_table->curr_entry = -1;
    sym_table->curr_sym = 0;
    sym_advance(sym_table);
}


/*
 * Function: sym_next
 * Parameters: sym_table - symbol table
 *             sym_name - pointer to a char * to set to point to the name
 *                        of the next symbol.  A null pointer can be passed.
 * Returns: the user_data pointer of the next symbol (this may be null)
 *          (sym_errno = SERR_OK) if there is a next symbol, 0 (sym_errno =
 *          SERR_NO_MORE_SYMS) if there is no next symbol.
 */

void *
sym_next(symptr sym_table, char **sym_name)
{
    struct sym *sym_ptr;
    sym_errno = SERR_OK;
    
    if ((sym_ptr = sym_table->curr_sym) == 0) {
	sym_errno = SERR_NO_MORE_SYMS;
	return 0;
    }
    
    sym_advance(sym_table);

    if (sym_name) {
	*sym_name = sym_ptr->sym_name;
    }
    return sym_ptr->user_data;
}


/*
 * Function: sym_delete_table
 * Parameters: sym_table - table to delete 
 *             tidyup - function to call for each symbol in the table.
 *                      Purpose and parameters as for sym_delete
 *                      Tidyup maybe 0.
 *
 * Deletes, and calls tidyup for, all symbols, then frees the table itself.
 */

void
sym_delete_table(symptr sym_table, tidyfunc tidyup)
{
    int i;
    struct sym *sym_ptr, *next_sym;
    
    for (i = 0; i < TABLE_SIZE; i++) {
	for (sym_ptr = sym_table->hash_table[i]; sym_ptr; sym_ptr = next_sym) {
	    next_sym = sym_ptr->sym_link;
	    sym_free(sym_ptr, tidyup);
	}
    }
    free(sym_table);
    sym_errno = SERR_OK;
}


/*
 * Function: sym_locate
 * Parameters: sym_table - table to search in
 *             sym_name - name to search for
 * Returns: pointer to pointer which points to the table entry for sym_name.
 *          If sym_name not found then points to the pointer that should be
 *          set to a new node containing sym_name.  sym_errno reports the
 *          result of the search (illegal name, not found, OK).
 */

static struct sym **
sym_locate(symptr sym_table, char *sym_name)
{
    struct sym **sym_ptr;

    sym_errno = SERR_OK;

    if (sym_check(sym_name)) {
	sym_errno = SERR_ILLEGAL_NAME;
	return 0;
    }

    sym_ptr = &sym_table->hash_table[sym_hash(sym_name)];
    while (*sym_ptr && strcmp(sym_name, (*sym_ptr)->sym_name) != 0) {
	sym_ptr = &(*sym_ptr)->sym_link;
    }

    if (*sym_ptr == 0) {
	sym_errno = SERR_NOT_FOUND;
    }
    return sym_ptr;
}


/*
 * Function: sym_advance
 * Parameter: sym_table - symbol table
 * Returns: nothing
 *
 * Advance curr_entry and curr_sym during a traverse.
 */

static void
sym_advance(symptr sym_table)
{
    if (sym_table->curr_entry == TABLE_SIZE) {
	sym_table->curr_sym = 0;
	/*
	 * Have already advanced to the end of the table.
	 */
	return;
    }

    if (sym_table->curr_sym && sym_table->curr_sym->sym_link) {
	/*
	 * Can simply advance to the next record in a chain.
	 */
	sym_table->curr_sym = sym_table->curr_sym->sym_link;
	return;
    }

    /*
     * Otherwise, look for the next used hash table entry.
     * This relies on the fact that befoer a traverse starts
     * curr_entry is -1.  Also curr_sym should be 0 for the 
     * previous test to work.
     */
    do {
	sym_table->curr_entry++;
    } while (sym_table->curr_entry < TABLE_SIZE &&
             sym_table->hash_table[sym_table->curr_entry] == 0);
	
    if (sym_table->curr_entry == TABLE_SIZE) {
	sym_table->curr_sym = 0;
    } else {
	sym_table->curr_sym = sym_table->hash_table[sym_table->curr_entry];
    }
}


/*
 * Function: sym_hash
 * Parameter: symbol - symbol to determine hash value of
 * Returns: position in hash table to use
 *
 * Treats symbol as a base 128 number and returns symbol % TABLE_SIZE
 */

static int
sym_hash(const char *symbol)
{
    int result = 0;

    for (; *symbol; symbol++) {
	result = (result * 128 + *symbol) % TABLE_SIZE;
    }
    return result;
}


/*
 * Function: sym_check
 * Parameter: symbol - symbol whose validity should be checked
 * Returns: 0 if symbol valid (length > 0, all characters printable),
 *          -1 otherwise.
 */

static int
sym_check(const char *symbol)
{
    if (!*symbol) {
	return -1;
    }

    for (; *symbol; symbol++) {
	if (!isprint(*symbol)) {
	    return -1;
	}
    }
    return 0;
}


/*
 * Function: sym_free
 * Parameters: sym_ptr - node to be freed
 *             tidyup - function to call with the user_data of the
 *                      symbol as a parameter so that any tidyup action
 *                      required can be performed.  Maybe 0.
 * Returns: nothing
 */

static void
sym_free(struct sym *sym_ptr, tidyfunc tidyup)
{
    if (tidyup) {
	(*tidyup)(sym_ptr->user_data);
    }
    free(sym_ptr->sym_name);
    free(sym_ptr);
}
