/*
 * $Id: totcl.c,v 1.2 1996/06/04 08:30:25 paul Exp $
 *
 * Based on "@(#)toxgrab.c 1.2 94/10/26 Paul Ashton";
 */

/*
 * totcl
 * =====
 *
 * This program takes the events in a logfile which together make up
 * an interaction network, and produces an output file containing
 * a form suitable for display by a tcl program
 *
 * totcl [-p] logfile [logfile ...]
 *
 * If -p is specified then the y-coordinates are computed with a gap of
 * 1 from the latest preceding node.  In this case the partial order
 * (Lamport's happened before relationship) is preserved, in that if
 * event A "happened before" event B then the the vertex of event A will have
 * a lower y-coordinate than that of the vertex of event B.  For "concurrent"
 * events there is no guarantee on ordering.
 *
 * If the -p option is not specified, y-coordinates are simply offsets
 * in microseconds from the beginning of the interaction.
 *
 * NOTE:  Times used here are stored in unsigned integers in tens of
 *        microseconds enough accuracy for interactions of length close
 *        to 12 hours (11.93).
 *
 * Change history:
 *
 * Who  Date       What
 * ======================================================================
 * PJA  May 91     Initial coding
 *
 * PJA  20/6/91    Added support for virtual arcs to ensure that process
 *                 "bursts" have their partial order preserved.
 *
 * PJA  15/7/91    Changed .h file inclusion to allow sun 4/4.1.1 compiles.
 *
 * PJA  26/10/94   Converted to the Amoeba version
 *
 * CIM  950814     write_nodes modified to print extra info.
 */

#include <assert.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#include "../../kernel/logging.h"

#include "logfiles.h"
#include "symlib.h"
#include "IN.h"
#include "messages.h"

#define HEAD_GAP 5
#define PROC_SPACE 1
#define EVENT_SPACE 1

/*
 * flags to specify how to calculate the y (time) coordinate.
 */
#define CALC_TIME 1
#define CALC_PARTIAL_ORDER 2

#define SUFFIX ".tcl_in"

#define USAGE "Usage: %s -p logfile [logfile ...]\n"


/*
 * A structure to contain additional event info that totcl needs
 * tacked on to the INnode_t structure via the user_hook pointer
 */

struct extra_info {
    unsigned int x, y;                /* event coordinates */
    unsigned int next_is_virtual:1,   /* used to flag virtual edges */
                 fork_is_virtual:1;
};


/*
 * A structure to hold information about process bursts.  It is structured
 * as a linked list of burst_start, burst_end pointers.  Bursts are relevant
 * only when the -p option has been specified.
 */

struct burst {
    INnode_t *burst_start, *burst_end;
    double start_time;
    struct burst *next;
};

struct burst_head {
    struct burst *head;
};

static char *progname;

static int yspacing = CALC_TIME;

static unsigned max_xpos = 0;
static unsigned max_ypos = 0;

static INnode_t *in;     /* global to allow access to host info */

static void usage(void);

static int  output_IN(char *log_file, char *outputdir);

static void add_virtual_links(INnode_t *ip);
static void find_bursts(INnode_t *ip, symptr proc_table);
static void follow_message(INnode_t *ip, symptr proc_table);
static void free_bursts(void *vp);

static void write_file(INnode_t *in, FILE *outfile);
static void write_nodes(INnode_t *ip, FILE *outfile);
static void write_edges(INnode_t *ip, FILE *outfile);
#if 0
static char *line_type(INnode_t *ip1, INnode_t *ip2);
#endif

static void assign_y_coords(INnode_t *ip, unsigned int ypos);
static int  assign_x_coords(INnode_t *ip, symptr xcoords,
			    unsigned int x0, unsigned int y0,
			    unsigned int *x1, unsigned int *y1);
static unsigned int calc_xcoord(INnode_t *ip, symptr xcoords);
static unsigned int next_proc_posn(void);

static void  totcl_sym_add(symptr sym_table, int pnr, unsigned int coord);
static char *totcl_sym_find(symptr sym_table, int pnr, int error_fatal);

#define max(x, y) ((x) > (y) ? (x) : (y))

#define timetof(ev) ((ev).e_secs * 1000000.0 + (ev).e_usecs)

/*
 * Function: main
 * Arguments: argc, argv - see USAGE for expected arguments
 * Returns: 0 if all OK, exits with a status of 1 if a fatal
 *          error is encountered.
 *
 * Responsible for processing the command line options, and for 
 * having each logfile specified in the command line processed
 */

int
main(int  argc, char *argv[])
{
    char   *outputdir = ".";

    progname = argv[0];

    /*
     * Check and process arguments. 
     */
    {
	int   option;
	extern int optind;
	
	while ((option = getopt(argc, argv, "p")) != -1)
	  switch(option) {
	    case 'p':
	      /*
	       * -p specifies process spacing for y coords
	       */
	      yspacing = CALC_PARTIAL_ORDER;
	      break;
      
	    default:
	      /*
	       * unknown argument.
	       */
	      usage();
	  }
	
	argc -= optind;       /* set to the number of args remaining */
	argv += optind;       /* step over already processed args */
    }

    if (argc < 1) {
	/*
	 * Whoops - no files to process
	 */
	(void) fprintf(stderr, USAGE, progname);
	exit(1);
    }

    while (argc > 0) {
	if (output_IN(*argv, outputdir) == -1) {
	    (void) fprintf(stderr, "%s: failed to output IN for %s\n",
			   progname, *argv);
	}
	argv++, argc--;
    }

    return 0;
}


static void
usage(void)
{
    fprintf(stderr, USAGE, progname);
    exit(1);
}


/*
 * Function: output_IN
 * Parameters: log_file - log file to read IN events from
 *             outputdir - directory to write output file to
 * Returns: 0 if OK, -1 if error.
 *
 * Reads an IN from log_file, opens the output file, writes
 * the file type to it (method of y-coordinate calculation), then calls 
 * write_file to do the rest of the work.
 *
 * XXX: Should the output file be unlinked if some error occurs?
 */

static int
output_IN(char *log_file, char *outputdir)
{
    FILE *out_file;

    if ((in = log_readIN(log_file)) == 0) {
	return -1;
    }

#ifdef notdef
    cp_find(in);           /* find the critical path */
#endif
    assert(in);

    /*
     * Virtual links are added to ensure that the ordering of process
     * bursts within a process are preserved for partial order
     * y-coords.  This is not necessary for time-based y-coords as
     * the y-coord is based on the time this preserves the ordering
     * of process bursts.
     */
    if (yspacing == CALC_PARTIAL_ORDER) {
	add_virtual_links(in);    /* add virtual links */
    }

    {
	char *basepos = strrchr(log_file, '/');
	char *output_file;

	/*
	 * Just take base of the log file name
	 */
	if (basepos) {
	    basepos++;             /* step over '/' */
	} else {
	    basepos = log_file;    /* no '/' in log_file */
	}

	output_file = malloc(strlen(basepos) + strlen(outputdir) +
			     strlen(SUFFIX) + 2);
	assert(output_file);
	sprintf(output_file, "%s/%s%s", outputdir, basepos, SUFFIX);
	out_file = fopen(output_file, "w");
	free(output_file);
	if (out_file == 0) {
	    return -1;
	}
    }

    fprintf(out_file, "%s\n", yspacing == CALC_TIME ? "T" : "P");
    write_file(in, out_file);
    log_freeIN(in, free);
    (void) fclose(out_file);
    return 0;
}


/* 
 * Function: add_virtual_links
 * Parameter: ip - pointer to an interaction network
 * Returns: nothing
 *
 * Adds virtual links to represent timing constraints between separate
 * bursts of process executions for a sub-task.
 *
 * Warnings:
 *   1. add_virtual_links doesn't setup prev and join innode ptrs
 *   2. Any lib function used subsequently will treat links added by
 *      add_virtual_links as real.
 */

static void
add_virtual_links(INnode_t *ip)
{
    symptr proc_table = sym_create_table();
    struct burst_head *head;
    struct burst *bp;
    struct extra_info *xp;

    find_bursts(ip, proc_table);
    sym_setup_traverse(proc_table);
    while (head = sym_next(proc_table, 0), sym_errno == SERR_OK) {
	assert(head);

	/*
	 * Note that we assume that events for a thread are linked strictly
	 * through their next links - this assumption is tested
	 * by the assertion.
	 */
	for (bp = head->head; bp->next != 0; bp = bp->next) {
	    assert(bp->burst_end->next == 0);
	    bp->burst_end->next = bp->next->burst_start;
	    bp->next->burst_start->ref_cnt++;
	    xp = bp->burst_end->user_hook;
	    if (xp == 0) {
		bp->burst_end->user_hook = malloc(sizeof(struct extra_info));
		assert(bp->burst_end->user_hook);
		xp = bp->burst_end->user_hook;
		xp->fork_is_virtual = 0;
		xp->x = xp->y = 0;
	    }
	    xp->next_is_virtual = 1;
	}
    }
    sym_delete_table(proc_table, free_bursts);
}


/*
 * Function: find_bursts
 * Parameters: ip - a sub-task to search for execution bursts
 *             proc_table - a symbol table mapping processes to burst
 *                          lists
 * Returns: nothing
 *
 * A process "burst" can start in one of three ways:
 *    - the IN source event is the start of a process burst
 *    - a NEW_THREAD is the start of a process burst for the created thread
 *    - a msg rcv is part of a process burst if the process is not
 *      currently performing a process burst.
 *
 * ip is an event that is the start of a process burst.  The process
 * burst is followed until it finishes, with fork events followed
 * as they are encountered.  The process burst is recorded in the
 * appropriate linked list.
 */

static void
find_bursts(INnode_t *ip, symptr proc_table)
{
    struct burst *bp;
    int pnr;
    
    bp = malloc(sizeof(*bp));
    assert(bp);
    bp->burst_start = bp->burst_end = ip;
    bp->start_time = timetof(ip->ev);
    bp->next = 0;

    /*
     * Traverse the burst, calling find_bursts recursively as appropriate.
     */
    for (pnr = ip->ev.e_pnr; ip && ip->ev.e_pnr == pnr; ip = ip->next) {
	bp->burst_end = ip;
	if (ip->fork) {
	    if (ip->ev.e_event == EV_FORK) {
		/*
		 * Because ip->fork is true there is at least one event
                 * in the new thread after the EV_FORK.
		 * This means that the event after the
		 * EV_FORK is recorded as the burst
		 * start event, but this shouldn't cause
		 * any problems because it is the first
		 * burst, and the burst_start in the
		 * first event is used.
		 */
		find_bursts(ip->fork, proc_table);
	    } else {
		follow_message(ip->fork, proc_table);
	    }
	}
    }	
		
    if (ip) {
	/*
	 * The process burst has become a message - probably
	 * an exit one.
	 */
	follow_message(ip, proc_table);
    }

    /*
     * At this point we have our burst info - all we need to do
     * is to slot it into the appropriate list.  Note that lists are
     * sorted into increasing time order.
     */
    {
	struct burst_head *head;
	
	head = (struct burst_head *)
	  totcl_sym_find(proc_table, bp->burst_start->ev.e_pnr, 0);

	if (head == 0) {
	    /*
	     * The first burst
	     */
	    head = malloc(sizeof(*head));
	    assert(head);
	    head->head = bp;
	    log_stn_sym_add(proc_table, bp->burst_start->ev.e_pnr, head);
	} else {
	    struct burst **curr_pos = &head->head;

	    while (*curr_pos && (*curr_pos)->start_time < bp->start_time) {
		curr_pos = &(*curr_pos)->next;
	    }
	    if (*curr_pos && (*curr_pos)->start_time == bp->start_time) {
		/*
		 * A duplicate - this can happen for
		 * reasons outlined in follow_message.
		 */
		free(bp);
	    } else {
		bp->next = *curr_pos;
		*curr_pos = bp;
	    }
	}
    }
}


/*
 * Function: follow_message
 * Parameters: ip - message event which we are following to look
 *                  for the start of process bursts.
 *             proc_table - symbol table of burst info
 * Returns: nothing
 *
 * Follows down a list of message events.  If a fork event is encountered then
 * follow_message is called to follow it.  When a process event is reached it
 * is checked to see if it is the start of a burst.  If so then find_bursts
 * is called to find and record the burst.  If not then follow_message
 * finishes - the process event will be traversed via a path that includes
 * the start of its burst.
 */

static void
follow_message(INnode_t *ip, symptr proc_table)
{
    /*
     * search for the start of a burst
    while (ip && ip->ev->process == EV_INVALID_PTID) {
	if (ip->fork) {
	    follow_message(ip->fork, proc_table);
	}
	ip = ip->next;
    }
     */

    /*
     * If ip is not null then we have reached a process-related event.
     * If neither the previous or join events belong to the same process 
     * then assume this is the start of a burst.  This method means
     * multi-way joins that are the start of a process burst will
     * be attempted to be included multiple times (but currently
     * there are no such events).  find_bursts filters out duplicates.
     */
    if (ip && !(ip->join && ip->join->ev.e_pnr == ip->ev.e_pnr) &&
	!(ip->prev && ip->prev->ev.e_pnr == ip->ev.e_pnr)) {
	find_bursts(ip, proc_table);
    }
}
	
	  
/*
 * Function: free_bursts
 * Parameter: vp - pointer to burst list to free
 * Returns: nothing
 */

static void
free_bursts(void *vp)
{
    struct burst *bp, *next;

    next = ((struct burst_head *)vp)->head;
    free(vp);
    while (next) {
	bp = next;
	next = bp->next;
	free(bp);
    }
}


/*
 * Function: write_file
 * Parameters: ip - interaction network to write out
 *             outfile - file to write it out to
 * Returns: nothing
 *
 * Calls functions to have x and y coordinates assigned to each event,
 * then calls functions to write out the nodes and edges.
 * The xcoords symbol table is used to record the x-coordinate assigned
 * to each thread.
 */

static void
write_file(INnode_t *in, FILE *outfile)
{
    symptr xcoords = sym_create_table();

    assert(in != 0);
    
    /*
     * Add the "0" proc - that to which all messages that never 
     * arrive at a thread (so far as the event records we have are
     * concerned) are destined.
     */
    totcl_sym_add(xcoords, 0, next_proc_posn());

    /*
     * calculate the x and y coordinates of each node.
     * The initial x0 is the "dangling comms" x0.
     */
    {
	unsigned int xdummy, ydummy;
	unsigned int x0 = *(unsigned int *)totcl_sym_find(xcoords, 0, 1);

	assign_y_coords(in, 0);
	log_IN_visited_reset(in);
	(void) assign_x_coords(in, xcoords, x0, 0, &xdummy, &ydummy);
	log_IN_visited_reset(in);
	sym_delete_table(xcoords, free);
    }

    /*
     * First print out maximum x and y coordinates
     */
    fprintf(outfile, "%d %d\n", max_xpos + 1, max_ypos + 1);

    /*
     * Now print our nodes and edges.  In each case we print a line
     * to indicate the end of that section of data,
     */
    write_nodes(in, outfile);
    log_IN_visited_reset(in);
    fprintf(outfile, "-1 -1 -1 DUMMY\n");

    write_edges(in, outfile);
    log_IN_visited_reset(in);
    fprintf(outfile, "-1 -1 -1\n");
}


/*
 * Function: write_nodes
 * Parameters: ip - interaction network to write node information from
 *             outfile - file to write it out to
 * Returns: nothing
 *
 * Traverses the interaction network writing out node outputs.  Assumes
 * that ip->visited is 0 for all nodes in the interaction network
 * before this function is first called.  
 */

static void
write_nodes(INnode_t *ip, FILE *outfile)
{
    struct extra_info *xp;
    char msg[80];


    if (ip == 0 || ip->visited) {
	return;
    }
    ip->visited = 1;

    xp = ip->user_hook;
    fprintf(outfile, "%d %d %d ", ip->seq_no, xp->x, xp->y);
    fprintf(outfile, "smx 0 %d %d %s, stn = %d, time = %d %06d",
	    ip->ev.e_pnr, ip->seq_no, log_event_name(ip->ev.e_event),
	    ip->ev.e_stn, ip->ev.e_secs, ip->ev.e_usecs);
    /* Generate process name */

    switch(ip->ev.e_event) {
    case EV_TTY_INPUT:
	if (ip->rest) {
	    fprintf(outfile, " Input = `");
	    fwrite(ip->rest, 1, ip->ev.e_bufflen, outfile);
	    fprintf(outfile, "'");
	}
	break;
    case EV_MSG_RECV:
        if (ip->rest) {
            fprintf(outfile, " Message = ");
	    message_is(ip->ev.e_pnr, *(int *)ip->rest, msg, sizeof(msg));
	    fprintf(outfile,"%s",msg);
        }
        break;
    case EV_NEW_NAME:
        if (ip->rest) {
	    fprintf(outfile, " New name = \"%s\"", (char *)ip->rest);
        }
        break;
    }
    fprintf(outfile, "\n");
    write_nodes(ip->next, outfile);
    if (ip->fork) {
	write_nodes(ip->fork, outfile);
    }
}
     

/*
 * Function: write_edges
 * Parameters: ip - interaction network whose edge details are to be written
 *             outfile - file to write edge details to
 * Returns: nothing
 *
 * Traverses the interaction network writing out 1 line per edge.  Assumes
 * that ip->visitied is 0 for all nodes in the interaction network
 * before this function is first called.
 */

static void
write_edges(INnode_t *ip, FILE *outfile)
{
    struct extra_info *xp;
    static int edge_seq_no = 0;
    
    for (; ip && !ip->visited; ip = ip->next) {
	ip->visited = 1;
	xp = ip->user_hook;
	if (ip->next && !xp->next_is_virtual) {
	    fprintf(outfile, "%d %d %d\n", edge_seq_no++, ip->seq_no,
		    ip->next->seq_no);
	}
	if (ip->fork && !xp->fork_is_virtual) {
	    fprintf(outfile, "%d %d %d\n", edge_seq_no++, ip->seq_no,
		    ip->fork->seq_no);
	    write_edges(ip->fork, outfile);
	}
    }
}


#if 0
/* Not used at present---something similar may be useful when CP's
 * are added to Amoeba INMON.
 */
/*
 * Function: line_type
 * Parameters: ip1, ip2 - two nodes which are to be connected by an arc
 *                        from ip1 to ip2.
 * Returns: a pointer to a string which is the xgrab "brush string"
 *          for the arc.
 *
 * If both events are on the CP then the arc is part of the CP, and
 * is dashed.  Otherwise the line is solid.
 */

static char *
line_type(INnode_t *ip1, INnode_t *ip2)
{
    if (ip1->is_on_CP == CP_NEXT && ip1->next == ip2 ||
	ip1->is_on_CP == CP_FORK && ip1->fork == ip2) {
	return "dotted";
    } else {
	return "solid";
    }
    return "solid";
}
#endif


/*
 * Function: assign_y_coords
 * Parameters: ip - node to assign y coordinate to
 *             ypos - coordinate to use for calaculating y coords on the
 *                    basis of partial order.
 * Returns: nothing.
 *
 * For CALC_TIME, the y coord is calculated based on the time that the
 * event occurred.  The y coordinate is simply the number of microseconds
 * since the beginning of the interaction. If the node had been previously
 * visited then the traversal stops at this node.
 *
 * For CALC_PARTIAL_ORDER, the y coordinate is supplied by the ypos parameter.
 * If the node had been visited, but the ypos is later than the current y
 * coord, then ypos is used as the y coord and the traversal continues to
 * propagate the new ypos to subsequent nodes.  When the recusrsive call
 * is made an appropriate value is subtracted added ypos to arrive at the
 * ypos for the next event(s) lower in the tree.
 */

static void
assign_y_coords(INnode_t *ip, unsigned int ypos)
{
    struct extra_info *xp;

    if (ip == 0) {
	return;
    }
    
    if ((xp = ip->user_hook) == 0) {
	xp = ip->user_hook = malloc(sizeof(struct extra_info));
	assert(ip->user_hook);
	xp->next_is_virtual = xp->fork_is_virtual = 0;
    }

    if (yspacing == CALC_TIME) {
	if (ip->visited) {
	    return;
	}
	xp->y = timetof(ip->ev);
    } else {
	if (ip->visited && ypos < xp->y) {
	    /*
	     * if we've already been there, and the y position
	     * suggested is above the current one then
	     * give up - otherwise continue to
	     * propagate the new y
	     */
	    return;
	}
	xp->y = ypos;
    }
    
    max_ypos = max(max_ypos, xp->y);
    ip->visited = 1;

    assign_y_coords(ip->next, ypos + EVENT_SPACE);
    if (ip->fork) {
	assign_y_coords(ip->fork, ypos + EVENT_SPACE);
    }
}


/*
 * Function: assign_x_coords
 * Parameters: ip - interaction network to be traversed
 *             xcoords - symbol table of process xcoords
 *             x0 - xcoord of sending process
 *             y0 - time of sending process
 *             x1 - xcoord of receiving process
 *             y1 - time of receiving process
 * Returns: 0 if ip is null, 1 (and x1 and time1 are assigned values) otherwise
 *
 * Calculates the x coordinates for an event node.  For process-related nodes,
 * and comms-related nodes that are sinks, the calc_xcoord function is
 * used to determine the x coordinate.  For other types of comms-related
 * nodes, the process end-points are determined thru x0 and x1, and the xcoord
 * is set to be between these two in proportion to the time of the event
 * and time0 and time1.
 */

static int
assign_x_coords(INnode_t *ip, symptr xcoords, unsigned int x0, unsigned int y0,
		unsigned int *x1, unsigned int *y1)
{
    struct extra_info *xp;
    unsigned int xnext, xfork, ynext, yfork;
    int ret_next, ret_fork = 0;

    if (ip == 0) {
	return 0;
    }

    xp = ip->user_hook;
    if (ip->visited) {
	*x1 = xp->x;
	*y1 = xp->y;
	return 1;
    }

    ip->visited = 1;

    if (!non_process_event(ip->ev.e_event)) {
	/*
	 * for an event associated with a process we
	 * can calculate x immediately.  Set
	 * both the x0, y0 details passed downwards,
	 * and the x1, y1 details passed upwards
	 * to be x and y of this vertex.
	 */
	y0 = *y1 = xp->y;
	xp->x = x0 = *x1 = calc_xcoord(ip, xcoords);
    }

    ret_next = assign_x_coords(ip->next, xcoords, x0, y0, &xnext, &ynext);
    if (ip->fork) {
	ret_fork = assign_x_coords(ip->fork, xcoords, x0, y0, &xfork, &yfork);
    }

    if (non_process_event(ip->ev.e_event)) {
	/*
	 * Have to calculate the x coordinate of communication
	 * events after all of the subsequent events have been decided
	 */
	if (ret_next == 0 && ret_fork == 0) {
	    /*
	     * A comms event which is a sink!  Make its
	     * x the "comms event sink" column.  Set x1 and
	     * y1 as this is the last event in the comm
	     * chain.
	     */
	    xp->x = *x1 = calc_xcoord(ip, xcoords);
	    *y1 = xp->y;
	} else {
	    double percent;
			
	    /*
	     * Just work out our own xcoord, and pass the x1
	     * and y1 we use back up.  If there is a
	     * fork child but not a next child, use the
	     * x and time returned by the fork child,
	     * otherwise use the x and time
	     * returned by the next child.
	     */
	    if (ret_next == 0 && ret_fork != 0) {
		*x1 = xfork;
		*y1 = yfork;
	    } else {
		*x1 = xnext;
		*y1 = ynext;
	    }
	    /*
	     * set percent to percentage of time this event
	     * is between y0 and y1, and xp->x to be the
	     * proportional difference (based on the y
	     * coord) between x0 and x1.
	     */
	    if ((int) xp->y - (int) y0 == 0) {
		percent = 0.5;
	    } else {
		percent = (double) ((int) xp->y - (int) y0) /
		  ((int) *y1 - (int) y0);
	    }
	    xp->x = ((int) *x1 - (int) x0) * percent + x0;
	}
    }
    return 1;
}


/*
 * Function: calc_xcoord
 * Parameters: ip - an innode to calculate the xcoord for
 *             xcoords - a symbol table containing the xccords for all
 *                       (process, machine) pairs encountered to date.
 * Returns: the xcoord for ip
 *
 * If the event is a comms one the just return the xcoord of the "comms"
 * dummy process.  Otherwise look to see whether this (host, thread) has
 * an entry in xcoords.  If so then just return this value.  If not
 * allocate a new one, record it in xcoords, and return it.
 */

static unsigned int
calc_xcoord(INnode_t *ip, symptr xcoords)
{
    unsigned int retval;
    unsigned int *coordp;

    if (non_process_event(ip->ev.e_event)) {
	retval = *(unsigned int *)totcl_sym_find(xcoords, 0, 1);
    } else {
	coordp = (unsigned int *) totcl_sym_find(xcoords, ip->ev.e_pnr, 0);
	if (coordp != 0) {
	    retval = *coordp;
	} else {
	    /*
	     * A new <host, thread> combo - allocate a new
	     * coord, and store it in the symbol table
	     */
	    retval = next_proc_posn();
	    max_xpos = max(retval, max_xpos);
	    totcl_sym_add(xcoords, ip->ev.e_pnr, retval);
	}
    }
    return retval;
}


/*
 * Function: next_proc_posn
 * Parameters: none
 * Returns: the next available xcoordinate for a process column
 */

static unsigned int
next_proc_posn(void)
{
    static int proc_posn = 0;
    int ret_val = proc_posn;

    proc_posn += PROC_SPACE;
    return ret_val;
}


/*
 * Function: totcl_sym_add
 * Parameters: sym_table - symbol table
 *             host - host index of the event's machine
 *             tid - thread related to the event
 *             coord - coordinate to add
 * Returns: nothing (note that log_stn_sym_add exits on error).
 *
 * Adds an (id, coord) to sym_table, where id is made up from host
 * and tid.
 */

static void
totcl_sym_add(symptr sym_table, int pnr, unsigned int coord)
{
    int *intp = malloc(sizeof(*intp));

    assert(intp);
    *intp = coord;
    log_stn_sym_add(sym_table, pnr, intp);
}


/*
 * Function: totcl_sym_find
 * Parameter: sym_table - symbol table
 *            host - host index of the event's machine
 *            tid - tid of thread event occurred in
 *            error_fatal - if true abort if symbol not found
 * Returns: pointer to data for inetnum, pid.
 */

static char *
totcl_sym_find(symptr sym_table, int pnr, int error_fatal)
{
    return log_stn_sym_find(sym_table, pnr, error_fatal);
}
