static char rcsid[] = "$Id: config.c,v 1.4 1996/07/09 04:46:53 paul Exp $";

/*
 * This functions in this file are responsible for reading a Minix
 * configuration file and setting up and cleaning up the smx devices.
 * The entry points are:
 *    - config - read the configuration file, and setup smx devices
 *    - prot_atoi - convert a protection level spec from a character string to
 *                  the corresponding integer.
 *    - reset_devices - clean up the console, and the named pipes created
 *                      for remote login.
 *    - enable_interrupts - make all (terminal and ethernet) input
 *                          descriptors send SIGIO on input.
 *
 */

#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <unistd.h>
#include <sys/types.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/termios.h>
#include <sys/file.h>
#include <stropts.h>
#include <sys/conf.h>
#include <sys/param.h>
#include <assert.h>

#undef HZ
#include "../../include/minix/config.h"
#undef NULL /* Defined again in const.h */
#include "../../include/minix/const.h"
#include "../kernel/const.h"   /* Need TERMINAL_FD etc */

#if ENABLE_NETWORKING
#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>
#endif

typedef unsigned char u8_t;
typedef unsigned short u16_t, U16_t;
#include "../../include/net/gen/ether.h"

#include "config.h"
#include "lib.h"


#define COMMENT_CHAR	'#'	/* lines the startup file that start with */
				/* COMMENT_CHAR are ignored */


static int setup_networking(const char *relay_file, ether_addr_t *eaddr);

static char *skipspace(char *ptr);
static char *nextspace(char *ptr);

static void set_sun_tty(void);
static void reset_sun_tty(void);

static void setup_smx_ttys(void);
static void remove_smx_ttys(void);


static char host_dir[MAXPATHLEN]; /* the host name */


/*
 * Function: config
 * Parameters: confname - name of config file specified on the command line
 *                        if any
 *             mem_size - if "memory" option appears, return its value via
 *                        mem_size
 *             image_name - a MAXPATHLEN array into which the value of the
 *                          "image" option (the pathname of the image file
 *                          containing the smx executables) is copied.
 *             prot_lev - if the "protect" option appears, return its value
 *                        via prot_lev
 *             debug - if the "debug" option appears, return its value
 *                     via debug
 *             eaddr - if the "relay_file" option exists, then the simulated
 *                     ethernet is enabled, and the local "address" is
 *                     returned via eaddr
 *
 * Opens the minix configuration file, then reads through it line by
 * line.  For each line that specifies an option that line is parsed
 * and the options it contains processed.   Finally, the controlling
 * SunOS terminal is setup to be the smx console.
 */
void config(const char *confname, long *mem_size, char *image_name,
	    int *prot_lev, int *debug, ether_addr_t *eaddr)
{
    int fd;			    /* general purpose file descriptor */
    FILE *config;		    /* the configuration file */
    char buffer[MAXPATHLEN];	    /* a config file line */
    char *name; 	            /* an option name */
    char *ptr;
    int opened_hdx0 = 0;

    image_name[0] = '\0';

    /*
     * Open the config file.  If a file name was specified use it.  Otherwise
     * try ./.minix, then ~/.minix.
     */
    if (confname != NULL) {
	if ((config = fopen(confname, "r")) == NULL) {
	    fprintf(stderr, "Minix: can't open file: %s\n", confname);
	    exit(1);
	}
    } else if ((config = fopen(".minix", "r")) == NULL && getenv("HOME")) {
	strcpy(buffer, getenv("HOME"));
	strcat(buffer, "/.minix");
	config = fopen(buffer, "r");
    }
  
    if (config == NULL) {
	fprintf(stderr, "Minix: can't open .minix file\n");
	exit(1);
    }
    
    host_dir[0] = '\0';
  
    while(fgets(buffer, sizeof(buffer), config), !feof(config)) {
        /*
	 * for each line in the config file, separate the option name from 
	 * the option's value.
	 */
	ptr = skipspace(buffer);            /* skip leading blanks */

	if (*ptr == '\0') continue; 	    /* skip blank lines */
    
	if (*ptr == COMMENT_CHAR) continue; /* skip comment lines */
    
	/*
	 * get the option name
	 */
	name = ptr;
	ptr = nextspace(ptr);
	if (*ptr == '\0') {
	    fprintf(stderr,
		    "Minix config file: missing value for option '%s'\n",
		    name);
	    exit(1);
	}
	*ptr = '\0';                        /* mark option name end */

	ptr = skipspace(ptr + 1);
	if (*ptr == '\0') {
	    fprintf(stderr,
		    "Minix config file: missing value for option '%s'\n",
		    name);
	    exit(1);
	}
	/*
	 * Currently all option values are a single "word" containing no
	 * white space.  Here we strip trailing stuff off the value.
	 */
	*nextspace(ptr) = '\0';
    
	/*
	 * Check through all of the options
	 */
	if (strcmp("memory", name) == 0) {
	    *mem_size = atoi(ptr);
	    if (*mem_size < 1536 || *mem_size > 32768){
		fprintf(stderr, "Minix config file:invalid memory size %ld\n",
			*mem_size);
		exit(1);
	    }
	    *mem_size *= 1024;
      
	} else if (strcmp("image", name) == 0){
	    strncpy(image_name, ptr, MAXPATHLEN);
	    image_name[MAXPATHLEN - 1] = '\0';    /* Just in case */
      
	} else if (strcmp("host", name) == 0){
	    /*
	     * mlogin's are allowed---setup the FIFOs to allow this.
	     */
	    strncpy(host_dir, "/tmp/" , 5);
	    strncpy(host_dir + 5, ptr, sizeof(host_dir) - 5);
	    setup_smx_ttys();

	} else if (strncmp("hdx", name, 3) == 0){
	    int disk_num;
	    
#if NR_SMX_DISKS > 10
#error Modify this file to handle more than 10 disks
#endif
	    disk_num = (int)(name[3] - '0');
	    if (disk_num < 0 || disk_num > NR_SMX_DISKS - 1 ||
		name[4] != '\0') {
		fprintf(stderr, 
			"Minix config file: invalid disk id: %s\n", name);
		exit(1);
	    }
	    if (disk_num == 0) opened_hdx0 = 1;

	    /*
	     * When opening an smx file-system, try to open it read-write
	     * first.  If that fails, try to open it read-only (it can
	     * still be mounted in smx so long as the -r option is specified
	     * when it is mounted).  The exeception to this is /dev/hdx0,
	     * the root fs, which must be read-write.
	     */
	    if ((fd = open(ptr, O_RDWR)) < 0) {
	        if (disk_num > 0) {
		    fd = open(ptr, O_RDONLY);
		}
		if (fd < 0) {
		    fprintf(stderr,
			   "Minix config file: can't open disk: '%s'\n", ptr);
		    exit(1);
		}
	    }
	    if (dup2(fd, DISK_FD + disk_num) < 0) {
		fprintf(stderr, "Minix config file: dup error\n");
		exit(1);
	    }  
	    close(fd);
            
	} else if (strcmp("protect", name) == 0){
	    *prot_lev = prot_atoi(ptr);
	    
	} else if (strcmp("debug", name) == 0) {
	    if (strcmp("on", ptr) == 0) 
		*debug = 1;
	    else if (strcmp("off", ptr) == 0)
		*debug = 0;
	    else {
		fprintf(stderr, "Minix config file: Bad debug flag\n");
		exit(1);
	    }
      
	} else if (strcmp("logfile", name) == 0) {
	    if ((fd = open(ptr, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0) {
		fprintf(stderr,
			"Minix config file: can't open logfile: '%s'\n", ptr);
		exit(1);
	    }
	    if (dup2(fd, LOGGING_FD) < 0) {
		fprintf(stderr, "Minix config file: dup error\n");
		exit(1);
	    }  
	    close(fd);
	    
	} else if (strcmp("relay_file", name) == 0) {
	    if (setup_networking(ptr, eaddr) < 0) {
		fprintf(stderr, "Failed to setup smx networking\n");
		exit(1);
	    }

	} else {
	    fprintf(stderr, "Minix config file: unknown config option '%s'\n",
		    name);
	    exit(1);
	}  /* end of checking for the various config options */
    
    } /* end of while loop for reading lines */
  
    fclose(config);

    if (opened_hdx0 == 0 || image_name[0] == '\0') {
        fprintf(stderr, 
		"Minix config file must specify image and hdx0 options\n");
	exit(1);
    }
    set_sun_tty();
}    


/*
 * Function: prot_atoi
 * Parameter: prot_str - name of protection option
 * Returns: the protection level that the name corresponds to.  If the name is
 *          invalid, then an error message is printed and exit is called.
 */
int prot_atoi(char *prot_str)
{
    extern char *progname;

    if (strcmp(prot_str, "none") == 0) {
	return NO_PROT;
    } else if (strcmp(prot_str, "half") == 0) {
	return HALF_PROT;
    } else if (strcmp(prot_str, "full") == 0) {
	return FULL_PROT;
    } else {
	fprintf(stderr, "%s: Unknown protection option %s\n",
		progname, prot_str);
	exit(1);
    }
}


/*
 * Function: setup_networking
 * Parameters: relay_file - the name of the file that contains the UDP
 *                          address of the ethernet packet relay program
 *             eaddr - used to return the our ethernet address.
 * Returns: 0 on success, -1 on error.
 *
 * Opens a UDP/IP socket on ETHER_FD to act as the smx ethernet port.
 * relay_file contains the host name and port number of the UDP port
 * at which the relay program is listening.  We send a simulated ethernet
 * packet to the relay to register our own "ethernet address" (the UDP
 * address of the socket opened on ETHER_FD).
 */
static int setup_networking(const char *relay_file, ether_addr_t *eaddr)
{
    int sock;
    struct sockaddr_in saddr;
    int addrlen;
    FILE *rfile;
    char host[MAXHOSTNAMELEN + 1];
    char packet_to_relay[ETH_HDR_SIZE];
    struct hostent *hp;
    int gethostname(char *name, int namelen);    /* not in any .h file */

#if !ENABLE_NETWORKING
    fprintf(stderr, "ENABLE_NETWORKING is 0; relay_file option ignored\n");
    return;
#endif

    if ((sock = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
	return -1;
    }

    if (dup2(sock, ETHER_FD) < 0) {
	return -1;
    }
    close(sock);

    /*
     * Bind an address to our socket.
     */
    saddr.sin_family = AF_INET;
    saddr.sin_port = htons(0);
    saddr.sin_addr.s_addr = htonl(INADDR_ANY);
    if (bind(ETHER_FD, (struct sockaddr *) &saddr, sizeof(saddr)) < 0) {
	return -1;
    }
    addrlen = sizeof(saddr);

    /*
     * Get the port number of our address.
     */
    if (getsockname(ETHER_FD, (struct sockaddr *) &saddr, &addrlen) == -1) {
	return -1;
    }

    /*
     * Get the IP number of our address.
     */
    if (gethostname(host, sizeof(host)) == -1) {
	return -1;
    }
    if ((hp = gethostbyname(host)) == 0) {
	return -1;
    }
    memcpy(&saddr.sin_addr, *hp->h_addr_list, hp->h_length);

    /*
     * Initialise eaddr to our "ethernet address", and the src field
     * of the packet we will send to the relay.
     */
    hosttoa(packet_to_relay + SRC_HOST, ntohl(saddr.sin_addr.s_addr));
    hosttoa((char *) eaddr, ntohl(saddr.sin_addr.s_addr));
    porttoa(packet_to_relay + SRC_PORT, ntohs(saddr.sin_port));
    porttoa((char *) eaddr + sizeof(saddr.sin_addr.s_addr),
	    ntohs(saddr.sin_port));


    /*
     * Read the host and port number of the relay from relay_file.
     */
    if ((rfile = fopen(relay_file, "r")) == 0) {
	return -1;
    }
    /*
     * 256 is MAXHOSTNAMELEN.  I don't like hardwiring it into the format
     * string, but there doesn't seem any easy way around it.
     */
    assert(MAXHOSTNAMELEN == 256);
    if (fscanf(rfile, "%256s %hu", host, &saddr.sin_port)
	!= 2) {
	return -1;
    }

    /*
     * Construct the UDP address of the relay, and connect our socket to
     * it (means we can simly use write(2) calls within the smx ethernet
     * driver).
     */
    saddr.sin_port = htons(saddr.sin_port);
    if ((hp = gethostbyname(host)) == 0) {
	return -1;
    }
    memcpy(&saddr.sin_addr, *hp->h_addr_list, hp->h_length);
    if (connect(ETHER_FD, (struct sockaddr *) &saddr, sizeof(saddr)) < 0) {
	return -1;
    }

    /*
     * Address our registration ethernet packet to the relay itself
     * (so it knows it is a contol message), then send it to the relay.
     */
    hosttoa(packet_to_relay + DEST_HOST, ntohl(saddr.sin_addr.s_addr));
    porttoa(packet_to_relay + DEST_PORT, ntohs(saddr.sin_port));
    
    if (write(ETHER_FD, packet_to_relay, sizeof(packet_to_relay)) !=
	sizeof(packet_to_relay)) {
	return -1;
    }

    if (fcntl(ETHER_FD, F_SETFL, O_NDELAY | O_RDWR) == -1) {
	return -1;
    }

    return 0;
}


/****************************************************************************
 *                  Functions to help parse lines in the config             *
 ****************************************************************************/

/*
 * Function: skipspace
 * Parameter: pos - string to search through
 * Returns: pointer to first non-blank character in pos
 */
static char *skipspace(char *pos)
{
    while (isspace(*pos)) {
  	pos++;
    }
    return(pos);
}


/*
 * Function: nextspace
 * Parameter: ptr - string to search through
 * Returns: pointer to first blank character in ptr, or the end of the
 *          string if there are no blank characters in the string.
 */
static char *nextspace(char *ptr)
{
    while (*ptr && !isspace(*ptr)) {
	ptr++;
    }
    return ptr;
}


/*
 * Function: reset_devices
 *
 * Called to tidy up after smx has terminated
 */
void reset_devices(void)
{
    reset_sun_tty();
    remove_smx_ttys();
}


/****************************************************************************
 *                Saving and restoring of tty state                         *
 ****************************************************************************/

static struct termios old_set;         /* Save area for settings */


/*
 * Function: set_sun_tty
 *
 * Setup the controlling terminal of the SunOS process to be the smx process.
 * This involves setting up the TERMINAL_FD and TERMINAL_FD + 1 descriptors,
 * saving the current terminal state, setting the terminal into a very "raw"
 * mode, and sets up all 4 terminal "lines" for non-blocking IO.
 *
 * Terminals must be non-blocking and asynchronous (causes SIGIO) with no 
 * echo, flow control or special control characters or signals.
 */
static void set_sun_tty(void)
{
    struct termios t;
    int fd;

    fd = open("/dev/tty", O_RDWR);
    if( (dup2(fd, TERMINAL_FD) < 0) ||  (dup2(fd, TERMINAL_FD + 1) < 0)) {
	fprintf(stderr, "Couldn't get main terminal\n");
	exit(1);
    }
    close(fd);

    ioctl(TERMINAL_FD, TCGETS, &old_set);
    t = old_set;
    t.c_lflag &= ~(ECHO | ICANON | ISIG);
    t.c_iflag &= ~(IXON | IXOFF);
    t.c_cc[VMIN] = 1;
    t.c_cc[VTIME] = 0;
    ioctl(TERMINAL_FD, TCSETS, &t);
    
    for (fd = 0; fd < 2 * NR_CONS; fd++) {
	fcntl(TERMINAL_FD + fd, F_SETFL, O_NDELAY | O_RDWR);
    }
}


/*
 * Function: reset_sun_tty
 *
 * Restore saved tty settings
 */
static void reset_sun_tty(void)
{
    ioctl(TERMINAL_FD, TCSETS, &old_set);
}


/*
 * Function: enable_interrupts
 *
 * Setup all Minix terminals and the ethernet interface to send SIGIO when
 * input becomes available.
 */
void enable_interrupts(void)
{
    int fd;

    ioctl(ETHER_FD, I_SETSIG, S_RDNORM);      
    for (fd = 0 ; fd < 2 * NR_CONS ; fd += 2) {
	ioctl(TERMINAL_FD + fd, I_SETSIG, S_RDNORM);      
    }
}


/****************************************************************************
 *   Functions to create and delete the FIFOs used for extra smx terminals  *
 ****************************************************************************/

/*
 * Function: setup_smx_ttys
 *
 * A directory needs to be setup.  For each terminal two FIFOs are
 * setup, one for tty input and one for tty output.  These FIFOs are
 * opened on appropriate fds.  One "lock" file is also created.  It contains
 * two integers---one giving the number of additional ttys (over and above
 * the console), and a bit map showing which terminals are currently free.
 * mlogin uses the lock file to record the ttys currently in use.
 */
static void setup_smx_ttys(void)
{
    int fd;			    /* general purpose file descriptor */
    char buffer[MAXPATHLEN];	    /* a config file line */
    int line;

    remove_smx_ttys();              /* get rid of anything that's there */
    if (mkdir(host_dir, 0755) == -1) {
        fprintf(stderr, "Can't create named pipe directory\n");
	exit(1);
    }

    /*
     * Create the input and output FIFOs, and open them on the relevant
     * descriptors.
     */
    for (line = 1; line < NR_CONS; line++){
        sprintf(buffer, "%s/in.%d", host_dir, line);
	 
	if (mknod(buffer, 0666 | S_IFIFO, 0) == -1){
	    fprintf(stderr, "Can't create input FIFO %s\n", buffer);
	    exit(1);
	}
	    
	fd = open(buffer, O_RDWR );
	if (dup2(fd, TERMINAL_FD + line * 2) < 0) {
	    fprintf(stderr, "Couldn't dup input pipe\n");
	    exit(1);
	}
	close(fd);

	sprintf(buffer, "%s/out.%d", host_dir, line);
	if (mknod(buffer, 0666 | S_IFIFO, 0) == -1){
	    fprintf(stderr, "Can't create output FIFO %s\n", buffer);
	    exit(1);
	}
	fd = open(buffer, O_RDWR );
	if (dup2(fd, TERMINAL_FD + line * 2 + 1) < 0) {
	    fprintf(stderr, "Couldn't dup output pipe\n");
	    exit(1);
	}
        close(fd);
	 
    }

    /*
     * Create and initialise lock file.
     */
    sprintf(buffer, "%s/lock", host_dir);
    if ((fd = open(buffer, O_CREAT | O_RDWR, 0644 )) == -1){
	fprintf(stderr, "Couldn't create lock file\n");
	exit(1);
    }
    line = NR_CONS - 1;   /* One is taken up by the system console */
    write(fd, &line, sizeof(line));
    line = ~(0xffffffff << (NR_CONS - 1));       /* NR_CONS - 1 "1" bits */
    write(fd, &line, sizeof(line));
    close(fd);
}


/*
 * Function remove_smx_ttys
 *
 * Remove the entire directory created to hold the FIFOs etc for remote
 * login.
 */
static void remove_smx_ttys(void)
{
    char command[MAXPATHLEN];

    if (host_dir[0] != '\0') {
	sprintf(command, "rm -rf %s", host_dir);
	system(command);
    }
}
