static char rcsid[] = "$Id: elf2smx.c,v 1.1 1996/06/04 08:29:40 paul Exp $";

/*
 * elf2smx
 *
 * This Solaris utility program converts a Solaris object file (in ELF 
 * format) into an smx object file.  Reading through the details of
 * the elf file, creating a header for the smx file, then writing
 * text and data segments for the smx file.  The text and read-only
 * data segments from the elf file are combined into a single text segment
 * in the smx file.
 */

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <unistd.h>
#include <assert.h>
#include <limits.h>

#include <libelf.h>

#include "../../include/minix/config.h"
#undef NULL
#include "../../include/minix/const.h"
#include "../../include/ansi.h"
#include "../../include/a.out.h"


/*
 * The section types that can be handled---indices into the secn_data array,
 * which is used to hold segment details read from the elf file.
 */

#define TEXT 0
#define DATA 1
#define RODATA 2
#define BSS 3
#define NUM_AREAS (BSS + 1)

struct secn_data {
    int        s_secnum;
    Elf32_Addr s_addr;
    Elf32_Word s_size;
} secn_data[NUM_AREAS];


static int debug = 0;

static char *progname;

/*
 * Local functions
 */
static int elf2smx(char *input, char *output, int stacksize);
static int do_scn(Elf32_Shdr *shdr, char *name, int secnum);
static int write_file(int outfile, Elf *prog, long entry, long stack);
static int advance(int outfile, long how_much);
static int write_secn(int outfile, Elf *prog, int sec_ind);
static void usage(void);


/*
 * Function: main
 * Parameters: argc, argv - see usage message for command line arguments
 *
 * Validates the command line arguments, then calls elf2smx to do the
 * conversion.
 */
int main(int argc, char *argv[])
{
    int stacksize = -1;
    int option;
    char *end, *p;
    int num;

    progname = argv[0];

    while ((option = getopt(argc, argv, "S:d")) != -1) {
        switch(option) {
	case 'S':
	    /*
	     * The following piece of code was taken from the minix 
	     * install command where it processes a specified stack size.
	     */
	    p = optarg;
	    stacksize = strtol(p, &end, 0);
	    if (end == p || stacksize < 0) usage();
	    p = end;
	    while (*p != 0) {
	        switch (*p++) {
		case 'm':
		case 'M': num = 1024 * 1024; break;
		case 'k':
		case 'K': num = 1024; break;
		case 'w':
		case 'W': num = 4; break;
		case 'b':
		case 'B': num = 1; break;
		default: usage();
		}
		if (stacksize > LONG_MAX / num) usage();
		stacksize *= num;
	    }
	    break;

	case 'd':
	    debug = 1;
	    break;

        default:
	    /*
	     * unknown argument.
	     */
	    usage();
	    break;
        }
    }

    argc -= optind;       /* set to the number of args remaining */
    argv += optind;       /* step over already processed args */
    if (argc != 2) {
        /*
	 * Whoops - need exactly two files to process
	 */
        usage();
    }

    if (elf2smx(argv[0], argv[1], stacksize) == -1) {
	fprintf(stderr, "%s: conversion from %s to %s failed\n", progname,
		argv[0], argv[1]);
	unlink(argv[1]);
    }
    return 0;
}


/*
 * Function: elf2smx
 * Parameters: input - elf file to convert
 *             output - name to give the smx executable
 *             stacksize - used to calculate how the total address space
 *                         needs of the smx executable.
 * Returns: 0 on success; -1 on failure
 *
 * Open the input and output files.  Scan through the elf file, recording
 * details of the segments of interest, the generate the smx file from
 * those segments.
 */
static int elf2smx(char *input, char *output, int stacksize)
{
    int fd;
    int outfile;
    int prog_cnt = 0;
    int secnum;

    Elf *prog, *file;
    Elf32_Ehdr *ehdr;
    Elf32_Shdr *shdr;
    Elf_Scn *scn;
    Elf_Cmd cmd;

    /*
     * open the Solaris object file
     */
    if ((fd = open(input, O_RDONLY)) == -1) {
        fprintf(stderr, "Can't open ELF file %s\n", input);
	return -1;
    }
    if (elf_version(EV_CURRENT) == EV_NONE) {
        return -1;
    }    
  
    /*
     * Create the SunOS output file.  Delete the current file under that
     * name (if any).
     */
    (void) unlink(output);
    outfile = open(output, O_RDWR | O_CREAT, 0666);
    if (outfile == -1) {
        fprintf(stderr, "Can't open %s\n", output);
	return -1;
    }

    /*
     * Read and get details about each program (should be only 1)
     */
    cmd = ELF_C_READ;
    if ((file = elf_begin(fd, cmd, (Elf *)0)) == 0) {
	return -1;
    }
    while ((prog = elf_begin(fd, cmd, file)) != 0) {
	if (++prog_cnt > 1) {
	    fprintf(stderr, 
    "%s: second program found in ELF file; aborting after converting first\n",
		    progname);
	    break;
	}

	if ((ehdr = elf32_getehdr(prog)) == 0) {
	    return -1;
	}

	/*
	 * Read and get details about each section
	 */
	scn = (Elf_Scn *)0;
	secnum = 0;
	while ((scn = elf_nextscn(prog, scn)) != 0) {
	    secnum++;
	    shdr = elf32_getshdr(scn);
	    if (do_scn(shdr, elf_strptr(file, ehdr->e_shstrndx, 
					(size_t)shdr->sh_name), secnum)== -1) {
		return -1;
	    }
	}
	
	/*
	 * Write the SunOS object file
	 */
	if (write_file(outfile, prog, ehdr->e_entry, stacksize)== -1) {
	    return -1;
	}

	cmd = elf_next(prog);
	elf_end(prog);
    }

    elf_end(file);
    close(outfile);
    return prog_cnt == 1 ? 0 : -1;
}


/*
 * Function: do_scn
 * Parameters: shdr - elf section header struct
 *             name - elf section name
 *             secnum - section number
 * Returns: -1 on error, 0 otherwise
 *
 * If the ELF section is one of the interesting ones, then record its
 * details in the secn_data array.
 */
static int do_scn(Elf32_Shdr *shdr, char *name, int secnum)
{
    int arr_ind;

    if (debug) printf("'%s'\n", name? name: "(null)");
  
    /*
     * Set arr_ind to indicate the section we are dealing with.
     */
    if (!strcmp(".text", name)) {
        arr_ind = TEXT;
    } else if (!strcmp(".rodata", name)) {
        arr_ind = RODATA;
    } else if (!strcmp(".data", name)) {
        arr_ind = DATA;
    } else if (!strcmp(".bss", name)) {
        arr_ind = BSS;
    } else {
        arr_ind = -1;
    }

    if (arr_ind != -1) {
	/*
	 * If we have one of the sections we are interested in,
	 * record its details.  We only expect one segment of each type,
	 * so give an error message if we encounter a second.
	 */
        if (secn_data[arr_ind].s_secnum != 0) {
	    fprintf(stderr, "%s: second section of type %s encountered\n",
		    progname, name);
	    return -1;
	}
        secn_data[arr_ind].s_secnum = secnum;
        secn_data[arr_ind].s_addr = shdr->sh_addr;
        secn_data[arr_ind].s_size = shdr->sh_size;
        if (debug) {
	    printf("    Addr = 0x%x, sh_size = 0x%x\n\n",
		   (unsigned) secn_data[arr_ind].s_addr,
		   (unsigned) secn_data[arr_ind].s_size);
	}
    }
    return 0;
}


/*
 * Function: write_file
 * Parameters: outfile - descriptor onto which the output file is to be
 *                       written.
 *             prog - elf input file
 *             entry - virtual address of program entry point
 *             stack - gap to leave after bss for heap and stack
 * Returns: 0 on success, -1 on failure
 *
 * The smx header is initialised and written.  Then the smx text segment
 * is written (elf text + read-only data segments), then the data segment.
 * Both smx text and data segments musy be click-aligned, and the
 * smx text segment must be a multiple of the click size.
 */
static int write_file(int outfile, Elf *prog, long entry, long stack)
{
    struct exec header;
    long bytes_written;

    /*
     * Some sections may not have been present in the ELF file.  So long as
     * we have a text segment and a data segment we're in business.
     */
    if (secn_data[TEXT].s_secnum == 0 || secn_data[DATA].s_secnum == 0) {
	fprintf(stderr, "%s: Missing text or data segment in ELF file\n",
		progname);
	return -1;
    }
    /*
     * If there is no read-only data segment set the read-only data
     * segment address to the end of the text segment
     * so that the calculations below still work.
     */
    if (secn_data[RODATA].s_secnum == 0) {
	secn_data[RODATA].s_addr = secn_data[TEXT].s_addr +
	    secn_data[TEXT].s_size;
    }

    /* Create the header */
    header.a_magic[0] = A_MAGIC0;
    header.a_magic[1] = A_MAGIC1;
    header.a_flags = A_EXEC | A_SEP;  /* Executable, separate instr and data */
    header.a_cpu = A_SUNOS;
    header.a_hdrlen = sizeof(header);
    header.a_unused = 0;
    header.a_version = 0;
    header.a_tbase = downclick(secn_data[TEXT].s_addr);
    header.a_text = upclick(secn_data[RODATA].s_addr + 
			    secn_data[RODATA].s_size) - header.a_tbase;
    header.a_dbase = downclick(secn_data[DATA].s_addr);
    header.a_data = secn_data[DATA].s_addr + secn_data[DATA].s_size -
      header.a_dbase;
    header.a_bss = secn_data[BSS].s_size;
    header.a_entry = entry;
    header.a_syms = 0;
    header.a_trsize = header.a_drsize = 0;
    if (stack >= 0) {
	/*
	 * Use the user specified "gap".
	 */
	header.a_total = header.a_data + header.a_bss + stack;
    } else {
	/*
	 * No gap specified.  Use 4 clicks or 20% of the text segment
	 * size, whichever is greater.
	 */
	header.a_total = header.a_data + header.a_bss + 
   	  (((header.a_text / 5) < CLICK_SIZE * 4) ?  CLICK_SIZE * 4 : 
          (header.a_text / 5));
    }
    write(outfile, &header, sizeof(header));
  
    /*
     * Write the text segment.  If the elf text segment is not click aligned,
     * then we have to leave some space before the elf text segment is
     * written.  There may be space between the text and read-only data
     * and space after the read-only data to get the smx text segment
     * click aligned.
     */
    bytes_written = advance(outfile, secn_data[TEXT].s_addr - header.a_tbase);
    bytes_written += write_secn(outfile, prog, TEXT);
    bytes_written += advance(outfile, secn_data[RODATA].s_addr - 
			     header.a_tbase - bytes_written);
    bytes_written += write_secn(outfile, prog, RODATA);
    (void) advance(outfile, header.a_text - bytes_written);

    /*
     * Write the data segment, again leaving space if the data segment
     * does not start on a click boundary,
     */
    (void) advance(outfile, secn_data[DATA].s_addr - header.a_dbase);
    (void) write_secn(outfile, prog, DATA);
    return 0;
}    


/*
 * Function: advance
 * Parameters: outfile - file to advance in
 *             how_much - the number of bytes to advance
 * Returns: how_much
 */
static int advance(int outfile, long how_much)
{
    if (how_much > 0) {
        if (lseek(outfile, how_much, SEEK_CUR) == -1) {
	    fprintf(stderr, "%s: lseek failed\n", progname);
	    exit(1);
	}
	return how_much;
    }
    return 0;
}


/*
 * Function: write_secn
 * Parameters: outfile - file to write to
 *             prog - elf file to copy the segment from
 *             sec_ind - section to write
 * Returns: the number of bytes written
 *
 * Write section "sec_ind" from elf file "prog" to "out_file".
 */
static int write_secn(int outfile, Elf *prog, int sec_ind)
{
    Elf_Scn *scn;
    Elf_Data *data;

    if (secn_data[sec_ind].s_secnum == 0) {
	return 0;        /* section is empty */
    }

    scn = elf_getscn(prog, secn_data[sec_ind].s_secnum);
    data = elf_getdata(scn, 0);
    if (data == 0) {
	fprintf(stderr, "%s: error getting elf section data\n", progname);
	exit(1);
    }
    assert(data->d_size == secn_data[sec_ind].s_size);
    if (write(outfile, data->d_buf, data->d_size) != data->d_size) {
	fprintf(stderr, "%s: error writing segment\n", progname);
	exit(1);
    }
	
    return data->d_size;
}
	   

/*
 * Function: usage
 */
static void usage(void)
{
    fprintf(stderr,
	    "Usage: %s [-d] [-S stack_size] solaris-filename sunos-filename\n",
	    progname);
    exit(1);
}



