static char *rcsid = "$Id: minix.c,v 1.4 1996/07/09 04:47:04 paul Exp $";

/*
 * This program is responsible for booting Solaris Minix. The
 * code in config.c is resposible for reading the configuration file
 * and setting up all smx devices.  The code in minix_load.c is
 * responsible for loading Minix executable programs into smx
 * "physical" memory.  The code in this file coordinates the whole process.
 *
 * Two processes are involved in running smx.  The parent process (the one
 * that starts by running main) forks off the child, and it is the
 * child that actually becomes smx.  When the child exits, the parent
 * is there to cleanup.
 *
 * The bootstrapping process involves reading the configuration file,
 * establishing all smx devices, allocating the "physical" memory for smx,
 * loading four programs from the Minix image file (kernel, mm, fs, init)
 * into this memory, then switching execution into smx.
 */
					     
#include <stdio.h>
#include <stdlib.h>
#include <malloc.h>
#include <signal.h>
#include <unistd.h>
#include <fcntl.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <sys/types.h>

#include "../../include/minix/config.h"
#undef HZ                       /* Defined again in minix/const.h */
#undef NULL                     /* Defined again in minix/const.h */
#include "../../include/minix/const.h"
#include "../kernel/const.h"   /* Need RAM_FD */
typedef unsigned char u8_t;
typedef unsigned short u16_t, U16_t;
#include "../../include/net/gen/ether.h"
#include "../kernel/bootinfo.h"

#undef PRIVATE                    /* Defined in <sys/mman.h? */
#undef printf

#include <sys/mman.h>


#include "config.h"
#include "minix_load.h"

#define DEF_MEM_SIZE	3072	/* default number of kbytes of smx memory */
#define DEF_PROTECT	1	/* default protection level */

#define USAGE "%s [-d] [-m (none|half|full)] [config-file]\n"


char *progname;
static int debug = 0;      /* Used in this program, and propagated to smx */

/*
 * Local functions.
 */
static struct smx_bootinfo *load_image(const char *imagefile, long core_size);
static void allocate_memory(unsigned long memory_base, long ram_size);

static void smx_parent(void);

static void usage(void);


/*
 * Function: main
 * Parameters: argc, argv - command line argument summary appears in the USAGE
 *                          string
 *
 * The main function begins by processing the command line arguments.
 * It then calls config to have the configuration file read and
 * the smx devices opened.  It then calls fork, with the parent
 * waiting to clean up after smx and the child loading the four initial
 * programs and jumping inside smx.
 */
int main(int argc, char *argv[])
{
    char image_name[MAXPATHLEN];
    int protect, configdebug, configprotect;
    long core_size;                     /* how much memory is allocated */
    char *config_file;
    struct smx_bootinfo *bootp;		/* this will actually be part of the */
					/* kernel's data segment */
    ether_addr_t eaddr = { {0} };       /* all 0s means disabled */

    progname = argv[0];

    /*
     * Check that the click size is a multiple of the pagesize - otherwise
     * protection of segments using mprotect(2) will fail.
     */
    if (CLICK_SIZE % sysconf(_SC_PAGESIZE) != 0) {
	(void) fprintf(stderr,
    "%s: Minix aborted - clicksize (%d) is not a multiple of pagesize (%ld)\n",
		       progname, CLICK_SIZE, sysconf(_SC_PAGESIZE));
	exit(1);
    }

    /*
     * Process options. Only the debug (-d) and the three memory
     * protection levels (-m none, -m half, -m full) are accepted
     */
    config_file = 0;       /* Initialise things that may be found .. */
    debug = 0;             /* on the command line */
    protect = -1;
    {
        int option;
        extern char *optarg;
        extern int optind;
                              
        while ((option = getopt(argc, argv, "dm:")) != -1) {
	    switch(option) {
            case 'd':
		debug = 1;
		break;

            case 'm':
		protect = prot_atoi(optarg);
		break;

            default:
		/*
		 * unknown argument.
		 */
		usage();
		break;
	    }
	}
        argc -= optind;       /* set to the number of args remaining */
        argv += optind;       /* step over already processed args */
    }

    if (argc > 1) {
	usage();
    }
    if (argc == 1) {
	config_file = *argv;
    }
  

    /*
     * Open all disk files etc. and get the desired memory size, image file
     * name, protection level and debug mode.  Three things that might be
     * specified in the config file are memory size, debug enable and
     * protection.  Variables are set to defaults before the call
     * to config, and if the relevant option is not specified in the config
     * file, the value ofthe variable is not changed.
     */
    core_size = DEF_MEM_SIZE * 1024;
    configprotect = DEF_PROTECT;
    configdebug = 0;
    config(config_file, &core_size, image_name, &configprotect, &configdebug,
	   &eaddr);
    /*
     * Round up core size to the next click
     */
    core_size = upclick(core_size);
  
    /*
     * Config file protection and debug options are only used if the relevant
     * command line option was not given.
     * file options.
     */
    if (!debug) {
        debug = configdebug;
    }
    if (protect == -1) {
        protect = configprotect;
    }

    /*
     * Create the process to run smx.
     */
    switch (fork()) {
    case -1:      /* error */
	reset_devices();
	fprintf(stderr, "%s: Unable to fork secondary Minix process\n",
		progname);
	exit(1);
    
    case 0:       /* child */
	break;
    
    default:      /* parent */
	smx_parent();
	/*NOTREACHED*/
    } /* fork() switch */

    bootp = load_image(image_name, core_size);
    bootp->debug = debug;
    bootp->prot_lev = protect;
    bootp->mem_bytes = core_size;
    bootp->ether_addr = eaddr;
  
    enable_interrupts();

    /* start the kernel running */
    ((void (*)()) bootp->prog[KERNEL_PROG].entry)();
    /*NOTREACHED*/
    exit(2);
}


/*
 * Function: load_image
 * Parameters: imagefile - file to load the 4 programs from
 *             core_size - number of bytes of smx "physical" memory to allocate
 * Returns: pointer to the smx_bootinfo struct which is at the start of the
 *          kernel's data segment.
 *
 * Details of the kernel's text segment are retrieved so that we know where
 * to position the smx memory, and where the kernel data segment starts
 * so that we can determine where the smx_bootinfo struct is.  The memory
 * is then allocated, and the four programs from the image loaded into it,
 * with program details stored in the bootinfo struct.
 */
static struct smx_bootinfo *load_image(const char *imagefile, long core_size)
{
    int fd, prog;
    struct smx_bootinfo *bootp;
    unsigned long memory_base;
    char *base;

    /*
     * Open the image file.
     */
    if ((fd = open(imagefile, O_RDONLY)) < 0){
  	fprintf(stderr, "%s: Unable to open image file\n", progname);
	exit(1);
    }
  
    /*
     * Find out where the kernel text segment starts (that's where we need
     * to map smx memory) and where the kernel data segment starts
     * (that's where the boot parameters go)
     */
    if (get_seg_details(fd, &memory_base, (unsigned long *)&bootp) < 0) {
  	fprintf(stderr, "%s: Failed to get address of kernel text segment\n",
		progname);
	exit(1);
    }
    if (debug) {
	fprintf(stderr, "bootp = 0x%x\n", (unsigned) bootp);
    }

    allocate_memory(memory_base, core_size);
    if (debug) {
	(void) fprintf(stderr, "Core size = 0x%x\nMemory base = 0x%lx\n",
		       (unsigned) core_size, memory_base);
    }  

    /*
     * For each program within the image file, load it, and
     * record it's details.  kernel, mm and fs must be loaded
     * into the expected virtual addresses, so this is checked
     * by minix_load for those programs.
     */
    base = (char *) memory_base;
    for (prog = 0; prog < NUM_PROGS; prog++)
    {
	if (minix_load(fd, base, &bootp->prog[prog], prog != INIT_PROG) < 0) {
	    fprintf(stderr, "%s: Unable to load program %d\n", progname,
		    prog);
	    fprintf(stderr,
      "\tRun make in src/tools to ensure that all ld map files are correct\n");
	    exit(1);
	}

	if (debug) {
	    (void) fprintf(stderr,
		         "Program %d, base 0x%x, t_size 0x%lx, d_size 0x%lx\n",
			   prog, (unsigned) base,
			   bootp->prog[prog].text_clicks << CLICK_SHIFT, 
			   bootp->prog[prog].data_clicks << CLICK_SHIFT);
	}
	base += (bootp->prog[prog].text_clicks +
		 bootp->prog[prog].data_clicks) << CLICK_SHIFT;
    }

    return bootp;    /* main() has more stuff to record as boot parameters */
}


/*
 * Function: allocate_memory
 * Parameters: memory_base - address to map memory in at
 *             ram_size - number of bytes to map
 *
 * We need to map a file in to act as smx memory because user programs
 * need to be mapped into low memory.  A temporary file is created,
 * extended to the appropriate size, and mapped into memory.  Files
 * created by tmpfile are automatically deleted when the creating process
 * exits.  The smx kernel expects the file to be open on descriptor
 * RAM_FD so that it can call mmap on it to run user programs.
 */
static void allocate_memory(unsigned long memory_base, long ram_size)
{
    char *ret_val;
    FILE *mem_file = tmpfile();

    if (mem_file == 0) {
  	fprintf(stderr, "%s: Unable to open RAM file\n", progname);
        exit(1);
    }
    if (dup2(fileno(mem_file), RAM_FD) < 0) {
        fprintf(stderr, "%s: Unable to dup RAM file fd\n", progname);
        exit(1);
    }
    fclose(mem_file);

    if (ftruncate(RAM_FD, ram_size) < 0) {
        fprintf(stderr, "%s: Unable to size RAM\n", progname);
        exit(1);
    }
    ret_val = mmap((char *)memory_base, ram_size,
		   PROT_READ | PROT_WRITE | PROT_EXEC,
		   MAP_SHARED | MAP_NORESERVE | MAP_FIXED, RAM_FD, 0);
    if (ret_val == MAP_FAILED) {
        fprintf(stderr, "%s: Unable to map RAM file into memory\n", progname);
        exit(1);
    }
}


/*
 * Function: smx_parent
 *
 * Code executed by the parent process after the fork.  It blocks all
 * but error signals while waiting for the child running smx to finish.
 * Then it cleans up the smx devices and terminal settings and exits.
 */
static void smx_parent(void)
{
    sigset_t sigs;
    int status;

    sigfillset(&sigs);
    sigdelset(&sigs, SIGILL);
    sigdelset(&sigs, SIGTRAP);
    sigdelset(&sigs, SIGEMT);
    sigdelset(&sigs, SIGFPE);
    sigdelset(&sigs, SIGBUS);
    sigdelset(&sigs, SIGSEGV);
    sigdelset(&sigs, SIGSYS);
    (void) sigprocmask(SIG_SETMASK, &sigs, NULL);

    wait(&status);
    
    /*
     * Tidy up smx device files and settings.
     */
    reset_devices();
    exit(0);
}


/*
 * Function: usage
 */
static void usage(void)
{
    fprintf(stderr, USAGE, progname);
    exit(1);
}

