static char *rcsid = "$Id: minix_load.c,v 1.2 1996/07/09 04:47:06 paul Exp $";

/*
 * This contains functions that deal with smx executables.  It contains two
 * functions, both entry points:
 *    - minix_load - load an smx executable into memory and return various
 *                   pieces of header information.
 *    - get_seg_details - return the virtual addresses of the text and data
 *                        segments of an smx executable.
 */

#include <stdio.h>
#include <string.h>
#include <unistd.h>

#include "../../include/minix/config.h"
#undef NULL /* Defined again in const.h */
#include "../../include/minix/const.h"
#include "../../include/ansi.h"
#include "../../include/a.out.h"
typedef unsigned char u8_t;
typedef unsigned short u16_t, U16_t;
#include "../../include/net/gen/ether.h"
#include "../kernel/bootinfo.h"

#include "minix_load.h"


/*
 * Function: minix_load
 * Parameters: fd - file to load the smx executable from.  Must be currently
 *                  positioned at the start of the header.
 *             base - where to load it in memory
 *             smp - struct to be filled out with header info.
 *             check_vaddrs - if true, check that the addresses the text
 *                            and data segment are loaded into match those
 *                            in the executable header.
 * Returns: -1 on failure, 0 on success
 *
 * Load an smx executable into memory.
 */
int minix_load(int fd, char *base, struct smx_proginfo *smp, int check_vaddrs)
{
    struct exec mxhead;

    if (read(fd, &mxhead, sizeof(mxhead)) != sizeof(mxhead)) {
	return -1;
    }

    /*
     * Validate the header.  The check on a_total differs from that
     * in mm/exec.c as the kernel has a gap of 0.
     */
    if (BADMAG(mxhead) || mxhead.a_hdrlen != sizeof(mxhead) ||
	mxhead.a_cpu != A_SUNOS || (mxhead.a_flags & A_SEP) == 0 ||
	mxhead.a_total == 0 || mxhead.a_text % CLICK_SIZE != 0 ||
	mxhead.a_total < mxhead.a_data + mxhead.a_bss) {
	return -1;
    }

    /*
     * Read in the text segment
     */
    if (check_vaddrs && (long) base != mxhead.a_tbase) {
	return -1;
    }
    if (read(fd, base, mxhead.a_text) != mxhead.a_text) {
	return -1;
    }

    /*
     * Read in the data segment, and zero bss, stack and the gap between.
     */
    base += mxhead.a_text;
    if (check_vaddrs && (long) base != mxhead.a_dbase) {
	return -1;
    }
    if (read(fd, base, mxhead.a_data) != mxhead.a_data) {
	return -1;
    }
    memset(base + mxhead.a_data, 0, upclick(mxhead.a_total) - mxhead.a_data);

    /*
     * Fill out the program details.  This is done after reading the data
     * segment as the struct being filled out is actually in the data
     * segment of the kernel---if the data segment is read after this
     * then all of the values in *smp will be overwritten.
     */
    smp->text_clicks = mxhead.a_text >> CLICK_SHIFT;
    smp->data_clicks = upclick(mxhead.a_total) >> CLICK_SHIFT;
    smp->text_vaddr = mxhead.a_tbase;
    smp->data_vaddr = mxhead.a_dbase;
    smp->entry = mxhead.a_entry;
    return 0;
}


/*
 * Function: get_seg_details
 * Parameters: fd - file to read the smx executable header.  Must be currently
 *                  positioned at the start of the header.
 *             tvaddr - where the virtual address of the text segment will be
 *                      returned.
 *             dvaddr - where the virtual address of the data segment will be
 *                      returned.
 * Returns: -1 on failure, 0 on success.
 *
 * Extracts the virtual addresses of the text and data segments from an
 * smx executable.  The file containing the smx executable is left positioned
 * at the start of the header.
 */
int get_seg_details(int fd, unsigned long *tvaddr, unsigned long *dvaddr)
{
    struct exec mxhead;

    /*
     * Read header then rewind the file to where it was before the read.
     */
    if (read(fd, &mxhead, sizeof(mxhead)) != sizeof(mxhead)) {
	return -1;
    }
    if (lseek(fd, -sizeof(mxhead), SEEK_CUR) == -1) {
	return -1;
    }
    *tvaddr = mxhead.a_tbase;
    *dvaddr = mxhead.a_dbase;
    return 0;

}


