static char rcsid[] = "$Id: mlogin.c,v 1.2 1996/06/04 08:29:53 paul Exp $";

/* 
 * mlogin
 *
 * This is a Solaris utility for connecting with a running smx.  It
 * is similar to 'rlogin'.  A "hostname" is given as the single command
 * line argument.  The directory /tmp/"hostname" contains a file called
 * "lock" that gives details of the number of consoles, and their
 * availability.  Also in the directory are one pair of FIFOs per console.
 * mlogin finds a "line" to use and opens the appropriate FIFOs.  It then goes
 * into a loop relaying characters between smx (i.e. the FIFIOs) and the
 * controlling terminal.  When the connection is severed (by either party)
 * the lock file is updated to reflect that the line is now available,
 */

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <stropts.h>

/*
 * Need to get MLOGIN_HANGUP from ../kernel/const.h (need config.h first)
 */ 
#include "../../include/minix/config.h"
#include "../kernel/const.h"
#undef printf   /* defined to printk in kernel/const.h */

#define BUFSIZE 200		    /* most characters read at one time */

static int termfd;		    /* /dev/tty fd */
static struct termios old_t;	    /* save area for terminal configuration */
static int ourline;                 /* number of the tty we are using */

static char lock_file[MAXPATHLEN];  /* name of the lock file */


/*
 * Local functions
 */
static void term(int pipeinfd, int pipeoutfd);
static int read_lock_file(int *nr_cons, int *cons_map);
static int write_lock_file(int fd, int cons_map);
static void exit_mlogin(const char *mess, int status);


/*
 * Function: main
 * Parameters: argc, argv - see usage
 *
 * We find a terminal line, open the FIFOs for that line, setup the
 * controlling tty and signal settings, then call term to relay data
 * to and from smx itself.
 */
int main(int argc, char *argv[])
{
    int pipeinfd;			/* the smx connection */
    int pipeoutfd;
    struct termios t;		        /* terminal setup */
    char host_name[MAXPATHLEN];	        /* host we are connecting to */
    int lockfd, cons_map, nr_cons;
    int count;
  
    /*
     * find the directory relating to the desired host name
     */
    if (argc != 2) exit_mlogin("Usage: mlogin <host name>\n", 1);
    
    /*
     * Get the map of free terminal lines and allocate ourselves one
     */
    sprintf(lock_file, "/tmp/%s/lock", argv[1]);
    if ((lockfd = read_lock_file(&nr_cons, &cons_map)) == -1) {
	exit_mlogin("Can't read lock file\n", 1);
    }
    ourline = 0;
    for (count = 0; count != nr_cons; count++) {
	if ((cons_map & (1 << count)) != 0) { 	
	    ourline = count + 1;        /* found a free terminal */
	    break;
	}
    }
    
    if (ourline == 0) exit_mlogin("Out of terminal lines\n", 1);

    cons_map &= ~(1 << count);          /* mark it as used */
    if (write_lock_file(lockfd, cons_map) == -1) {
	exit_mlogin("Couldn't write to lock file\n", 1);
    }

    /*
     * Open the two FIFOs, and set them to non-blocking IO.
     */
    sprintf(host_name, "/tmp/%s/out.%d", argv[1], ourline);
    if ((pipeinfd = open(host_name, O_RDONLY)) == -1 ||
	fcntl(pipeinfd, F_SETFL, O_NDELAY | O_RDWR) == -1) {
	exit_mlogin("Can't setup from smx FIFO\n", 1);
    }
    sprintf(host_name, "/tmp/%s/in.%d", argv[1], ourline);
    if ((pipeoutfd = open(host_name, O_RDWR)) == -1 ||
	fcntl(pipeoutfd, F_SETFL, O_NDELAY | O_RDWR) == -1) {
	exit_mlogin("Can't setup to smx FIFO\n", 0);
    }

    /*
     * open the user's tty, and save its settings
     */
    termfd = open("/dev/tty", O_RDWR);
    if (termfd < 0) {
	exit_mlogin("Can't open /dev/tty\n", 1);
    }
    ioctl(termfd, TCGETS, &old_t);
  
    /*
     * setup the user terminal mode (no echo, no signals, non-blocking etc) 
     */
    t = old_t;
    t.c_lflag &= ~(ECHO | ICANON | ISIG );
    t.c_iflag &= ~(IXON | IXOFF);
    t.c_cc[VMIN] = 1;
    t.c_cc[VTIME] = 0;
    ioctl(termfd, TCSETS, &t);
    fcntl(termfd, F_SETFL, O_NDELAY | O_RDWR);
  
    /*
     * ignore all signals 
     */
    for (count = 1; count < _SIGRTMAX; count++) {
	signal(count, SIG_IGN);
    }

    printf("Connected to host %s\n", argv[1]);

    term(pipeinfd, pipeoutfd);        /* Transfer the data */
    
    /*NOTREACHED*/
    return 0;
}


/*
 * Function: term
 * Parameters: pipeinfd - data from smx received on this descriptor
 *             pipeoutfd - data to smx sent on this descriptor
 * Returns: never
 *
 * Loop forever, transfering data to and from smx.  If the user types
 * ^Q^U^I^T then we tear down the connection and exit.
 */
static void term(int pipeinfd, int pipeoutfd)
{
    char buffer[BUFSIZE];	    /* where the characters are stored */
    char quit = (char) MLOGIN_HANGUP; /* Minix's hangup character */
    int length;			    /* how much was read? */
    fd_set reads;
    int max_fd;
    int cnt;

    /*
     * terminal emulator escape sequence ^Q^U^I^T
     */
    char disco[] = {021, 025, 011, 024};    
    int discnt = 0;			/* how far through this sequence? */
  
    max_fd = termfd > pipeinfd ? termfd : pipeinfd;
    discnt = 0;

    /*
     * forever until death do us part 
     */
    while (1) {
	FD_ZERO(&reads);
	FD_SET(termfd, &reads);
	FD_SET(pipeinfd, &reads);
  
	/* 
	 * wait until there is some data to transfer in one or both directions
	 */
	select(max_fd + 1, &reads, 0, 0, 0);

	if (FD_ISSET(termfd, &reads)) {
	    /* 
	     * characters typed on the terminal 
	     */
	    length = read(termfd, buffer, BUFSIZE);
    
	    /* 
	     * part of escape sequence? 
	     */
	    for (cnt = 0; cnt < length; cnt++) {
		if (buffer[cnt] == disco[discnt]) {
		    discnt++;
		    if (discnt == sizeof(disco)) {
			break;     /* ^Q^U^I^T has been typed! */
		    }
		} else {
		    discnt = 0;    /* back to square one */
		}
	    }

	    /*
	     * If the read failed or ^Q^U^I^T has been typed then time to
	     * exit.  Tell smx by writing a quit_char to it.
	     */
	    if (length <= 0 || discnt == sizeof(disco)) {
		write(pipeoutfd, &quit, 1);
		exit_mlogin("\n\nExiting mlogin\n", 0);
	    }

	    /* 
	     * Pass characters on to Minix 
	     */
	    write(pipeoutfd, buffer, length);
	}
	
	if (FD_ISSET(pipeinfd, &reads)){
	    /* 
	     * characters from smx to display on the terminal
	     */
	    length = read(pipeinfd, buffer, BUFSIZE);
	    if (length <= 0) {
		write(pipeoutfd, &quit, 1);
		exit_mlogin("\n\nExiting mlogin\n", 0);
	    }
	    write(termfd, buffer, length);
	}
    } /* Infinite while loop */
}


/*
 * Function: read_lock_file
 * Parameters: nr_cons - used to return the number of consoles
 *             cons_map - bit map of free consoles
 * Returns: fd of open and locked "lock" file, -1 upon error
 *
 * Open and lock "lock" file and return its contents.  A call
 * to write_lock_file should be done soon after to unlock and
 * close the "lock" file.
 */
static int read_lock_file(int *nr_cons, int *cons_map)
{
    int fd;

    if ((fd = open(lock_file, O_RDWR)) == -1) {
	return -1;
    }

    if (lockf(fd, F_LOCK, 0) != 0) {
	return -1;
    }

    if (read(fd, nr_cons, sizeof(*nr_cons)) != sizeof(*nr_cons) ||
	read(fd, cons_map, sizeof(*cons_map)) != sizeof(*cons_map)) {
	return -1;
    }
    return fd;
}


/*
 * Function: write_lock_file
 * Parameters: fd - "lock" file is open and locked on this descriptor
 *             cons_map - map to write back to the log file.
 * Returns: 0 on success, -1 on error
 *
 * Update "lock" file with new cons map, then unlock and close it.
 */
static int write_lock_file(int fd, int cons_map)
{
    if (lseek(fd, sizeof(int), SEEK_SET) == -1 || 
	write(fd, &cons_map, sizeof(cons_map)) != sizeof(cons_map) ||
	lockf(fd, F_ULOCK, 0) == -1 ||
	close(fd) == -1) {
	return -1;
    }
    return 0;
}


/*
 * Function: exit_mlogin
 * Parameters: mess - a parting message
 *             status - the status to call exit(2) with.
 * Returns: doesn't
 *
 * mlogin is about to exit.  Print the exit message, mark the terminal
 * line as free, and restore the terminal to its former state.
 */
static void exit_mlogin(const char *mess, int status)
{
    int lockfd, cons_map, nr_cons;

    fprintf(stderr, mess);
  
    /*
     * Mark this terminal line as free
     */
    if ((lockfd = read_lock_file(&nr_cons, &cons_map)) != -1) {
	cons_map |= (1 << (ourline - 1));
	(void) write_lock_file(lockfd, cons_map);
    }

    /*
     * Reset the UNIX keyboard characteristics 
     */
    ioctl(termfd, TCSETS, &old_t);

    exit(status);
}
