static char *rcsid =  "$Id: relay.c,v 1.2 1996/08/01 02:21:08 paul Exp $";

/*
 * relay
 * =====
 *
 * The relay program serves as a packet switch and simulates an ethernet
 * segment.  When it starts up it opens a UDP/IP socket, and publishes its
 * addresses in a file (whose name is specified as the single command
 * line argument).  It then waits for incoming ethernet packets.  Each
 * of the two "ethernet addresses" in the ethernet header is actually
 * a UDP/IP address (both are 6 bytes!).  Incoming packets fall into the
 * following categories:
 *
 *    1. Destination address is our address.  A packet from a new smx instance
 *       registering itself (its address is the source address).
 *    2. A broadcast, that is forwarded to all registered parties.
 *    3. A packet sent to some other destination.  It is forwarded to that
 *       destination.
 *
 * It may seem inefficient to route all traffic through the relay (when
 * traffic of type 3 can potentially be sent direct from one smx instance
 * to another), but the fact that the SunOS library calls to enable direct
 * sending are not available means that everything is done via relay.
 *
 * XXX Currently there is no way for an smx instance to "de-register",
 * so if "relay" runs for a long time it may send up forwarding broadcasts
 * to a lot of smx instances, many of which are long deceased.
 */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <errno.h>
#include <assert.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <netdb.h>

#include "lib.h"

#define MIN_PKT_SIZE 12     /* packets must have at least dest and src addrs */

/*
 * Local functions.
 */
static void usage(void);

static int open_socket(u_short *port);
static int write_address_file(const char *file_name, u_short port);
static char *get_ourhostname(void);

static int relay_messages(int sock, u_short port);
static int is_our_machine(u_int host);
static unsigned int *our_ip_nums(void);
static void forward_mesg(int sock, u_int host, u_short port, char *buffer,
			 int buff_len);

static char *progname;


/*
 * Function: main
 * Parameters: argc, argv - the name of the address file must be the only
 *             command line argument.
 *
 * This function coordinates everthing by opening a socket, writing
 * its address to a file, and the calling relay_messages to perform
 * the relaying.
 */
int main(int argc, char *argv[])
{
    int sock;
    u_short port;

    progname = argv[0];

    if (argc != 2) {
	usage();
    }

    if ((sock = open_socket(&port)) == -1) {
	fprintf(stderr, "%s: failed to open socket\n", progname);
	exit(1);
    }

    if (write_address_file(argv[1], port) == -1) {
	fprintf(stderr, "%s: failed to write address file\n", progname);
	exit(1);
    }

    relay_messages(sock, port);

    return 0;
}


/*
 * Function: usage
 *
 * Prints a usage message then calls exit. 
 */
static void usage(void)
{
    fprintf(stderr, "Usage: %s socket_address_file\n", progname);
    exit(1);
}


/*
 * Function: open_socket
 * Parameter: port - used to return the port number of the UDP address
 *                   assigned to the socket.
 * Returns: the descriptor of the open socket on success, -1 on error
 *
 * Creates a UDP/IP socket, binds an address to it, and returns the port
 * number for that address.
 */
static int open_socket(u_short *port)
{
    struct sockaddr_in dsock;
    int addrlen;
    int ds;

    if ((ds = socket(PF_INET, SOCK_DGRAM, 0)) < 0) {
	return -1;
    }

    dsock.sin_family = AF_INET;
    dsock.sin_port = htons(0);
    dsock.sin_addr.s_addr = htonl(INADDR_ANY);
    if (bind(ds, (struct sockaddr *) &dsock, sizeof(dsock)) < 0) {
	return -1;
    }

    addrlen = sizeof(dsock);
    if (getsockname(ds, (struct sockaddr *) &dsock, &addrlen) == -1) {
	return -1;
    }

    *port = ntohs(dsock.sin_port);
    return ds;
}


/*
 * Function: write_address_file
 * Parameters: file_name - name of file to write to
 *             port - port number to write to file
 * Returns: 0 on success, -1 on error
 *
 * Writes the host name of the current machine and the specified
 * port number to the specified file.
 */
static int write_address_file(const char *file_name, u_short port)
{
    FILE *addr_file;

    addr_file = fopen(file_name, "w");
    if (addr_file == 0) {
	return -1;
    }
    fprintf(addr_file, "%s %d\n", get_ourhostname(), port);
    if (fflush(addr_file) == EOF || ferror(addr_file)) {
	return -1;
    }
    if (fclose(addr_file) == EOF) {
	return -1;
    }

    return 0;
}


/*
 * Function: get_ourhostname
 * Returns: a pointer to a static array containing the name of this
 *          (SunOS) machine.
 */
static char *get_ourhostname(void)
{
    static char our_name[MAXHOSTNAMELEN + 1];
    static int first_time = 1;
    int gethostname(char *name, int namelen);    /* not in any .h file */

    if (first_time) {
	if (gethostname(our_name, sizeof(our_name)) == -1) {
	    fprintf(stderr, "%s: gethostname error %d\n", progname, errno);
	    exit(1);
	}
	first_time = 0;
    }
    return our_name;
}


/*
 * Function: relay_messages
 * Parameters: sock - socket on which to receive messages
 *             port - UDP port number bound to sock
 * Returns: -1 on error, doesn't return otherwise
 *
 * Sits in an infinite loop receiving and processing messages.  Any messages
 * addressed to relay itself cause a new smx instance to be added to the
 * list.  Any broadcasts are forwarded to all registered smx instances.
 * All other messages are forwarded to their specified destinations.
 * The list of addresses of smx instances is held on the contacts array,
 * which is dynamically increased in size (REALLOC_CHUNK elements at a time)
 * as needed.
 */
#define REALLOC_CHUNK 20

static int relay_messages(int sock, u_short port)
{
    struct contact_list {
	u_int ip;
	u_short port;
    } *contacts;
    int array_size, num_contacts;

    u_int to_host, from_host;
    u_short to_port, from_port;
    char buffer[2000];
    int chars_read;

    array_size = REALLOC_CHUNK;
    num_contacts = 0;
    if ((contacts = malloc(array_size * sizeof(*contacts))) == 0) {
	return -1;
    }

    for (;;) {
	chars_read = read(sock, buffer, sizeof(buffer));
#ifdef DEBUG
        printf("Got message of %d chars\n", chars_read); fflush(stdout);
#endif
	if (chars_read <= 0) {
	    fprintf(stderr, "%s: %d returned by read on socket; exiting\n",
		    progname, errno);
	    exit(1);
	}
	if (chars_read < MIN_PKT_SIZE) {
	    continue;
	}
	to_host = atohost(buffer + DEST_HOST);
	to_port = atoport(buffer + DEST_PORT);
	from_host = atohost(buffer + SRC_HOST);
	from_port = atoport(buffer + SRC_PORT);
#ifdef DEBUG
	printf("to: host 0x%x, port %d; ", to_host, to_port);
	printf("from: host 0x%x, port = %d; our port = %d\n",
	       from_host, from_port, port);
	fflush(stdout);
#endif

	if (is_our_machine(to_host) && to_port == port) {
	    /*
	     * This message is directed to the relay server from an smx
	     * bootstrap---the UDP address of the new smx is in the source
	     * field.
	     */
	    if (num_contacts == array_size) {
		array_size += REALLOC_CHUNK;
		if ((contacts = realloc(contacts, array_size *
					sizeof(*contacts))) == 0) {
		    return -1;
		}
	    }
	    contacts[num_contacts].ip = from_host;
	    contacts[num_contacts].port = from_port;
#ifdef DEBUG
	    printf("relay: Adding host 0x%x/%d\n", from_host, from_port);
	    fflush(stdout);
#endif
	    num_contacts++;
	    continue;
	}

	if (to_host == 0xffffffff && to_port == 0xffff) {
	    int i;

	    /*
	     * Broacdast to everyone
	     */
	    for (i = 0; i < num_contacts; i++) {
		if (contacts[i].ip == from_host &&
		    contacts[i].port == from_port) {
		    /* Hosts don't get their own broadcasts */
		    continue;
		}
		forward_mesg(sock, contacts[i].ip, contacts[i].port, buffer,
			     chars_read);
	    }
	} else {
	    forward_mesg(sock, to_host, to_port, buffer, chars_read);
	}
    }
}


/*
 * Function: is_our_machine
 * Parameter: host - ip address to be checked
 * Returns: true if host is an IP address of this machine, false otherwise.
 */
static int is_our_machine(u_int host)
{
    static u_int *our_nums = 0;
    u_int *np;

    if (our_nums == 0) {
	if ((our_nums = our_ip_nums()) == 0) {
	    fprintf(stderr, "%s: falied to get local IP numbers\n", progname);
	    exit(1);
	}
    }

    for (np = our_nums; *np; np++) {
	if (*np == host) {
	    return 1;
	}
    }
    return 0;
}


/*
 * Function: our_ip_nums
 * Returns: a pointer to a malloc'ed arrat containing all of the ip addresses
 *          for this machine returned by gethostname.  The list is terminated
 *          by an ip address of 0.
 */
static unsigned int *our_ip_nums(void)
{
    struct hostent *hp;
    u_int *nums;
    u_int **ip;
    int cnt;

    if ((hp = gethostbyname(get_ourhostname())) == 0) {
	return 0;
    }

    assert(sizeof(*nums) == hp->h_length);
    ip = (unsigned int **) hp->h_addr_list;
    for (cnt = 0; *ip; ip++, cnt++) {
	;
    }
    if ((nums = malloc((cnt + 1) * sizeof(*nums))) == 0) {
	return 0;
    }
    ip = (unsigned int **) hp->h_addr_list;
    for (cnt = 0; *ip; ip++, cnt++) {
	nums[cnt] = **ip;
    }
    nums[cnt] = 0;
    return nums;
}


/*
 * Function: forward_mesg
 * Parameters: sock - socket to send message on.
 *             host, port - UDP/IP address to send to
 *             buffer - packet to send
 *             buff_len - length of packet to send
 *
 * Sends the specified packet to the specified UDP/IP address on the specified
 * socket.
 */
static void forward_mesg(int sock, u_int host, u_short port, char *buffer,
			 int buff_len)
{
    struct sockaddr_in addr;

    addr.sin_addr.s_addr = htonl(host);
    addr.sin_port = htons(port);
    if (sendto(sock, buffer, buff_len, 0, (struct sockaddr *) &addr,
	       sizeof(addr)) != buff_len) {
	fprintf(stderr, "%s: send failed to %d/%d; errno %d\n", progname,
		host, port, errno);
    }
}

