static char rcsid[] = "$Id: logging.c,v 1.5 1996/08/01 02:08:34 paul Exp $";

#include "kernel.h"
#include "logging.h"
#include <sun/syscall.h>
#include "proc.h"

/*
 * This file contains functions to log event records to a disk file.
 * It is part of the package of the interaction network recording 
 * package that has been added to smx.
 *
 * The entry points are:
 *    - log_init - set logging_enabled if the minx bootstrap opened a lof file
 *    - event_log - add an event record to the log (if logging is enabled)
 *    - next_stn  - get the next sub-task number
 *
 * Event records are built up in log_buffer.  It is written to the file
 * when full, or when a TTY_INPUT event occurs (means that all event
 * record for the previous interaction are most likely flushed to disk).
 *
 * If LOG_ALL is defined then all events (and not just those that result
 * user inputs) are recorded (if monitoring is enabled). Also, event
 * records are flushed to SunOS rather than buffered so that is smx crashes
 * no event records are lost.
 */

/*#define LOG_ALL*/

#define LBUFF_SIZE 8192

static char log_buffer[LBUFF_SIZE];   /* event records accumulated here */
static int buff_pos = 0;              /* current posn in log_buffer */

int logging_enabled = 0;


/*
 * Function: log_init
 *
 * Checks to see if logging is enabled.  If LOGGING_FD is a valid descriptor
 * then the Minix bootstrap must have open a logging file on it, so logging
 * is enabled.
 */
void log_init(void)
{
    int fd;

    if ((fd = SunOS(SYS_dup, LOGGING_FD)) != -1) {
        SunOS(SYS_close, fd);
	logging_enabled = 1;
    }
}


/*
 * Function: log_flush
 *
 * Writes log_buffer to the log file then resets buff_pos back to the
 * beginning of log_buffer.
 */
static void log_flush(void)
{
    if (logging_enabled && buff_pos > 0) {
	SunOS(SYS_write, LOGGING_FD, log_buffer, buff_pos);
	buff_pos = 0;
    }
}


/*
 * Function: write_to_log
 * Parameters: buffer - buffer to add to the log
 *             buff_len - length of buffer
 *
 * If buff_len is < 1 or > than the buffer size the buffer is quietly
 * ignored.  If there isn't enough space in log_buffer for the
 * contents of buffer then log_buffer is flushed to the log file to
 * make room for buffer.  If LOG_ALL is defined, then a flush occurs
 * for each call to write_to_log.
 */
static void write_to_log(char *buffer, int buff_len)
{
    if (buff_len < 1 || buff_len > LBUFF_SIZE) return;

#ifndef LOG_ALL
    if (buff_pos + buff_len > LBUFF_SIZE) {
	log_flush();
    }
#endif

    memcpy(log_buffer + buff_pos, buffer, buff_len);
    buff_pos += buff_len;
#ifdef LOG_ALL
    log_flush();
#endif
}


/*
 * Function: event_log
 * Parameters: event - the code of the event that has occurred
 *             p - process event occurred in.
 *             other_stn - an additional sub-task number (relevant to some
 *                         event types).
 *             buff, buff_len - per-event additional info.
 *
 * If logging is enabled, and the process stn shows that it is part of
 * an interaction of interest, then an event record is constructed and
 * logged.
 */
void event_log(int event, struct proc *p, int other_stn, void *buff,
	       int buff_len)
{
    struct ev_record ev;

    if (!logging_enabled) {
        return;
    }

#ifndef LOG_ALL
    if (p->p_stn == INVALID_STN) return;

    if (event == EV_TTY_INPUT) {
	/*
	 * Ensure that all event records for the previous input are safely
	 * flushed to SunOS.
	 */
	log_flush();
    }
#endif

    ev.e_event = event;
    ev.e_pnr = p->p_nr;
    ev.e_stn = p->p_stn;
    ev.e_other_stn = other_stn;
    SunOS(SYS_gettimeofday, &ev.e_secs, 0);
    ev.e_bufflen = buff_len;
    write_to_log((char *) &ev, sizeof(ev));
    write_to_log(buff, buff_len);
}
    

/*
 * Function: next_stn
 *
 * Returns the next sub-task number (it is just a simple sequence number).
 */
int next_stn(void)
{
    static int stn = 0;

    if (++stn < 0) {
	stn = 0;
    }
    return stn;
}
