/* This file contains the main program of MINIX.  The routine main()
 * initializes the system and starts the ball rolling by setting up the proc
 * table, interrupt vectors, and scheduling each task to run to initialize
 * itself.
 *
 * The entries into this file are:
 *   main:		MINIX main program
 *   panic:		abort MINIX due to a fatal error
 */

#include "kernel.h"
#include <unistd.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "proc.h"
#if (MACHINE == SUN)
#include <minix/boot.h>
#include <sun/syscall.h>
#include "logging.h"
#include <net/gen/ether.h>
#include "bootinfo.h"
#endif

#if (MACHINE == SUN)
/*
 * Must be first thing main's data segment (initialisation ensures that it 
 * goes into the data segment).  Also, main.o must be the first file in the
 * link that has a data segment.
 *
 * Because R8192 had to be removed from the data segment entry, the bootstrap
 * program can no longer rely on the start of the data segment being aligned
 * on an 8192 boundary---it may now start anywhere in the first page reserved
 * for the data segment.  The minix bootstrap still aligns the smxboot_info
 * to an 8Kb boundary, however.  We declare the smxboot_space structure
 * so that we can be sure that there is enough space for the smx_bootinfo
 * structure, even if the data segment is 8Kb aligned.  We also use the
 * address of smxboot_space to initialise smxboot, as they must both lie
 * on the same 8Kb page.
 */
struct smx_bootinfo smxboot_space = {1};
struct smx_bootinfo *smxboot;
#endif

static char rcsid[] = "$Id: main.c,v 1.5 1996/07/09 19:41:35 paul Exp $";   /* Must follow smxboot! */


/*===========================================================================*
 *                                   main                                    *
 *===========================================================================*/
PUBLIC void main()
{
/* Start the ball rolling. */

  register struct proc *rp;
  register int t;
  int sizeindex;
  phys_clicks text_base;
  vir_clicks text_clicks;
  vir_clicks data_clicks;
  phys_bytes phys_b;
  reg_t ktsb;			/* kernel task stack base */
  struct memory *memp;
  struct tasktab *ttp;
#if (MACHINE == SUN)
  unsigned long addr;
  int prog_num;
  smxboot = (struct smx_bootinfo *) downclick((int) &smxboot_space);
  
  log_init();

  /* fetch values that were placed by the minix bootstrap */
  code_base = smxboot->prog[KERNEL_PROG].text_vaddr;
  tot_mem_size = smxboot->mem_bytes >> CLICK_SHIFT;
  set_debug_level(smxboot->debug);
#if ENABLE_NETWORKING
  se_init((char *)&smxboot->ether_addr);
#endif
  

  debug_str("Starting kernel - debugging on; smxboot addr = ");
  debug_int((int) &smxboot);
  debug_str("\n");

  /*
   * Unmap the text and data segments of the minix bootstrap program.  They
   * are no longer needed, and we will map in init and all other user programs
   * in that address range.
   */
  for (addr = 0x10000; addr < (unsigned long)code_base; addr += CLICK_SIZE) {
        SunOS(SYS_munmap, addr, CLICK_SIZE);
  }

  lock();    /* Lock out non-error SunOS signals until initialisation done */
#endif


#if (CHIP == INTEL)
  /* Initialize the interrupt controller. */
  intr_init(1);
#endif

  /* Interpret memory sizes. */
  mem_init();

  /* Clear the process table.
   * Set up mappings for proc_addr() and proc_number() macros.
   */
  for (rp = BEG_PROC_ADDR, t = -NR_TASKS; rp < END_PROC_ADDR; ++rp, ++t) {
	rp->p_flags = P_SLOT_FREE;
#if (MACHINE == SUN)
	rp->p_stn = INVALID_STN;   /* not currently part of an interaction */
#endif
	rp->p_nr = t;		/* proc number from ptr */
        (pproc_addr + NR_TASKS)[t] = rp;        /* proc ptr from number */
  }

  /* Set up proc table entries for tasks and servers.  The stacks of the
   * kernel tasks are initialized to an array in data space.  The stacks
   * of the servers have been added to the data segment by the monitor, so
   * the stack pointer is set to the end of the data segment.  All the
   * processes are in low memory on the 8086.  On the 386 only the kernel
   * is in low memory, the rest if loaded in extended memory.
   */

  /* Task stacks. */
  ktsb = (reg_t) t_stack;

  for (t = -NR_TASKS; t <= LOW_USER; ++t) {
	rp = proc_addr(t);			/* t's process slot */
	ttp = &tasktab[t + NR_TASKS];		/* t's task attributes */
	strcpy(rp->p_name, ttp->name);

#if (MACHINE == SUN)
	/*
	 * Initialise the context for each process with the current
	 * context of the SunOS process.  Critical values (sp, pc, npc)
	 * will be set before the context is used to start the smx process.
	 */
	SunOS(SYS_context, 0, &rp->p_reg);
#endif

	if (t < 0) {
		if (ttp->stksize > 0) {
			rp->p_stguard = (reg_t *) ktsb;
			*rp->p_stguard = STACK_GUARD;
		}
		ktsb += ttp->stksize;
		rp->p_reg.sp = ktsb;
		text_base = code_base >> CLICK_SHIFT;
					/* tasks are all in the kernel */
		sizeindex = 0;		/* and use the full kernel sizes */
		memp = &mem[0];		/* remove from this memory chunk */
	} else {
		sizeindex = 2 * t + 2;	/* MM, FS, INIT have their own sizes*/
	}
	rp->p_reg.pc = (reg_t) ttp->initial_pc;
	rp->p_reg.psw = istaskp(rp) ? INIT_TASK_PSW : INIT_PSW;

#if (MACHINE == SUN)
	/*
	 * Use the boot info to determine the sizes of the text and
	 * data segments.  All tasks are within the kernel.
	 */
	if (istaskp(rp) || isidlehardware(proc_number(rp))) {
	        prog_num = KERNEL_PROG;
	} else {
	        prog_num = proc_number(rp) + (MM_PROG - MM_PROC_NR);
	}

	text_clicks = smxboot->prog[prog_num].text_clicks;
	data_clicks = smxboot->prog[prog_num].data_clicks;
#endif

	rp->p_map[T].mem_phys = text_base;
	rp->p_map[T].mem_len  = text_clicks;
	rp->p_map[D].mem_phys = text_base + text_clicks;
	rp->p_map[D].mem_len  = data_clicks;
	rp->p_map[S].mem_phys = text_base + text_clicks + data_clicks;
	rp->p_map[S].mem_vir  = data_clicks;	/* empty - stack is in data */

#if (MACHINE == SUN)
	/*
	 * Servers and user programs are mapped into the standard
	 * SunOS addresses.  The Virtual addresses are specified in
	 * the boot info.  The kernel is executed "in-situ", so the
	 * virtual addresses are the same as physical addresses.
	 */
	if (istaskp(rp) || isidlehardware(proc_number(rp))) {
	        rp->p_map[T].mem_vir = rp->p_map[T].mem_phys;
		rp->p_map[D].mem_vir = rp->p_map[D].mem_phys;
		rp->p_map[S].mem_vir = rp->p_map[S].mem_phys;
	} else {
	        rp->p_map[T].mem_vir = smxboot->prog[prog_num].text_vaddr >>
		    CLICK_SHIFT;
	        rp->p_map[D].mem_vir = smxboot->prog[prog_num].data_vaddr >>
		    CLICK_SHIFT;
	        rp->p_map[S].mem_vir = rp->p_map[D].mem_vir + data_clicks;
	        rp->p_reg.pc = smxboot->prog[prog_num].entry;
   	        rp->p_reg.sp = rp->p_map[S].mem_vir << CLICK_SHIFT;
	}
	if (prog_num == INIT_PROG) {
	    /*
	     * Take the last click of the data/stack segment for the stack.
	     * The src/tools/Makefile must ensure that the gap is big
	     * enough to allow this.
	     */
	    rp->p_map[S].mem_len = 1;
	    rp->p_map[S].mem_phys -= 1;
	    rp->p_map[S].mem_vir = (code_base >> CLICK_SHIFT) -
		rp->p_map[S].mem_len;
	    rp->p_reg.sp = (rp->p_map[S].mem_vir + rp->p_map[S].mem_len) <<
		CLICK_SHIFT;
	    rp->p_map[D].mem_len -= 1;
	} else {
	    rp->p_map[S].mem_len = 0;
	}
	rp->p_reg.npc = rp->p_reg.pc + 4;
	rp->p_reg.sp -= INIT_SP;       /*leave room for a stack frame*/

#endif /* MACHINE == SUN */
	debug_str("Process "); debug_int(t);
	debug_str(": pc = "); debug_int(rp->p_reg.pc);
	debug_str(", sp = "); debug_int(rp->p_reg.sp);
	debug_str("\n");
	text_base += text_clicks + data_clicks;	/* ready for next, if server */

	memp->size -= (text_base - memp->base);
	memp->base = text_base;			/* memory no longer free */

#if (CHIP == INTEL)
	if (t >= 0) {
		/* Initialize the server stack pointer.  Take it down one word
		 * to give crtso.s something to use as "argc".
		 */
		rp->p_reg.sp = (rp->p_map[S].mem_vir +
				rp->p_map[S].mem_len) << CLICK_SHIFT;
		rp->p_reg.sp -= sizeof(reg_t);
	}

#if _WORD_SIZE == 4
	/* Servers are loaded in extended memory if in 386 mode. */
	if (t < 0) {
		memp = &mem[1];
		text_base = 0x100000 >> CLICK_SHIFT;
	}
#endif
#endif /* CHIP == INTEL */
	if (!isidlehardware(t)) lock_ready(rp);	/* IDLE, HARDWARE neveready */
	rp->p_flags = 0;

#if (CHIP == INTEL)
	alloc_segments(rp);
#endif
  }

  proc[NR_TASKS+INIT_PROC_NR].p_pid = 1;/* INIT of course has pid 1 */
  bill_ptr = proc_addr(IDLE);		/* it has to point somewhere */
  lock_pick_proc();

#if (MACHINE == SUN)
  init_sun_handlers();                 /* Initialise sun signal handling */
  /*
   * Setup the smx boot information.
   */
  boot_parameters.bp_rootdev = DEV_FD0;
  boot_parameters.bp_ramimagedev = 0;
  boot_parameters.bp_ramsize = 0;
  boot_parameters.bp_processor = CHIP;

  /*
   * Initialise the protection according to the protection level
   * specified from the minix bootstrap.
   */
  protect_init(code_base, smxboot->prog[KERNEL_PROG].text_clicks,
	       smxboot->prog[KERNEL_PROG].data_clicks, tot_mem_size,
	       smxboot->prot_lev);

  /* enable all SunOS signals */
  unlock();
#endif /* MACHINE == SUN */

  debug_str("In kernel main---about to call restart\n");

  /* Now go to the assembly code to start running the current process. */
  restart();
}


/*===========================================================================*
 *                                   panic                                   *
 *===========================================================================*/
PUBLIC void panic(s,n)
_CONST char *s;
int n;
{
/* The system has run aground of a fatal error.  Terminate execution.
 * If the panic originated in MM or FS, the string will be empty and the
 * file system already syncked.  If the panic originates in the kernel, we are
 * kind of stuck.
 */

  if (*s != 0) {
	printf("\nKernel panic: %s",s);
	if (n != NO_NUM) printf(" %d", n);
	printf("\n");
  }
#if (MACHINE == SUN)
  /* a useful register and context dump */
  do_dump();
#endif
  wreboot(RBT_PANIC);
}
