static char *rcsid = "$Id: mpx.c,v 1.2 1996/07/09 19:41:39 paul Exp $";

/*
 * This file contains fucntions to handle context switching and SunOS signals
 * in smx.  As signals cause context switches, the two are closely related.
 * The entry points for the file are:
 *
 *       init_sun_handlers - installs handlers for all SunOS signals
 *       restart - called from the bootstrap code to schedule the first process
 *       lock - disable non-error SunOS signals
 *       unlock - restore the signal mask to its state prior to the lock call
 *       idle_task - the main body of the idele task.
 *
 * The SunOSsig function could also be considered an entry point.  It is
 * the handler for all SunOS signals, and so is called whenever a SunOS
 * signal occurs.  It is the sole entry point for the layer 1 code once
 * the first process has been scheduled for the first time.
 */

#include "kernel.h"
#include "proc.h"
#include <sun/syscall.h>
#include <sun/signal.h>

/*
 * Check that the context size calculated by make_context (UC_SIZE)
 * matches the smx CONTEXT_SIZE.
 */
#if (CONTEXT_SIZE != UC_SIZE)
error "p_reg is the wrong size"
#endif

/*
 * Local functions.
 */
static void SunOSsig(int sig, void *siginfo, reg_t ucontext[]);
static void s_call(void);
static void smx_clock_handler(void);
static so_sigset_t get_inkernel_mask(void);

/*
 * vectors specifies for each SunOS signal the kernel function that
 * should be called to handle it.
 */ 
static void (*vectors[_SIGRTMAX + 1])();

static double gwin_save_area[GWIN_SIZE / sizeof(double) + 1];
struct proc *gwin_proc = 0;

/*
 * Function: init_sun_handlers
 *
 * Sets up the handling of all SunOS signals by:
 *    - installing the kernel stack as the alternate signal stack so that
 *      all signal handling will make use of that stack.
 *    - installing SunOSsig as the handler for all functions
 *    - setting up vectors to record the kernel function to call on the
 *      occurrence of each smx signal.
 */
void init_sun_handlers(void)
{
    struct so_sigaction sig; 
    struct so_sigaltstack sstk;
    int t;

    /*
     * tell SunOS the location of the stack to use for handling 
     * SunOS signals
     */
    sstk.ss_sp = (char *)getksp() - K_STACK_BYTES;
    sstk.ss_size = K_STACK_BYTES; 
    sstk.ss_flags = 0;
    SunOS(SYS_sigaltstack, &sstk, (so_stack_t *)0);
    debug_str("Kernel stack from ");
    debug_int((int) sstk.ss_sp); debug_str("\n");

    /* 
     * each of SunOS's signals go through a special handler that takes the
     * current Minix process's context and stores it in the proc table 
     */ 
    sig.sa_flags = SUN_SA_ONSTACK | SUN_SA_SIGINFO;
    sig.sa_handler = SunOSsig; 
    sig.sa_mask = get_inkernel_mask(); 
    for (t = 1; t <= _SIGRTMAX; t++) {
 	SunOS(SYS_sigaction, t, &sig, (struct sigaction *) 0); 
    }

    /* 
     * The SunOSsig signal handler uses 'vectors' to decide which function
     * should be called to handle each SunOS signal (eg. SIGIO).
     */
    for (t = 0; t <= _SIGRTMAX; t++) { 	
	vectors[t] = exception; 
    }

    vectors[SO_SIGUSR1] = s_call; 
    vectors[SO_SIGIO] = sunio_interrupt;
    vectors[SO_SIGALRM] = smx_clock_handler; 
}


/*
 * mpx_start and mpx_end are dummy functions used to delimit the section
 * of this file (the resume and SunOSsig functions) that must remain
 * executable even when execution is outside the kernel.  These labels
 * are needed by code in sunprotect.c
 */

void mpx_start() {}

/*
 * Function: resume
 *
 * Resume execution of the the current process as given by proc_ptr.
 * We make a local copy of the context to restore for two reasons: to double
 * word align it, and so that if full memory protection is enabled we
 * still have access to the context (which is on the kernel stack)
 * after leaving_kernel is called.
 */
static void resume(reg_t *ctxt)
{
    /*
     * The Sun context structure is double word aligned, so it's
     * best that our copy is as well.
     */
    int i;
    double local_save_area[GWIN_SIZE / sizeof(double) + 1];
    
    memcpy(ctxt, &proc_ptr->p_reg, sizeof(proc_ptr->p_reg));
    
    /*
     * Clear signal mask---no SunOS signals are masked when execution is
     * outside layer 1.
     */
    for (i = 0; i < sizeof(so_sigset_t) / sizeof(i); i++) {
	ctxt[UC_SIGMASK + i] = 0;
    }
    /*
     * Set the sig mask flag bit to ensure the signal mask gets restored.
     */
    ctxt[UC_FLAGS] |= 1;
    if (ctxt[UC_GWINS] != 0) {
	gwin_proc = 0;
	memcpy(local_save_area, gwin_save_area, sizeof(local_save_area));
	ctxt[UC_GWINS] = (reg_t) local_save_area;
    }
    leaving_kernel();

    /*
     * Do a context switch.  This call should never return, with layer 1 code
     * begin re-entered via SunOSsig.
     */
    SunOS(SYS_context, 1, ctxt);
    printf("Gwin = 0x%x, fpuq = 0x%x, ctxt = 0x%x\n", ctxt[UC_GWINS],
	   ctxt[UC_FPU_Q], ctxt);
    panic("resume: failed to resume an smx process", errno);
}    


/*
 * Function: SunOSsig
 * Parameters: sig - the SunOS signal that has occurred
 *             siginfo - additional info about the signal
 *             ucontext - context for the smx process that was running when
 *                        the SunOS signal ocurred.
 *
 * This function is the handler for all signals received by the SunOS
 * process running smx.  The 'ucontext' structure pushed onto the
 * kernel (signal) stack by Solaris is copied into the Minix proc
 * table entry of the currently executing Minix process. The array
 * 'vectors' is used to determine which function within the Minix
 * kernel should then take responsibilty for the signal. Finally, the
 * context of an smx process is restored.
 */
static void SunOSsig(int sig, void *siginfo, reg_t ucontext[])
{
    entering_kernel();

    /*
     * Save the ucontext structure into the current proc table entry.
     */
    if (ucontext[UC_GWINS] != 0) {
	if (gwin_proc != 0) {
	    panic("SunOSsig---save area used\n", NO_NUM);
	}
	if (!isuserp(proc_ptr)) {
	    panic("UC_GWINS non-zero in process", proc_number(proc_ptr));
	}
	gwin_proc = proc_ptr;
	memcpy(gwin_save_area, (char *) ucontext[UC_GWINS],
	       sizeof(gwin_save_area));
    }
    memcpy(&proc_ptr->p_reg, ucontext, sizeof(proc_ptr->p_reg));


    (*vectors[sig])(sig, siginfo, ucontext);      /* Call the handler */
	
    lock_pick_proc();         /* (possibly) choose a new process */

    resume(ucontext);
    /*NOTREACHED*/
}

void mpx_end() {}


/*
 * Function: restart
 *
 * The 'restart' function is called (only once) from 'main.c' in order
 * to get the the first process running. 
 */
void restart(void)
{
    struct stackframe_s dummy_regs;

    k_reenter = 0;      /* Nested interrupt handling is not supported */

    resume((reg_t *) &dummy_regs);
    /*NOTREACHED*/
}


/*****************************************************************************
 * Some of the functions found in the vectors array.                         *
 *****************************************************************************/

/*
 * Function: s_call
 *
 * s_call is called from SunOSsig to handle the SunOS USR1 signal,
 * which is sent by an smx process in layers 2-4 to force execution
 * into layer 1 of smx.  The three system calls are send, receive,
 * and sndrcv.  s_call simply sets up parameters for a call to the
 * standard minix sys_call function, and passes the return result
 * back to the calling smx process.
 */
static void s_call(void)
{
    phys_bytes param_ptr;       /* ptr to the system call parameters */
    int function;		/* SEND, RECEIVE, or BOTH */
    int src_dest;		/* source to receive from or dest to send to */
    message *m_ptr;		/* pointer to message */
    int params[3];
    struct proc *this_proc = proc_ptr;

    /*
     * The arguments to sys_call are on the stack.  For the SPARC
     * stack frame, the first argument is at sp + 92.
     */
    param_ptr = umap(this_proc, D, this_proc->p_reg.sp + 92, sizeof(params));
    if (param_ptr == 0) {
	panic("s_call: Can't map in parameter to system call", NO_NUM);
    }

    /* Load the parameters */
    phys_copy(param_ptr, vir2phys(&params[0]), sizeof(params));
    function = params[0];
    src_dest = params[1];
    m_ptr = (message *)params[2];   /* Assumes ints can hold a pointer */

    this_proc->p_reg.retreg = sys_call(function, src_dest, m_ptr);
}


/*
 * Function: smx_clock_handler
 *
 * Called by SunOSsig whenever a SIGALRM occurs.  Calls clock handler
 * with 0 ticks, as in smx the SunOS clock is always consulted to
 * update the time on a clock tick.
 */
static void smx_clock_handler(void)
{
    clock_handler(0);
}


/*==========================================================================*
 *                             lock etc                                     * 
 *==========================================================================*/

/*
 * It is assumed that lock/unlock calls are strictly made in pairs 
 * (lock, unlock, lock, unlock,...).  If this isn't the case, then
 * the use of a single variable to store the signal mask saved during a
 * locked period will not work.  Lock and Unlock contain checks to 
 * ensure that the alternation of calls is maintained.  Furthermore,
 * the initialisation of several variables depends on main calling lock
 * then unlock.
 */

so_sigset_t savemask;       /* Saved signal mask during a loced period */
static int locked = 0;      /* Used to enforce alternation. */


/*
 * Function: get_inkernel_mask
 * Returns: the SunOS signal mask that should be installed while execution
 * is in layer 1.  All but "error" signals should be masked out.
 */
static so_sigset_t get_inkernel_mask(void)
{
    static so_sigset_t lockmask;
    static int called_before = 0;

    if (!called_before) {
	/*
	 * Initialise lockmask.  Only Sun error signals should be allowed
	 * through.
	 */
	sunsigfillset(&lockmask);
	sunsigdelset(&lockmask, SO_SIGILL);
	sunsigdelset(&lockmask, SO_SIGTRAP);
	sunsigdelset(&lockmask, SO_SIGEMT);
	sunsigdelset(&lockmask, SO_SIGFPE);
	sunsigdelset(&lockmask, SO_SIGBUS);
	sunsigdelset(&lockmask, SO_SIGSEGV);
	sunsigdelset(&lockmask, SO_SIGSYS); 
	called_before = 1;
    }
    return lockmask;
}


/*
 * Function: lock
 *
 * Sets the signal mask to allow delivery of "error" signals only, and
 * saves previous signal mask
 */
void lock()
{
    static so_sigset_t lockmask;    /* Signal mask to use in locked periods */
    static int called_before = 0;

    lockmask = get_inkernel_mask();
    if (!called_before) {
	/*
	 * The first call to lock is made in main().  When main calls 
	 * unlock(), we want to install an empty signal mask, so 
	 * the savemask is set to an empty set here.
	 */
	SunOS(SYS_sigprocmask, SO_SIG_SETMASK, &lockmask, 0);
	sunsigemptyset(&savemask);

	called_before = 1;
    } else {
	SunOS(SYS_sigprocmask, SO_SIG_SETMASK, &lockmask, &savemask);
    }

    if (locked) {   /* A recursive lock! */
	panic("lock() called when a lock was already in force", NO_NUM);
    }
    locked = 1;
}


/*
 * Function: unlock
 *
 * Unblocks Solaris signals by restoring the signal mask that was 
 * installed by lock.
 */

void unlock(void) 
{
    if (!locked) {
	panic("unlock() called without a prior call to lock()", NO_NUM);
    }
    locked = 0;

    SunOS(SYS_sigprocmask, SO_SIG_SETMASK, &savemask, 0);
}

	
/*==========================================================================*
 *                             idle_task                                    * 
 *==========================================================================*/

#define SIGPAUSE 0x1000

/* Does nothing but sleep waiting for a signal*/

void idle_task(void)
{
    for (;;) {
	SunOS(SYS_signal, (SIGPAUSE | 1));
    }
}
