static char rcsid[] = "$Id: sunconsole.c,v 1.4 1996/08/01 02:08:48 paul Exp $";

/*
 * The file contains the code for transferring Minix's terminal output into
 * SunOS.  The entry points for this file are:
 *     - scr_init - sets up output for the specified terminal line
 *     - putk - called by printk to print a character to the console.
 *     - handup - a handup has occurred for a terminal line.
 *
 * scr_init puts pointers to cons_write and cons_echo into tty structs,
 * so cons_write and cons_echo are indirectly callable from outside.
 */
 
#include "kernel.h"
#include <termios.h>
#include "proc.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include "tty.h"
#include <signal.h>
#include <sun/syscall.h>

#define CONSOLE		0

/*
 * Local functions.
 */
static void cons_write(tty_t *tp);
static void cons_echo(tty_t *tp, int c);
static void out_char(tty_t *tp, char c);

/*
 * Function: scr_init
 * Parameter: tp - tty struct whose line is being initialised.
 */
void scr_init(tty_t *tp)
{
    /* Initialize the keyboard driver. */
    kb_init(tp);

    /* Output functions. */
    tp->tty_devwrite = cons_write;
    tp->tty_echo = cons_echo;
}


/*
 * Function: cons_write
 * Parameter: tp - tty to write to (it data is queued for it)
 *
 * If there is data to output, and output is not inhibited, than all
 * queued data is output.  Terminal 0 uses TERMINAL_FD for input and
 * TERMINAL_FD + 1 for output; terminal 1 uses TERMINAL_FD + 2 for input
 * and TERMINAL_FD + 3 for output and so on.  That is why the output
 * descriptor for terminal n is TERMINAL_FD + 2n + 1.
 */
static void cons_write(tty_t *tp)
{
    int count;
    char buf[64];
    phys_bytes user_phys;

    /*
     * Check quickly for nothing to do, so this can be called often without
     * unmodular tests elsewhere.
     */
    if ((count = tp->tty_outleft) == 0 || tp->tty_inhibited) return;
  
    do {
	if (count > sizeof(buf)) count = sizeof(buf);
	user_phys = proc_vir2phys(proc_addr(tp->tty_outproc), tp->tty_out_vir);
	phys_copy(user_phys, vir2phys(buf), (phys_bytes) count);

	SunOS(SYS_write, TERMINAL_FD + 1 + tty_line(tp) * 2, buf, count);
	
	tp->tty_out_vir += count;
	tp->tty_outcum += count;
	tp->tty_outleft -= count;

    } while ((count = tp->tty_outleft) != 0 && !tp->tty_inhibited);

    /*
     * Reply to the writer if all output is finished.
     */
    if (tp->tty_outleft == 0) {
	tty_reply(tp->tty_outrepcode, tp->tty_outcaller, tp->tty_outproc,
		  tp->tty_outcum);
	tp->tty_outcum = 0;
    }
}
 

/*
 * Function: cons_echo
 * Parameters: tp - tty struct for terminal to write to
 *             c - character to write to tp
 */
static void cons_echo(tty_t *tp, int c)
{
    out_char(tp, c);
}


/*
 * Function: putk
 * Parameter: c - character to write
 *
 * Write character c to the system console.  If newline is being written,
 * output a carriage return first.  
 *
 * This procedure is used by the version of printf() that is linked with
 * the kernel itself.  The one in the library sends a message to FS, which is
 * not what is needed for printing within the kernel.
 */
void putk(char c)
{
    if (c) {
	if (c == '\n') out_char(&tty_table[CONSOLE], (int) '\r');
	out_char(&tty_table[CONSOLE], (int) c);
    }
}


/*
 * Function: out_char
 * Parameters: tp - tty struct for terminal to write to
 *             c - character to write to tp
 *
 * Output a single character to a terminal.  See above for discussion of
 * calculation of the fd to specify.
 */

static void out_char(tty_t *tp, char c)
{
    SunOS(SYS_write, TERMINAL_FD + 1 + tty_line(tp) * 2, &c, 1);
}


/*
 * Function: hangup
 * Parameter: connum - terminal number
 *
 * When a remote smx user exits from their terminal program, a special code is
 * sent and we must close the two file descriptors and kill all of the
 * session's processes.
 */
 
void hangup(int connum)
{
    struct tty *tp = &tty_table[connum];

    scr_init(tp);
    sigchar(tp, SIGHUP);
}

