static char rcsid[] = "$Id: sunether.c,v 1.2 1996/08/01 02:08:52 paul Exp $";

/*
 * sunether.c
 *
 * This file contains a "ethernet device driver" for smx.
 *
 * The valid messages and their parameters are:
 *
 *   m_type	  DL_PORT    DL_PROC   DL_COUNT   DL_MODE   DL_ADDR
 * |------------+----------+---------+----------+---------+---------|
 * | HARDINT	|          |         |          |         |         |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_WRITE	| port nr  | proc nr | count    | mode    | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_WRITEV	| port nr  | proc nr | count    | mode    | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_READ	| port nr  | proc nr | count    |         | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_READV	| port nr  | proc nr | count    |         | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_INIT	| port nr  | proc nr | mode     |         | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_GETSTAT	| port nr  | proc nr |          |         | address |
 * |------------|----------|---------|----------|---------|---------|
 * | DL_STOP	| port_nr  |         |          |         |	    |
 * |------------|----------|---------|----------|---------|---------|
 *
 * The messages sent are:
 *
 *   m-type	  DL_POR T   DL_PROC   DL_COUNT   DL_STAT   DL_CLCK
 * |------------|----------|---------|----------|---------|---------|
 * |DL_TASK_REPL| port nr  | proc nr | rd-count | err|stat| clock   |
 * |------------|----------|---------|----------|---------|---------|
 *
 *   m_type	  m3_i1     m3_i2       m3_ca1
 * |------------+---------+-----------+---------------|
 * |DL_INIT_REPL| port nr | last port | ethernet addr |
 * |------------|---------|-----------|---------------|
 */

/*
 * The smx ethernet device driver operates in a very simple fashion.
 * During smx boot, a file the ETHER_FD file descriptor is opened
 * as a UDP/IP socket connected to a relay process running under SunOS.
 * All outgoing packets are written to this descriptor.  Anything
 * received on the descriptor is treated as an incoming ethernet packet,
 * and is relayed back to the inet server.  The entry points are:
 *
 *    se_init - called during bootstrap
 *    dp8390_task - the ethernet device driver task's main body
 *    se_interrupt - called when an incoming packet has arrived.
 *
 * During the MINIX bootstrap, se_init is called to inform this driver of its
 * "ethernet address", which is actually the UDP address bound to the
 * ETHER_FD socket.  se_init also sets up interrupt handling for
 * incoming messages.  Later in the bootstrap dp8390_task is called
 * as the main body of the ethernet task is created (the name is kept the
 * same as the PC function to minimise changes).
 *
 * Outgoing packets are sent immediately by writing them to ETHER_FD, so
 * writing processes are never suspended.  A reading process is suspended
 * if there are no input packets waiting.  Up to NR_READ_BUFFERS incoming
 * packets can be buffered---any further packets will be dropped.
 *
 * ISSUES:
 *    - if packets are often dropped, then we should change the
 *      buffering, so that each packet is stored in a buffer of the
 *      appropriate size, and not one of maximum size.
 *    - currently, once do_init has been called the device driver remains
 *      in the "enabled" state forever.  Should a DL_STOP diable the driver
 *      (as in the dp8390 driver)?
 *    - currently, broadcasts are permitted whenever networking is enabled.
 *      Should we allow them only if DEF_BROAD was specified when the
 *      ethernet device driver was enabled?
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <net/gen/ether.h>
#include <net/gen/eth_io.h>

#include <sun/syscall.h>

#include "proc.h"
#include "assert.h"

#if ENABLE_NETWORKING

/*
 * There is only 1 "ethernet port"---ETHER_FD!
 */
#define SMX_PORT_NR 1


/*
 * IO vector definition.  The definition, plus the functions for manipulating
 * IO vectors, are taken straight from dp8390.c.
 */
#define IOVEC_NR        16

typedef struct iovec_dat {
    iovec_t iod_iovec[IOVEC_NR];
    int iod_iovec_s;
    int iod_proc_nr;
    vir_bytes iod_iovec_addr;
} iovec_dat_t;

/*
 * Constants for use with se_rwiovec specifying whether copying is from
 * the driver to the IO vector (STORE) or vice versa (LOAD).
 */
#define LOAD 1
#define STORE 2


/*
 * Information about a reader process awaiting an incoming packet, and about
 * queued incoming packets, is held in the read_info structure.  The buffered
 * packets start in read_buffers[first_used], and go to
 * read_buffers[(first_used + buffs_used - 1) % NR_READ_BUFFERS.  The lengths
 * of the buffered packets are stored in the corresponding entries of
 * chars_read.
 */
#define NR_READ_BUFFERS 4

typedef struct read_info {
    int  reader_waiting;           /* Is there a reader waiting? */
    int  client;                   /* Who is it? */
    iovec_dat_t read_iovec;        /* Where should the incoming packet go? */

    int  buffs_used;               /* number of buffer used */
    int  first_buff;               /* first buffer used */
    unsigned read_buffers[NR_READ_BUFFERS][ETH_MAX_PACK_SIZE / sizeof(unsigned) + 1];
    int chars_read[NR_READ_BUFFERS];
} read_info_t;


static read_info_t read_info;     /* Waiting read and buffered packet info */
static ether_addr_t smx_eaddr;    /* Our "ethernet address" */
static eth_stat_t estats;         /* Ethernet stats---most unused in smx. */
static int smxeth_tasknr;         /* Ethernet driver task number */
static int eth_enabled = 0;


/*
 * Local fucntions
 */
static void se_interrupt(int fd);

static void do_init(message *mp);
static int se_ethernet_on(ether_addr_t *eaddr);
static void do_stop(message *mp);

static void do_vwrite(message *mp, int vectored);
static void do_vread(message *mp, int vectored);
static void do_int(void);
static int se_flush(int fd);
static int se_recv(void);

static void se_rwiovec(char *buffer, iovec_dat_t *iovp, vir_bytes count,
		       int rw);
static int calc_iovec_size(iovec_dat_t *iovp);
static void se_next_iovec(iovec_dat_t *iovp);
static void get_userdata(int user_proc, vir_bytes user_addr, vir_bytes count,
			 void *loc_addr);
static void put_userdata(int user_proc, vir_bytes user_addr, vir_bytes count,
			 void *loc_addr);

static void do_getstat(message *mp);

static void se_reply(int whoto, int err, int status, vir_bytes bytes_read);


/*
 * Function: se_init
 * Parameter: eaddr - our ethernet address.
 *
 * Called from kernel main() during booting.  We record our ethernet
 * address, install se_interrupt as the hander for SIGIO signals on ETHER_FD,
 * and initialise read_info.
 */
void se_init(char *eaddr)
{
    smx_eaddr = *(ether_addr_t *) eaddr;
    read_info.reader_waiting = 0;
    read_info.buffs_used = read_info.first_buff = 0;
    sunio_install_handler(ETHER_FD, se_interrupt);
}


/*
 * Function: se_interrupt
 * Parameter: fd - SunOS descriptor input is available on.
 *
 * An ethernet packet is avilable.  Wakeup the smx ethernet driver to 
 * read the packet.  This is layer 1 code.
 */
static void se_interrupt(int fd)
{
    interrupt(smxeth_tasknr);
}



/*
 * Function: dp8390_task
 *
 * The main body of the smx ethernet device driver task.
 */
void dp8390_task(void)
{
    message m;
    int r;
  
    smxeth_tasknr = proc_number(proc_ptr);     /* For se_interrupt to use */
    
    while(1) {
	if ((r = receive(ANY, &m)) != OK) {
	    panic("smx_ether: receive failed", r);
	}
	switch (m.m_type)
	{
	case DL_WRITE:	do_vwrite(&m, FALSE);	break;
	case DL_WRITEV:	do_vwrite(&m, TRUE);	break;
	case DL_READ:	do_vread(&m, FALSE);	break;
	case DL_READV:	do_vread(&m, TRUE);	break;
	case DL_INIT:	do_init(&m);		break;
	case DL_GETSTAT: do_getstat(&m);	break;
	case DL_STOP:	do_stop(&m);		break;
	case HARD_INT:  do_int();               break;
	default:
	    panic("smx_ether: illegal message", m.m_type);
	}
    }
}


/*
 * Function: do_init
 * Parameter: mp - incoming DL_INIT message
 *
 * We mark the device as enabled, and return the address and number of
 * interfaces.  Initialisation fails if the port number to be initialised
 * is not 0 (smx supports only 1 interface), if smx has been started
 * with networking disabled (in which case our ethernet address is all
 * zeroes) or the mode requested involved mutlicast mode or promiscuous
 * mode (the relay program needs modifying to handle these).
 */
static void do_init(message *mp)
{
    message reply_mess;

    reply_mess.m_type = DL_INIT_REPLY;

    if (mp->DL_PORT != 0 || !se_ethernet_on(&smx_eaddr) ||
	(mp->DL_MODE & DL_PROMISC_REQ) || (mp->DL_MODE & DL_MULTI_REQ)) {
	reply_mess.m3_i1  = ENXIO;
	if (send(mp->m_source, &reply_mess) != OK) {
	    panic("smx_ether: unable to send in DL_INIT", NO_NUM);
	}
	return;
    }

    memset(&estats, 0, sizeof(estats));
    reply_mess.m3_i1 = mp->DL_PORT;
    reply_mess.m3_i2 = SMX_PORT_NR;
    *(ether_addr_t *) reply_mess.m3_ca1 = smx_eaddr;
    eth_enabled = 1;

    if (send(mp->m_source, &reply_mess) != OK) {
	panic("smx_ether: unable to send in DL_INIT", NO_NUM);
    }
}


/*
 * Function: se_ethernet_on
 * Parameter: eaddr - our ethernet address
 * Returns: true of networking is enabled (eaddr is non-zero), false if it
 *          is disabled.
 *
 * The minix bootstrap passes an all zeroes eaddr if networking is off.
 */
static int se_ethernet_on(ether_addr_t *eaddr)
{
    int i;
    
    for (i = 0; i < sizeof(eaddr->ea_addr); i++) {
	if (eaddr->ea_addr[i] != 0) {
	    return 1;
	}
    }
    return 0;
}


/*
 * Function: do_stop
 * Parameter: mp - incoming DL_STOP message
 *
 * Currently we just validate the port number.  Should we set eth_enabled
 * to false here, and discard buffered input?  What if a reader is waiting?
 */
static void do_stop(message *mp)
{
    assert(eth_enabled);

    if (mp->DL_PORT != 0) {
	panic("smx_ether: illegal port", mp->DL_PORT);
    }
}


/*
 * Function: do_vwrite
 * Parameters: mp - incoming DL_WRITE or DL_WRITEV message
 *             vectored - true if DL_WRITEV, false otherwise
 *
 * Sets up an iovec (based on the supplied iovec for a DL_WRITEV,
 * the buffer supplied for a DL_WRITE), copies the message into
 * a local buffer, then writes the complete packet to ETHER_FD.
 */
static void do_vwrite(message *mp, int vectored)
{
    iovec_dat_t iovec, tmp_iovec;
    int count, size;
    static char write_buffer[ETH_MAX_PACK_SIZE];
    
    assert(eth_enabled);

    if (mp->DL_PORT != 0) {
	panic("smx_ether: illegal port", mp->DL_PORT);
    }

    count = mp->DL_COUNT;
    iovec.iod_proc_nr = mp->DL_PROC;
    if (vectored) {
	/*
	 * A vectored write.  Read in the vector (or at least the first part
	 * of it if it is a long one), and calculate the total number of
	 * bytes in the iovec.  We do this on a copy of the iovec because
	 * calc_iovec_size is destructive.
	 */
	get_userdata(mp->DL_PROC, (vir_bytes) mp->DL_ADDR,
		     (count > IOVEC_NR ? IOVEC_NR : count) *
		     sizeof(iovec_t), iovec.iod_iovec);
	iovec.iod_iovec_s = count;
	iovec.iod_iovec_addr = (vir_bytes) mp->DL_ADDR;
	tmp_iovec = iovec;
	size = calc_iovec_size(&tmp_iovec);
    } else {  
	iovec.iod_iovec[0].iov_addr = (vir_bytes) mp->DL_ADDR;
	iovec.iod_iovec[0].iov_size = mp->DL_COUNT;
	iovec.iod_iovec_s = 1;
	iovec.iod_iovec_addr = 0;
	size = mp->DL_COUNT;
    }
    if (size < ETH_MIN_PACK_SIZE || size > ETH_MAX_PACK_SIZE) {
	panic("smx_ether: invalid packet size", size);
    }

    /*
     * Load the packet from user space, then output it
     */
    se_rwiovec(write_buffer, &iovec, size, LOAD);
    if (SunOS(SYS_write, ETHER_FD, write_buffer, size) != size) {
	panic("smx_ether: SunOS write failed", errno);
    }

    estats.ets_packetT++;
    se_reply(mp->DL_PROC, OK, DL_PACK_SEND, 0);
}


/*
 * Function: do_vread
 * Parameters: mp - incoming DL_READ or DL_READV message
 *             vectored - true if DL_READV, false otherwise
 *
 * First we set up an iovec (based on the supplied iovec for a DL_READV,
 * the buffer supplied for a DL_READ).  We then call se_recv which will
 * transfer a buffered packet and send a reply message, or will tell us
 * that there are no waiting packets.
 */
static void do_vread(message *mp, int vectored)
{
    iovec_dat_t tmp_iovec;
    int count;
    int size;

    assert(eth_enabled);

    count = mp->DL_COUNT;
    if (mp->DL_PORT != 0) {
	panic("smx_ether: illegal port", mp->DL_PORT);
    }

    if (read_info.reader_waiting) {
	panic("smx_ether: read already in progress", NO_NUM);
    }

    read_info.read_iovec.iod_proc_nr = mp->DL_PROC;
    if (vectored) {
	/*
	 * A vectored read.  Read in the vector (or at least the first part
	 * of it if it is a long one), and calculate the total number of
	 * bytes in the iovec.  We do this on a copy of the iovec because
	 * calc_iovec_size is destructive.
	 */
	get_userdata(mp->DL_PROC, (vir_bytes) mp->DL_ADDR,
		     (count > IOVEC_NR ? IOVEC_NR : count) *
		     sizeof(iovec_t), read_info.read_iovec.iod_iovec);
	read_info.read_iovec.iod_iovec_s = count;
	read_info.read_iovec.iod_iovec_addr = (vir_bytes) mp->DL_ADDR;

	tmp_iovec = read_info.read_iovec;
	size = calc_iovec_size(&tmp_iovec);
    } else {
	read_info.read_iovec.iod_iovec[0].iov_addr = (vir_bytes) mp->DL_ADDR;
	read_info.read_iovec.iod_iovec[0].iov_size = mp->DL_COUNT;
	read_info.read_iovec.iod_iovec_s = 1;
	read_info.read_iovec.iod_iovec_addr = 0;
	size = count;
    }
    if (size < ETH_MAX_PACK_SIZE) {
	panic("smx_ether: wrong packet size", size);
    }
    read_info.reader_waiting = 1;

    if (se_recv() == 0) {
	/*
	 * Nothing was waiting---reply to that effect:
	 */
	se_reply(mp->DL_PROC, OK, 0, 0);
    } /* else---se_recv has already replied */
}


/*
 * Function: do_int
 *
 * One or more packets are available for input.  These packets are read into
 * the available buffers following any packets already buffered.  Once all
 * packets have been read, se_recv is called to check to see whether it is now
 * possible to reply to a previous read.
 */
static void do_int(void)
{
    int pkt_size;
    int num_read;
    int buff;
    
    if (!eth_enabled) {
	se_flush(ETHER_FD);
	return;
    }

    if (read_info.buffs_used == 0) {
	buff = read_info.first_buff = 0;
    } else {
	buff = (read_info.first_buff + read_info.buffs_used) % NR_READ_BUFFERS;
    }

    for (;;) {
	if (read_info.buffs_used == NR_READ_BUFFERS) {
	    if (se_flush(ETHER_FD)) {
		estats.ets_missedP++;
/*		printf("Ethernet packet dropped because of full buffers\n");*/
	    }
	    break;
	}
	num_read = SunOS(SYS_read, ETHER_FD, read_info.read_buffers[buff],
			 sizeof(read_info.read_buffers[buff]));
	if (num_read <= 0) {
	    break;
	}
	if (num_read < ETH_MIN_PACK_SIZE) {
	    panic("smx_ether: received ethernet packet too small", num_read);
	}

	estats.ets_packetR++;
	read_info.chars_read[buff] = num_read;
	read_info.buffs_used++;
	
	buff = (buff + 1) % NR_READ_BUFFERS;
    }
    se_recv();
}


/*
 * Function: se_flush
 * Parameter: fd - SunOS descriptor
 * Returns: 1 if anything read from fd, 0 otherwise
 *
 * Read and discard all input currently available on fd.
 */
static int se_flush(int fd)
{
    int anything_read = 0;
    char flush_buff[100];

    for (;;) {
	if (SunOS(SYS_read, fd, flush_buff, sizeof(flush_buff)) <= 0) {
	    break;
	}
	anything_read = 1;
    }
    return anything_read;
}


/*
 * Function: se_recv
 * Returns: 1 if data was transferred and a reply sent to a process waiting
 *          to read a packet, 0 otherwise.
 *
 * If we have one or more buffered packets, and a process waiting to read
 * a packet, then we can transfer a packet to the reader.
 */
static int se_recv(void)
{
    if (!read_info.reader_waiting || read_info.buffs_used == 0) {
	return 0;       /* Transfer not possible */
    }

    /*
     * Transfer packet and reply to waiting reader.
     */
    se_rwiovec((char *)read_info.read_buffers[read_info.first_buff],
		  &read_info.read_iovec,
		  read_info.chars_read[read_info.first_buff], STORE);
    se_reply(read_info.read_iovec.iod_proc_nr, OK, DL_PACK_RECV,
	     read_info.chars_read[read_info.first_buff]);

    read_info.reader_waiting = 0;
    read_info.buffs_used--;
    if (read_info.buffs_used == 0) {
	read_info.first_buff = 0;
    } else {
	read_info.first_buff = (read_info.first_buff + 1) % NR_READ_BUFFERS;
    }
    return 1;
}

		  
/*
 * Function: se_rwiovec
 * Parameters: buffer - buffer withing smx_ether.
 *             iovp - details of an iovec in another process
 *             count - number of bytes to copy
 *             rw - LOAD if iovp -> buffer; STORE if buffer ->iovp
 *
 * Transfer count bytes between buffer and iovp in the direction given
 * by rw.  This operation is destructive on the information in iovp.
 */
static void se_rwiovec(char *buffer, iovec_dat_t *iovp, vir_bytes count,
		       int rw)
{
    int bytes, i;

    i = 0;
    while (count > 0) {
	if (i >= IOVEC_NR) {
	    /*
	     * Need next section of the io vector
	     */
	    se_next_iovec(iovp);
	    i = 0;
	    continue;
	}

	assert(i < iovp->iod_iovec_s);
	bytes = iovp->iod_iovec[i].iov_size;
	if (bytes > count) {
	    bytes = count;
	}

	if (rw == LOAD) {
	    get_userdata(iovp->iod_proc_nr, iovp->iod_iovec[i].iov_addr,
			  bytes, buffer);
	} else {
	    put_userdata(iovp->iod_proc_nr, iovp->iod_iovec[i].iov_addr,
			  bytes, buffer);
	}

	count -= bytes;
	buffer += bytes;
	i++;
    }
    assert(count == 0);
}


/*
 * Function: calc_iovec_size
 * Parameter: iovp - iovec whose size is to be determined.
 * Returns: the number of bytes of storage within the iovec
 *
 * Calculate the size of a request. Note that the iovec_dat
 * structure will be unusable after calc_iovec_size.
 */
static int calc_iovec_size(iovec_dat_t *iovp)
{
    int size;
    int i;

    size = 0;
    i= 0;
    while (i < iovp->iod_iovec_s) {
	if (i >= IOVEC_NR) {     /* get the next chunk */
	    se_next_iovec(iovp);
	    i = 0;
	    continue;
	}
	size += iovp->iod_iovec[i].iov_size;
	i++;
    }
    return size;
}


/*
 * Function: se_next_iovec
 * Parameter: iovp - iovec_dat that we want the next chunk of
 *
 * Load the next chunk of the iovec whose detail are recorded in iovp
 */
static void se_next_iovec(iovec_dat_t *iovp)
{
    assert(iovp->iod_iovec_s > IOVEC_NR);

    iovp->iod_iovec_s -= IOVEC_NR;

    iovp->iod_iovec_addr += IOVEC_NR * sizeof(iovec_t);

    get_userdata(iovp->iod_proc_nr, iovp->iod_iovec_addr, 
		 (iovp->iod_iovec_s > IOVEC_NR ? IOVEC_NR : iovp->iod_iovec_s)
		 * sizeof(iovec_t), iovp->iod_iovec); 
}


/*
 * Function: get_userdata
 * Parameters: user_proc - process data is coming from
 *             user_addr - address of buffer in user_proc
 *             count - number of bytes to transfer
 *             loc_addr - address of local buffer to copy the user buffer to
 *
 * Copy a buffer from user_proc to a local buffer.
 */
static void get_userdata(int user_proc, vir_bytes user_addr, vir_bytes count,
			 void *loc_addr)
{
    phys_bytes src;

    src = numap(user_proc, user_addr, count);
    if (!src) {
	panic("smx_ether: umap failed", NO_NUM);
    }

    phys_copy(src, vir2phys(loc_addr), (phys_bytes) count);
}


/*
 * Function: put_userdata
 * Parameters: user_proc - process data is going to
 *             user_addr - address of buffer in user_proc
 *             count - number of bytes to transfer
 *             loc_addr - address of local buffer to copy the user buffer from
 *
 * Copy a buffer from a local buffe to user_proc.
 */
static void put_userdata(int user_proc, vir_bytes user_addr, vir_bytes count,
			 void *loc_addr)
{
    phys_bytes dst;

    dst = numap(user_proc, user_addr, count);
    if (!dst) {
	panic("smx_ether: umap failed", NO_NUM);
    }
    phys_copy(vir2phys(loc_addr), dst, (phys_bytes) count);
}


/*
 * Function: do_getstat
 * Parameters: mp - incoming DL_GETSTAT message
 *
 * Copies the ethernet statistics to the requersting process.
 */
static void do_getstat(message *mp)
{
    if (mp->DL_PORT != 0) {
	panic("smx_ether: illegal port", mp->DL_PORT);
    }

    assert(eth_enabled);

    put_userdata(mp->DL_PROC, (vir_bytes) mp->DL_ADDR,
		 (vir_bytes) sizeof(estats), &estats);
    se_reply(mp->DL_PROC, OK, 0, 0);
}



/*
 * Function: se_reply
 * Parameters: whoto - who to reply to
 *             err - the error status of the request made
 *             status - a bit map, containing details on packet
 *                      send and receive
 *             bytes_read - number of bytes read (only non-zero if
 *                          status includes DL_PACK_RECV).
 *
 * Send a reply to a previous request message.
 */
static void se_reply(int whoto, int err, int status, vir_bytes bytes_read)
{
    message reply;
    int r;

    reply.m_type = DL_TASK_REPLY;
    reply.DL_PORT = 0;             /* The only port under smx */
    reply.DL_PROC = whoto;
    reply.DL_STAT = status | ((u32_t) err << 16);
    reply.DL_COUNT = bytes_read;
    reply.DL_CLCK = get_uptime();
    r = send(whoto, &reply);
    if (r < 0) {
	panic("smx_ether: send failed:", r);
    }
}

#endif /* ENABLE_NETWORKING */
