static char rcsid[]="$Id: sunexception.c,v 1.4 1996/07/09 19:41:47 paul Exp $";

/*
 * This file contains two functions, both of which are entry points:
 *     - exception - handles all SunOS signals that don't have
 *                   special-purpose handlers.
 *     - do_dump - called from panic to dump relevant informatiomn
 *
 * A SunOS signals is either passed on to the currently executing smx
 * process (SIGSEGV, etc), completely ignored (eg. SIGINT) or can cause
 * smx Minix to shutdown (SIGTERM).  If a signal in the first category
 * occurs within the kernel or one of the servers, then this is regarded
 * as a fatal error, and smx is shutdown after displaying some debugging
 * information.
 */

#include "kernel.h"
#include "proc.h"
#include <sun/signal.h>
#include <sun/syscall.h>

#include <signal.h>

/*
 * Given a SunOS signal number, this table is used to determine what to do
 * that signal. If the entry is a '0' the signal is ignored.  Otherwise,
 * the entey contains a Minix signal number that the SunOS signal number
 * should be translated to.  The SunOS IO, ALRM and USR1 signals have
 * special handlers because smx uses those signals for its own purposes),
 * so exception() is not called to handle those signals.
 */
int signum[_SIGRTMAX + 1] = {
    0,       /* Signal 0---no such SunOS signal */
    0,       /* SO_SIGHUP  */
    0,       /* SO_SIGINT  */
    0,       /* SO_SIGQUIT */
    SIGILL,  /* SO_SIGILL */
    SIGTRAP, /* SO_SIGTRAP */
    SIGABRT, /* SO_SIGABRT */
    0,       /* SO_SIGEMT  */
    SIGFPE,  /* SO_SIGFPE  */
    SIGKILL, /* SO_SIGKILL */
    SIGSEGV, /* SO_SIGBUS  */
    SIGSEGV, /* SO_SIGSEGV */
    SIGSEGV, /* SO_SIGSYS  */
    /* All zero from here */
};
 

/*
 * Function: exception
 * Parameters: sig - SunOS signal that has occurred
 *             siginf - extra information on that signal
 *             scp - saved context of the interrupted smx process.
 *
 * This function is called by the SunOS signal handler. The SunOS signal
 * is either passed on, ignored or causes a shutdown.
 */
void exception(int sig, void *siginf, struct stackframe_s *scp)
{
    int mx_sig = signum[sig];
    unsigned *sinf = siginf;
    struct proc *p;

    if (sig == SO_SIGTERM) {
  	printf("\nMinix terminated\n");
  	SunOS(SYS_exit, 0);
    }
    
    /* ignore some signals */
    if (mx_sig == 0) {printf("Ignoring sig %d\n", sig);return;}


    /*
     * First we need to check whether we have got a segmentation violation
     * which really indicates that stack growth is needed.  This happens
     * if we have a segmentation violation in a user process
     * between the stack and data segments.
     */
    if (isuserp(proc_ptr) && sig == SO_SIGSEGV &&
	sinf[3] < (proc_ptr->p_map[S].mem_vir << CLICK_SHIFT) &&
	sinf[3] > (proc_ptr->p_map[D].mem_vir + proc_ptr->p_map[D].mem_len)
	<< CLICK_SHIFT) {
	cause_sig(proc_number(proc_ptr), SIGSTKFLT);
	return;
    }
    
    /*
     * if an smx user process was executing, send it the appropriate
     * smx signal.  We check that the pc at which the signal arose
     * was within the text segment of the current process, and that
     * the current process was a user proces.  If the "signalling" pc
     * was outside the text segment of the running process then either
     * the pc was in layer 1 kernel code, or something has gone badly wrong,
     * so we abort minix in this situation.
     */
    if (scp->pc >= proc_ptr->p_map[T].mem_vir << CLICK_SHIFT &&
	scp->pc <= (proc_ptr->p_map[T].mem_vir + proc_ptr->p_map[T].mem_len)
	<< CLICK_SHIFT && isuserp(proc_ptr)) {
       	/*
	 * Send a signal to an smx process.  Uncomment the print to get
	 * useful debugging info, especially if the boot sequence is
	 * failing somehow.
	 */
printf("SunOS sig %d in %d (%s) accessing 0x%x at 0x%x\n", sig, proc_number(proc_ptr), proc_ptr->p_name, sinf[3], scp->pc);
  	cause_sig(proc_number(proc_ptr), mx_sig);
	return;
    }
    
    /*
     * An "error" signal within the kernel, MM or FS causes a shutdown
     */
    printf("\nSunOS signal number %d received within kernel\n", sig);
    printf("PC = 0x%x, accessing 0x%x, task = %d\n", 
	   proc_ptr->p_reg.pc, sinf[3], proc_number(proc_ptr));
    panic("Signal within kernel", NO_NUM);
}


/*
 * Function: do_dump
 *
 * Display debugging information. This is called from within panic.
 */
PUBLIC void do_dump()
{
    printf("\nSP = %x PSW = %x", proc_ptr->p_reg.sp, proc_ptr->p_reg.psw);
    printf("\n\n");
    p_dmp();
    map_dmp();
}
