static char rcsid[] = "$Id: sunfloppy.c,v 1.3 1996/06/04 09:01:39 paul Exp $";

/*
 * This file contains the device driver for accessing smx filesystems, each
 * which is stored as a SunOS file.  While called sunfloppy.c, this file
 * actually has nothing to do with floppy disks at all!
 *
 * The file contains one entry point:
 *   - floppy_task: main "program" for the floppy task.  Calls driver_task
 *                  to do the work, with callbacks being made to this file.
 *
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/boot.h>
#include <sun/syscall.h>
#include <sys/ioctl.h>
#include "driver.h"

static  int f_device;		/* current device */

static struct device *f_prepare(int device);
static int f_schedule(int proc_nr, struct iorequest_s *iop);
static int f_do_open(struct driver *dp, message *m_ptr);
static int f_do_ioctl(struct driver *dp, message *m_ptr);
static void f_init(void);


/*
 * Entry points to this driver.
 */
static struct driver f_dtab = {
    no_name,	/* current device's name */
    f_do_open,	/* open or mount */
    do_nop,	/* nothing on a close */
    f_do_ioctl,	/* ioctl */
    f_prepare,	/* prepare for I/O on a given minor device */
    f_schedule,	/* do the I/O */
    nop_finish,	/* schedule does the work, no need to be smart */
    nop_cleanup	/* nothing's dirty */
};


/*
 * Function: floppy_task
 *
 * The main program of the floppy task.  Calls driver task to do all the
 * work, supplying f_dtab to allow call backs into this .c file.
 */
void floppy_task(void)
{
    driver_task(&f_dtab);
    /* NOTREACHED */
}


/*
 * Function: f_prepare
 * Parameter: device - the device we are about to access
 * Returns: a null pointer on failer, a non-null "pointer" on success
 *
 * Prepare for I/O on a device. *
 */
static struct device *f_prepare(int device)
{
  if (device < 0 || device >= NR_SMX_DISKS) return(NIL_DEV);
  f_device = device;

  return (struct device *)1;     /* A bit ugly */
}


/*
 * Fucntion: f_schedule
 * Parameters: proc_nr - process making the IO request
 *             iop - the IO request
 * Returns: OK if the transfer is successful; a minix error code if an
 *          error occurs
 *
 * Do a transfer between process proc_nr and an smx filesystem
 * as specified by iop.
 */
static int f_schedule(int proc_nr, struct iorequest_s *iop)
{
    int opcode;
    int r;
    phys_bytes user_phys;

    /*
     * Isolate the Type of request
     */
    opcode = iop->io_request & ~OPTIONAL_IO;

    /*
     * Determine address where data is to go or to come from.
     */
    user_phys = numap(proc_nr, (vir_bytes) iop->io_buf, iop->io_nbytes);
    if (user_phys == 0) return(iop->io_nbytes = EINVAL);

    /*
     * With SunOS disks (disk files) we can simply lseek() to the correct
     * position.
     */
    if ( (r = SunOS(SYS_lseek, DISK_FD + f_device, iop->io_position, 0)) == -1)
  	return(iop->io_nbytes = EIO);

    /*
     * Next we can read or write a block by using SunOS's read and write.
     * lock() is needed in case the read or write gets interrupted by an
     * asynchronous signal, such as ALRM or IO.  Also, we need to ensure
     * that we have access to the area being trasnferred to/form.  Note how
     * the SunOS fd is determined by adding the device number to DISK_FD.
     * The disks are opened by the smx bootstrap programs on these fds.
     */
    set_protect(user_phys, iop->io_nbytes, PROT_READ_WRITE);          
    lock();	
    if (opcode == DEV_READ)	
  	r = SunOS(SYS_read, DISK_FD + f_device, user_phys, iop->io_nbytes);
    else
  	r = SunOS(SYS_write, DISK_FD + f_device, user_phys, iop->io_nbytes);
    unlock();
    set_protect(user_phys, iop->io_nbytes, PROT_NONE);
  
    /*
     * This extra case is for when we wish to make the SunOS disk files
     * read only. Even though they may mounted as read only from Minix's point
     * of view, it will still be written to when updating times etc.
     */
    if (r != iop->io_nbytes && !(opcode == DEV_WRITE && errno == 9)) {
	return iop->io_nbytes = EIO;
    }
    return(iop->io_nbytes = OK);
}


/*
 * Function: f_do_open
 * Parameters: dp - driver structure (will be ptr to f_dtab)
 *             m_ptr - device open message
 * Returns: OK if the operation is successful; ENXIO if the specified
 *          device does not exist.
 *
 * An smx file system is being opened.  All we do is call prepare to make
 * sure that the device number is in the right range.  Perhaps
 * the SunOS fd corresponding to the device should be checked to make sure
 * that a file has been opened on it.
 */
static int f_do_open(struct driver *dp, message *m_ptr)
{
    if (f_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

    return(OK);
}


/*
 * Function: f_do_open
 * Parameters: dp - driver structure (will be ptr to f_dtab)
 *             m_ptr - device open message
 * Returns: OK if the ioctl is successful; a minix error code if an
 *          error occurs.
 *
 * Currently the FIOISRDONLY ioctl is the only one supported.  For any
 * other ioctls, ENOTTY is returned in line with other Minix disk
 * drivers.
 */
static int f_do_ioctl(struct driver *dp, message *m_ptr)
{
    int fcntl_ret;
    unsigned int ro;
    phys_bytes get_phys;

    if (f_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

    if (m_ptr->REQUEST == FIOISRDONLY) {
	/*
	 * Use fcntl to see if the disk is writable
	 */
        fcntl_ret = SunOS(SYS_fcntl, DISK_FD + f_device, 3 /*F_GETFL*/, 0);
	if (fcntl_ret == -1) {
	    return(EIO);
	}

	ro = ((fcntl_ret & 03) == 0);   /* Read-only if write bit not set */

	/*
	 * Return the read-only flag to the caller.
	 */
        get_phys = numap(m_ptr->m_source, (vir_bytes) m_ptr->ADDRESS,
			 sizeof(ro));
        if (get_phys == 0) return(EINVAL);
        phys_copy(vir2phys(&ro), get_phys, (phys_bytes) sizeof(ro));

    } else {
        /*
	 * No other ioctls are implemented.
	 */
        return(ENOTTY);
    }
    return(OK);
}
