static char rcsid[] = "$Id: sunkeyboard.c,v 1.5 1996/08/01 02:08:56 paul Exp $";

/*
 * This file contains the keyboard driver for SunOS Minix.  There are three
 * groups of functions in this file associated with the three entry points:
 *    - keyboard_intr - the function called when a SIGIO is received
 *                      indicating that terminal input is available.
 *    - kb_init - called to initialise keyboard data structures for a 
 *                specified terminal line.  Also, kb_read is installed as
 *                the function to call to read input from the specified
 *                terminal line.
 *    - wreboot - Shutdown smx.  Doesn't really belong in this file.
 */
 
#include "kernel.h"
#include <termios.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "tty.h"
#include "proc.h"
#include <sun/syscall.h>

#define KB_IN_BYTES	 1024	/* size of keyboard input buffer */

#define min(x,y) ((x) < (y) ? (x) : (y))

#define kb_addr(tp)	   (&kb_lines[tty_line(tp)])

/*
 * Keyboard structure, 1 per tty.
 */
struct kb_s {
    char *ihead;			/* next free spot in input buffer */
    char *itail;			/* first used spot in input buffer */
    int icount;			        /* # codes in buffer */

    char ibuf[KB_IN_BYTES];	        /* input buffer */
};

static struct kb_s kb_lines[NR_CONS];


/*
 * Local functions
 */
static int hw_read(tty_t *tp, char *buffer, int len);
static void input_flush(tty_t *tp);

static void kb_read(tty_t *tp);
static int func_key(tty_t *tp, int scode);


/*===========================================================================*
 *		   Functions dealing with the SunOS IO signal    	     *
 *===========================================================================*/

/*
 * Function: keyboard_intr
 * Parameter: fd - SunOS file descriptor on which input is available
 *
 * The function is called when there is input available on the SunOS
 * descriptor fd, and fd is associated with one of the smx console devices.
 * keyboard_instr must transfer the input to a terminal buffer, and inform
 * the terminal driver that new input has arrived. Any terminal input that
 * won't fit into the keyboard buffer is discarded.  
 *
 * Maybe the TTY task should transfer input into the buffer (but
 * it's not awoken until the next clock tick).  On the other hand,
 * the PC MINIX keyboard interrupt handle also puts characters directly
 * into the buffers of the TTY task.
 */
static void keyboard_intr(int fd)
{
    struct kb_s *kb;
    tty_t *tp;
    int line;
    int num_read, to_read;

    line = (fd - TERMINAL_FD) / 2;
    if (line < 0 || line >= NR_CONS) {
	panic("Bad fd supplied to keyboard_intr", fd);
    }

    tp = &tty_table[line];
    kb = kb_addr(tp);  
    if (kb->icount < KB_IN_BYTES) {
	/*
	 * If there is space in the buffer then we go ahead and
	 * try to read from the tty.  If the free buffer space
	 * (KB_IN_BYTES - kb->icount) is in one chunk then we
	 * try to fill all of it in a single read.  Otherwise
	 * we free the space between head and the end of the
	 * buffer with the first read and, if all of that space
	 * is filled, wrap head around to the beginning of the
	 * buffer and try to fill the free space there.
	 */
	to_read = min(KB_IN_BYTES - kb->icount,
		      KB_IN_BYTES - (kb->ihead - kb->ibuf));
	num_read = hw_read(tp, kb->ihead, to_read);
	
	if (num_read > 0) {
	    tp->tty_events = 1;
	    force_timeout();
	    kb->icount += num_read;
	    kb->ihead += num_read;

	    if (kb->ihead == kb->ibuf + KB_IN_BYTES) {
		kb->ihead = kb->ibuf;
		if (kb->icount < KB_IN_BYTES) {
		    num_read = hw_read(tp, kb->ihead,
				       KB_IN_BYTES - kb->icount);

		    if (num_read > 0) {
			kb->icount += num_read;
			kb->ihead += num_read;
			/*
			 * Read at least 1 byte the first time; ihead
			 * cannot wraparound this time.
			 */
		    }
		} /* if space in input buffer for a second read */
	    } /* if first read wraps buffer */
	} /* if first read returns at least one char */
    } /* if space in input buffer */

    if (kb->icount == KB_IN_BYTES) {
	/*
	 * If the buffer is full then there may still be data buffered
	 * by SunOS.  Flush it to make sure that we get a SIGIO
	 * when the next character gets entered.
	 */
	input_flush(tp);
    }
}


/*
 * Function: hw_read
 * Parameters: tp - terminal to read from
 *             buffer - buffer to read into
 *             len - size of buffer
 * Returns: value returned by call to SunOS read.
 *
 * The terminal input descriptors are the even ones of the terminal
 * descriptor pair, so the FD to read is simply
 * TERMINAL_FD + 2 line_number.  The smx bootstrap program sets up
 * these terminal descriptors.
 */
static int hw_read(tty_t *tp, char *buffer, int len)
{
    return SunOS(SYS_read, TERMINAL_FD + tty_line(tp) * 2, buffer, len);
}


/*
 * Function: input_flush
 * Parameter: tp - terminal to be flushed
 *
 * Discard all buffered input for the spcified terminal.
 */
static void input_flush(tty_t *tp)
{
    static char flushbuf[32];

    while (hw_read(tp, flushbuf, sizeof(flushbuf)) > 0) {
	;
    }
}


/*===========================================================================*
 *			  The keyboard driver functions			     *
 *===========================================================================*/

/*
 * Function: kb_init
 * Parameter: tp - terminal whose keyboard driver is to be initialised
 *
 * Initialize the keyboard driver for the specified terminal line.
 */
void kb_init(tty_t *tp)
{
    register struct kb_s *kb;

    tp->tty_devread = kb_read;               /* Input function. */

    /*
     * Install kerboard_intr as the handler for input received on the
     * associated SunOS file descriptor.
     */
    sunio_install_handler(TERMINAL_FD + tty_line(tp) * 2, keyboard_intr);

    kb = kb_addr(tp);
    kb->ihead = kb->itail = kb->ibuf;        /* The input buffer is empty */
    kb->icount = 0;
}


/*
 * Function: kb_read
 * Parameter: tp - terminal to read input from
 *
 * Process characters from the circular keyboard buffer.  Characters are
 * removed from the keyboard buffer one at a time.  Each is checked to
 * see if it is a function key used by the kernel.  If so, the character
 * is actioned and not passed on; otherwise the character is passed on to
 * in_process().
 */
static void kb_read(tty_t *tp)
{
    struct kb_s *kb;
    int scode;
    char ch;
    
    kb = kb_addr(tp);
    
    lock();
    while (kb->icount > 0) {
	scode = *kb->itail++;			/* take one key scan code */
	if (kb->itail == kb->ibuf + KB_IN_BYTES) kb->itail = kb->ibuf;
	kb->icount--;

	/*
	 * Is it a function keys used for a debug dump?
	 */
	if (func_key(tp, scode)) continue;

	/*
	 * It's not a function key.  in_process calls lock, so go into it
	 * "unlocked" as otherwise we'll get a recursive lock error.
	 */
	ch = scode;
	unlock();
	(void) in_process(tp, &ch, 1);
	lock();
    }
    unlock();
}


/*
 * Function: func_key
 * Parameters: tp - terminal line the input was received on
 *             scode - scan code for key received.
 * Returns: TRUE if scode should be passed on as user input, FALSE if
 *          it has been consumed by the kernel
 * 
 * Some key-strokes are intercepted by the kernel for its own purposes.
 * At present two codes are intercepted.  Code 0x1f (^_) is used to
 * cause an emegency panic when the system is not responding (panic
 * generates various pieces of debugging info).  The other code is
 * MLOGIN_HANGUP, which us sent by mlogin when it is about to exit.
 */
static int func_key(tty_t *tp, int scode)
{
    switch(scode) {
    case 0x1f: /* ^_, I'm desperate---ignored unless typed on the console */
	if (tty_line(tp) != 0) return FALSE;
	panic("TERMINAL PANIC", NO_NUM);
	break;

    case MLOGIN_HANGUP:
	hangup(tty_line(tp));
	break;

    default:
	return FALSE;
    }
    return TRUE;
}


/*==========================================================================*
 *			     reboot section    		  	            *
 *==========================================================================*/

/*
 * Function: wreboot
 * Parameter: how - unused in smx
 *
 * Terminates smx by exiting.
 */
void wreboot(int how)
{
    printf("Exiting smx\n");
    SunOS(SYS_exit, 0);
    /*NOTREACHED*/
}
