static char rcsid[] = "$Id: sunsigio.c,v 1.1 1996/07/09 19:42:47 paul Exp $";

/*
 * This file contains support for handling SunOS SIGIO signals.  A SIGIO
 * signal is sent whenever input is available on a descriptor that has had
 * SIGIO delivery enabled for it (this enabling is not carried out by these
 * functions).  There are two entry points:
 *    - sunio_install_handler - Called to record an association between
 *                              a SunOS descriptor and a handler in the
 *                              smx handler.
 *    - sunio_interrupt - a SIGIO has occurred.  Check all descriptors
 *                        registered with us, and call handler that input
 *                        is available for.
 */
 
#include "kernel.h"
#include <sun/syscall.h>

#define NUM_CALLBACKS 10

/*
 * Call back functions registered with us.
 */
typedef struct callback {
   void (*handler)(int fd);
} callback_t;

callback_t callbacks[NUM_CALLBACKS];
static int num_cbs = 0;

/*
 * POLLIN and the pollfd struct must agree with /usr/include/sys/poll.h
 * We supply fds to polls(2) to see which descriptors have input available
 * on them.
 */
#define POLLIN 0x1

static struct pollfd {
    int fd;          /* file descriptor */
    short events;    /* requested events */
    short revents;   /* returned events */
} fds[NUM_CALLBACKS];


/*
 * Local functions
 */
static void check_for_input(void);
static int check_desc(int cb);


/*
 * Function: sunio_install_handler
 * Parameters: fd - descriptor to install handler for
 *             handler - SIGIO handler for fd
 *
 * Record the fd/handler association, and include fd in the array supplied
 * to poll.
 */
void sunio_install_handler(int fd, void (*handler)(int fd))
{
    if (num_cbs == NUM_CALLBACKS) {
	panic("sunio_install_handler: number of callbacks exceeds array size",
	      NO_NUM);
    }
    callbacks[num_cbs].handler = handler;
    fds[num_cbs].fd = fd;
    fds[num_cbs].events = POLLIN;
    num_cbs++;
}

 
/*
 * Function: sunio_interrupt
 *
 * The function is called upon receiving a SunOS SIGIO signal.  It checks
 * all registered fd's for input, and calls handlers for those fd's that
 * have input available.
 */
void sunio_interrupt(void)
{
    int poll;

    lock();

    /*
     * Poll all descriptors for input
     */
    check_for_input();
    for (poll = 0; poll < num_cbs; poll++) {
	if (check_desc(poll)) {
	    (*callbacks[poll].handler)(fds[poll].fd);
	}
    }
    
    unlock();
}


/*
 * Function: check_for_input
 * 
 * This function calls poll to initialise the revent fields of fds in 
 * preparation for subsequent calls to check_desc.
 */
static void check_for_input(void)
{
    int i;

    for (i = 0; i < num_cbs; i++) {
	fds[i].revents = 0;
    }

    if (SunOS(SYS_poll, &fds, num_cbs, 0) == -1) {
	panic("poll failed", NO_NUM);
    }
}


/*
 * Function: check_desc
 * Parameter: cb - index into callback info
 * Returns: true if input is available for callback cb, false otherwise
 */
static int check_desc(int cb)
{
    return fds[cb].revents & POLLIN == POLLIN;
}


