/* 
 * The data segment grows upward and the stack grows downward.  The
 * procedures in this file deal with the growth of the data and stack segments.
 *
 * The entry points into this file are:
 *   do_brk:	   BRK/SBRK system calls to grow or shrink the data segment
 *   adjust_stack: make the stack segment extend down to the specified sp.
 *   size_ok:	   see if the segment sizes are feasible
 *   stack_fault:  grow the stack segment
 */

#include "mm.h"
#include <signal.h>
#include "mproc.h"
#include "param.h"

#define isuserp(p) ((p) >= &mproc[INIT_PROC_NR])

/*===========================================================================*
 *				do_brk  				     *
 *===========================================================================*/
PUBLIC int do_brk()
{
/* Perform the brk(addr) system call.
 *
 * The parameter, 'addr' is the new virtual address in D space.
 */

  register struct mproc *rmp;
  vir_bytes v;
  vir_clicks new_clicks;
  phys_bytes new_dataseg;

  rmp = mp;
  v = (vir_bytes) addr;
  new_clicks = (vir_clicks) ( ((long) v + CLICK_SIZE - 1) >> CLICK_SHIFT);
  if (new_clicks < rmp->mp_seg[D].mem_vir ||
      new_clicks >= rmp->mp_seg[S].mem_vir) {
	res_ptr = (char *) -1;
	return(ENOMEM);
  }
  new_clicks -= rmp->mp_seg[D].mem_vir;
  if (new_clicks <= rmp->mp_seg[D].mem_len) {
        /*
	 * We already have enough memory allocated just return.  Should
	 * we shrink the data segment if we can?
	 */
        res_ptr = addr;
        return(OK);
  }
  
  if (!isuserp(rmp)) {
        return(ENOMEM);   /* Cannot expand system processes */
  }
    
/*printf("extra data clicks = %d\n", new_clicks - rmp->mp_seg[D].mem_len);*/

  /*
   * Copy the current data segment into into the new (larger) memory area.
   */
  if ((new_dataseg = alloc_mem(new_clicks)) == NO_MEM) {
        res_ptr = (char *) -1;
	return(ENOMEM);
  }
  sys_copy(ABS, 0, rmp->mp_seg[D].mem_phys << CLICK_SHIFT, ABS, 0,
	   new_dataseg << CLICK_SHIFT, rmp->mp_seg[D].mem_len << CLICK_SHIFT);
  zero_mem((new_dataseg + rmp->mp_seg[D].mem_len) << CLICK_SHIFT,
	   (new_clicks - rmp->mp_seg[D].mem_len) << CLICK_SHIFT);
  free_mem(rmp->mp_seg[D].mem_phys, rmp->mp_seg[D].mem_len);

  /*
   * Fix up the memory map and report it to the kernel
   */
  rmp->mp_seg[D].mem_phys = new_dataseg;
  rmp->mp_seg[D].mem_len = new_clicks;
  sys_newmap(who, rmp->mp_seg);

  res_ptr = addr;
  return(OK);
}


/*===========================================================================*
 *				adjust_stack				     *
 *===========================================================================*/
PUBLIC int adjust_stack(rmp, sp)
register struct mproc *rmp;	/* whose memory is being adjusted? */
vir_bytes sp;			/* new value of sp */
{
/* Make sure that the allocated stack contains sp.  If not, extend it.
 * Memory is never freed.
 */

  register struct mem_map *mem_sp;
  vir_clicks new_clicks, extra_clicks;
  phys_bytes new_stackseg;

  mem_sp = &rmp->mp_seg[S];	/* pointer to stack segment map */
  new_clicks =  mem_sp->mem_vir + mem_sp->mem_len -
      (vir_clicks) ( ((long) sp) >> CLICK_SHIFT);
  if (new_clicks < 1) {
	return(ENOMEM);
  }
  extra_clicks = new_clicks - mem_sp->mem_len;
  if (mem_sp->mem_vir - extra_clicks <
      rmp->mp_seg[D].mem_vir + rmp->mp_seg[D].mem_len) {
          return(ENOMEM);     /* Cannot expand stack into data segment */ 
  }      
  if (extra_clicks < 1) {
        /*
	 * We already have enough memory allocated just return.  Should
	 * we shrink the stack segment if we can?
	 */
        return(OK);
  }
  
  if (!isuserp(rmp)) {
          return(ENOMEM);   /* Cannot expand system processes */ 
  }

/*printf("extra stack clicks = %d\n", extra_clicks);*/

  if ((new_stackseg = alloc_mem(new_clicks)) == NO_MEM) {
	return(ENOMEM);
  }
  /*
   * Copy the existing stack into the top of the new area (the extra space
   * is needed at the bottom/
   */
  sys_copy(ABS, 0, mem_sp->mem_phys << CLICK_SHIFT,
	   ABS, 0, (new_stackseg + extra_clicks) << CLICK_SHIFT, 
	   mem_sp->mem_len << CLICK_SHIFT);
  zero_mem(new_stackseg << CLICK_SHIFT, extra_clicks << CLICK_SHIFT);
  free_mem(mem_sp->mem_phys, mem_sp->mem_len);

  
  /*
   * Fix up the memory map and report it to the kernel.  Note that the
   * new stack is mapped at a lower virtual address to keep the addresses
   * of things currently in the stack the same.
   */
  mem_sp->mem_phys = new_stackseg;
  mem_sp->mem_len = new_clicks;
  mem_sp->mem_vir -= extra_clicks;
  sys_newmap(rmp - mproc, rmp->mp_seg);

  return(OK);
}


/*===========================================================================*
 *				size_ok  				     *
 *===========================================================================*/
PUBLIC int size_ok(file_type, tc, dc, sc, dvir, s_vir)
int file_type;			/* SEPARATE or 0 */
vir_clicks tc;			/* text size in clicks */
vir_clicks dc;			/* data size in clicks */
vir_clicks sc;			/* stack size in clicks */
vir_clicks dvir;		/* virtual address for start of data seg */
vir_clicks s_vir;		/* virtual address for start of stack seg */
{
/* Check to see if the sizes are feasible and enough segmentation registers
 * exist.  On a machine with eight 8K pages, text, data, stack sizes of
 * (32K, 16K, 16K) will fit, but (33K, 17K, 13K) will not, even though the
 * former is bigger (64K) than the latter (63K).  Even on the 8088 this test
 * is needed, since the data and stack may not exceed 4096 clicks.
 */

#if (PAGE_SIZE > 1)
  int pt, pd, ps;		/* segment sizes in pages */

  pt = ( (tc << CLICK_SHIFT) + PAGE_SIZE - 1)/PAGE_SIZE;
  pd = ( (dc << CLICK_SHIFT) + PAGE_SIZE - 1)/PAGE_SIZE;
  ps = ( (sc << CLICK_SHIFT) + PAGE_SIZE - 1)/PAGE_SIZE;

  if (file_type == SEPARATE) {
	if (pt > MAX_PAGES || pd + ps > MAX_PAGES) return(ENOMEM);
  } else {
	if (pt + pd + ps > MAX_PAGES) return(ENOMEM);
  }
#endif

  if (dvir + dc > s_vir) return(ENOMEM);

  return(OK);
}


/*===========================================================================*
 *				stack_fault  				     *
 *===========================================================================*/
PUBLIC void stack_fault(proc_nr)
int proc_nr;			/* tells who got the stack fault */
{
/* Handle a stack fault by growing the stack segment until sp is inside of it. 
 * If this is impossible because data segment is in the way, kill the process.
 */

  register struct mproc *rmp;
  int r;
  vir_bytes new_sp;

  rmp = &mproc[proc_nr];
  sys_getsp(proc_nr, &new_sp);

  r = adjust_stack(rmp, new_sp);
  if (r == OK) return;

printf("Can't expand stack---killing process\n");
  /* Stack has bumped into data segment.  Kill the process. */
  sigdelset(&rmp->mp_catch, SIGSEGV);	/* don't catch this signal */
  sig_proc(rmp, SIGSEGV);	/* terminate process */
}
