static char rcsid[] = "$Id: IN.c,v 1.1 1996/03/20 21:38:14 paul Exp $";

/*
 * Change history:
 *
 * Who  Date        What
 * ======================================================================
 * PJA  May 91     Initial coding
 *
 * PJA  19/6/91    Added the actual_mask parameter to log_readIN
 * 
 * PJA  20/6/91    PERF_SVCRPC_WAIT events which are the last event in a
 *                 sub-task are no longer included in the sub_task_ends,
 *                 as this event irepresents the end of the sub-task, and
 *                 should not be linked to a join vertex.
 *
 * PJA  15/7/91    Changed .h file inclusion to allow sun 4/4.1.1 compiles.
 *
 * PJA  21/10/94   Converted to Amoeba IN software, based on: 
 *                     sccsid[] = "@(#)IN.c 1.3 94/09/12 Paul Ashton";
 */

/*
 * The main entry point for this file is log_readIN, which reads an interaction
 * network from a task log file, and loads it into a directed acyclic graph 
 * of INnode structures.  There are also a couple of graph-maintenance
 * entry points.  log_freeIN frees up all storage associated with an 
 * interaction network graph, and log_in_visited_reset resets all of the
 * visited fields of INnode structures in the specified graph.
 *
 * The remaining three entry points are related to manipulation (symbol
 * (add/find/delete) of symbol tables where identification is based on
 * sub-task numbers.  These functions are used by log_readIN (and 
 * associated functions).  They are entry points because they are also
 * needed by other .c files (the log_stn_sym_* functions should
 * really be in their own .c file).
 */
 
#include <stdio.h>

#include <stdlib.h>
#include <unistd.h>
#include <memory.h>
#include <assert.h>

#include "logging.h"

#include "logfiles.h"
#include "symlib.h"
#include "IN.h"


static INnode_t *log_readINevents(FILE *lp, symptr sub_task_ends,
				  symptr fork_table);
static int log_readINnode(FILE *lp, INnode_t **newp);

static char *stntoa(int stn);

static void log_addjoinlinks(INnode_t **ip, symptr sub_task_ends, 
			    symptr fork_table);
static void log_trim(INnode_t **ip);



/*
 * Function: log_readIN
 * Parameters: filename - name of task log file
 * Returns: a pointer to a properly formed interaction network if 
 *          successful, 0 otherwise.
 *
 * Reads all of the events from the task log file filename into a properly
 * formed interaction network.  It is assumed that the logfile has been
 * produced by insplit, so that all events in the same sub-task appear
 * together, and in increasing time order, and that each sub-task chain
 * (other than the first) appears sometime after the fork event record
 * recording the beginning of the sub-task.
 *
 * The basic algorithm is as follows.  Events are read in one by one.  If
 * an event is in the same sub-task as the previous event then it is
 * simply linked onto the previous event.  If it is in a different sub-task,
 * then it is the first event in a new sub-task.  The event is either from the 
 * first sub-task (which is inserted as the first event in an IN), or
 * it isn't (in which case it is attached to the appropriate fork).
 * Note that pointers to fork type events are maintained in the fork_table
 * symbol table, which records <new sub-task number, pointer to first event>
 * pairs. 
 *
 * Once all of the event records have been read, additional links to join
 * vertices are added.  Doing this requires knowing, for each sub-task,
 * what its last event is.  These are recorded in the sub_task_end
 * symbol table.
 */

INnode_t *
log_readIN(char *filename)
{
    INnode_t *ip = 0;
    symptr sub_task_ends = 0;
    symptr fork_table = 0;
    FILE *log_file;


    if ((log_file = fopen(filename, "r")) == 0) {
	return 0;
    }

    sub_task_ends = sym_create_table();
    fork_table = sym_create_table();
    ip = log_readINevents(log_file, sub_task_ends, fork_table);
    fclose(log_file);

    /*
     * Now add the second edge for each join type record.  This sets the
     * visited flag in each node, so after adding the edges visited
     * must be reset.
     */
    log_addjoinlinks(&ip, sub_task_ends, fork_table);
    log_IN_visited_reset(ip);

    log_trim(&ip);
    log_IN_visited_reset(ip);
    sym_delete_table(sub_task_ends, 0);
    sym_delete_table(fork_table, 0);
    return ip;
}


/*
 * Have a check such that the total events in the tree = the number of
 * events read from the logfile?
 */

/*
 * Function: log_readINevents
 * Parameters: lp - open log file to read from
 *             sub_task_ends - symbol table to add <STN name, last node
 *                             in sub-task> pairs to. The calling routine
 *                             is expected to create this symbol table
 *                             and fork_table.
 *             fork_table - symbol table to add <forked stn, fork node ptr>
 *                          pairs to.
 * Returns: pointer to interaction network constructed from events
 *          in logfile lp if successful, 0 otherwise.
 *
 * Events are read in one by one.  If an event is in the same sub-task
 * as the previous event then it is simply linked onto the previous
 * event.  If it is in a different sub-task, then it is the first event
 * in a new sub-task then it is either the first sub-task (which is
 * inserted as the first event in an IN),  or it isn't (in which case
 * it is attached to the appropriate fork).
 */

static INnode_t *
log_readINevents(FILE *log_file, symptr sub_task_ends, symptr fork_table)
{
    INnode_t *ip = 0;
    INnode_t *newp, *prevp=0;

    while (log_readINnode(log_file, &newp)) {
	/*
	 * If this event is a fork event then add the forked stn and
	 * a pointer to the innode to the fork symbol table.
	 */
	if (newp->evtype == frk) {
	    log_stn_sym_add(fork_table, newp->ev.e_other_stn, newp);
	}

	if (prevp == 0) {
	    /*
	     * source vertex
	     */
	    ip = newp;
	} else if (newp->ev.e_stn == prevp->ev.e_stn) {
	    /*
	     * Continuation of sub-task chain -
	     * just link it to the previous event.
	     */
	    prevp->next = newp;
	    newp->prev = prevp;
	} else {
	    /*
	     * The end of the previous sub-task, and the
	     * beginning of the next.  Add a pointer to the end
	     * of the previous sub-task to the sub_task_ends
	     * symbol table, and link the new sub-task into
	     * the appropriate place in the tree.
	     *
	     * [Comment from SunOS version, but should do this
	     * once an equivalent event is recorded].
	     * Note that if the last event in a sub-task
	     * as a PERF_SVCRPC_WAIT then we don't record
	     * it in sub_task_ends as it is the end of the
	     * sub-task.
	     */
	    INnode_t *forkp;
						       
/* This test will do for Amoeba for the timebeing. 
	    if (prevp->ev->event != EV_RPC_REP_SEND) { */
		log_stn_sym_add(sub_task_ends, prevp->ev.e_stn, prevp);
/*	    }
	    */
	    forkp = log_stn_sym_find(fork_table, newp->ev.e_stn, 1);
	    forkp->fork = newp;
	    newp->prev = forkp;
	    log_stn_sym_delete(fork_table, newp->ev.e_stn);
	}
		
	/*
	 * Whatever happened above, the new event record is
	 * now pointed to by one pointer.
	 */
	newp->ref_cnt = 1;
		
	prevp = newp;
    }
		
    if (prevp != 0) {
	/*
	 * The final sub-task does not have its final event
	 * added to sub_task_ends in the while loop - do it here.
	 */
	log_stn_sym_add(sub_task_ends, prevp->ev.e_stn, prevp);
    }
    
    return ip;
}


/*
 * Function: log_readINnode
 * Parameters: lp - open log file information
 *             newp - pointer to variable to set point to new INnode struct
 * Returns: 0 if OK, 1 if EOF, 2 if read error
 *
 * Reads the next event, and stores thes results in a new INnode struct.
 */

static int
log_readINnode(FILE *log_file, INnode_t **newp)
{
    INnode_t *ip;
    static int seqnumber = 0;
    struct ev_record ev;
    void *rest;
    
    if (!log_read_event(log_file, &ev, &rest)) {
	return 0;
    }

    ip = malloc(sizeof(*ip));
    assert(ip);
    ip->ev = ev;
    ip->rest = rest;
    ip->evtype = log_event_type(ip->ev.e_event);
    ip->next = ip->fork = ip->prev = ip->join = 0;
    ip->visited = 0;
/*    ip->is_on_CP = CP_NO; */
    ip->ref_cnt = 0;
    ip->user_hook = 0;
    ip->seq_no = seqnumber++;
    assert(ip->evtype != notype);

    *newp = ip;
    return 1;
}


/*
 * Function: log_addjoinlinks
 * Parameters: ip - pointer to an interaction network pointer
 *             sub_task_ends - a symbol table containing
 *                             <STN, ptr to last event in sub-task> pairs.
 *             fork_table - a symbol table containing
 * 			    <forked STN, forking event ptr> pairs.
 * Returns: 0 if OK, -1 if error.
 *
 * Adds all of the join links.  The Interaction network in its current
 * form is a tree because of the way in which it has been constructed thus
 * far - sub-task chains connected to the fork events that created them.
 * The tree is traversed, and links are added during the traversal.  The
 * visited field of the node record is used during the traversal - it
 * is assumed that this field is 0 for all events in the IN when this
 * function is called!!!  During the traversal one type of join
 * links is added.
 * 1. For each join event, if the last event of the sub-task of the
 *    previous stn is in the sub_task_ends symbol table then this
 *    last event has its next ptr set to point at the join event.
 *    If not then the fork_table is checked to see whether the sub-task
 *    consists of no events in its own right - is an edge straight from
 *    a fork vertex to a join vertex.  If this is the case then the 
 *    fork pointer of the fork node is set to point to the join node.
 */

static void
log_addjoinlinks(INnode_t **ip, symptr sub_task_ends, symptr fork_table)
{
    INnode_t *localp = *ip;

    if (localp == 0 || localp->visited) {
	return;
    }

    localp->visited = 1;

    /*
     * If this is a join event then see if we have a sub-task end to
     * link it to.  Note that in some cases the forkjoin_stn is the same
     * as the event's stn.  This can occur because of (for example)
     * aborted RPCs.  In these cases we don't want to add a join link
     * as we'll create a cycle in the IN!
     */
    if (localp->evtype == join &&
	localp->ev.e_stn != localp->ev.e_other_stn) {
	INnode_t *end_event;

	end_event = log_stn_sym_find(sub_task_ends,
				     localp->ev.e_other_stn, 0);
	if (end_event != 0) {
	    /*
	     * This is supposed to be the last sub-task
	     * in the chain.
	     */
	    assert(end_event->next == 0 && localp->join == 0);

	    end_event->next = localp;
	    localp->join = end_event;
	    localp->ref_cnt++;
	    log_stn_sym_delete(sub_task_ends, localp->ev.e_other_stn);
	} else {
	    end_event = log_stn_sym_find(fork_table,
					 localp->ev.e_other_stn, 0);
	    if (end_event  != 0) {
		/*
		 * there are supposed to be no events in the
		 * chain
		 */
		assert(end_event->fork == 0 && localp->join == 0);

		end_event->fork = localp;
		localp->join = end_event;
		localp->ref_cnt++;
	    }
	}
    }

    log_addjoinlinks(&localp->next, sub_task_ends, fork_table);
    if (localp->evtype == frk) {
	log_addjoinlinks(&localp->fork, sub_task_ends, fork_table);
    }	  
}


/*
 * Function: log_trim
 * Parameter: ip - pointer to an innode pointer
 * Returns: nothing
 *
 * Trims off EV_SINK events related to the next interaction.
 */

static void
log_trim(INnode_t **ip)
{
    INnode_t *localp = *ip;

    if (localp == 0 || localp->visited) {
	return;
    }

    localp->visited = 1;

#ifdef notdef
    /*
     * This is what was done in the SunOS version.  We'll
     * probably want to do something similar after adding some more probes
     * to Amoeba.
     */
    if (localp->ev->event == PERF_SETRQ && localp->next &&
	localp->next->ev->event == PERF_RESUME &&
	(localp->next->next == 0 ||
	 localp->next->next->ev->event == EV_SINK)) {
	assert(localp->next->next == 0 ||
	       localp->next->next->next == 0 &&
	       localp->next->next->ev->event == EV_SINK);
#endif

    /*
     * Currently, EV_SINK's are generated at the time of the input
     * event of the next interaction.  They don't beloing to this interactio
     * so delete them.
     */
    if (localp->ev.e_event == EV_SINK) {
	*ip = 0;       /* disconnect the chain */
	log_freeIN(localp, 0);
    } else {
	log_trim(&localp->next);
	if (localp->evtype == frk) {
	    log_trim(&localp->fork);
	}	  
    }
}


/*
 * Function: log_freeIN
 * Parameter: ip - interaction network whose storage we want to free
 * Returns: nothing
 * Frees all storage allocated to an interaction network.
 */

void
log_freeIN(INnode_t *ip, free_spare_func tidyup)
{
    if (ip == 0) {
	return;
    }

    if (--ip->ref_cnt > 0) {
	/*
	 * Still more references to this node
	 */
	return;
    }

    log_freeIN(ip->next, tidyup);
    if (ip->evtype == frk) {
	log_freeIN(ip->fork, tidyup);
    }
    if (tidyup && ip->user_hook) {
	(*tidyup)(ip->user_hook);
    }

    free(ip->rest);
    free(ip);
}


/*
 * Function: log_IN_visited_reset
 * Parameter: ip - a pointer to an interaction network.  ALL OF THE EVENTS
 *                 IN ip ARE ASSUMED TO HAVE VISITED = 1.
 * Returns: nothing
 *
 * Resets all of the visited values to 0 based on the above assumption.
 */

void
log_IN_visited_reset(INnode_t *ip)
{
    if (ip == 0 || !ip->visited) {
	return;
    }

    ip->visited = 0;
    log_IN_visited_reset(ip->next);
    if (ip->evtype == frk) {
	log_IN_visited_reset(ip->fork);
    }
}

/*****************************************************************************/

/*
 * Functions: log_stn_sym_{add,find,delete}
 * 
 * These functions are a set of wrappers for the associated symbol table
 * functions.  They fulfil two purposes:
 * (i) Converting stn's to characters strings which can be used with the
 *     symbol table functions.
 * (ii) Exiting the program if any errors occur.
 */

void
log_stn_sym_add(symptr sym_table, int stn, void *user_data)
{
    char *cp = stntoa(stn);

    (void) sym_add(sym_table, cp, user_data);
    assert(sym_errno == 0);
    free(cp);
}


void *
log_stn_sym_find(symptr sym_table, int stn, int error_fatal)
{
    char *cp = stntoa(stn);
    void *user_data;

    user_data = sym_find(sym_table, cp);
    if (error_fatal) {
	assert(sym_errno == 0);
    }
    free(cp);
    return user_data;
}


void
log_stn_sym_delete(symptr sym_table, int stn)
{
    char *cp = stntoa(stn);

    sym_delete(sym_table, cp, 0);
    assert(sym_errno == 0);
    free(cp);
}

#define CHARS_FOR_STN 8       /* It takes 8 chars to represent an stn in hex */

/*
 * Function: stntoa
 * Parameter: stn - the stn whose character representation is desired
 * Returns: a pointer to a malloc'ed char array containing a null
 *          terminated character string representation of stn.
 *
 * At the moment stn_t is unsigned long, so we just print it as an 8 char hex
 * string.
 */

static char *
stntoa(int stn)
{
    char *retval = malloc(CHARS_FOR_STN+1);

    assert(retval);
    (void) sprintf(retval, "%08lx", stn);
    return retval;
}


#if 0

/* Complete this function if it is required */

/* Put in perflib.h */

/*
 * defines for log_findINstn()
 */
#define EVENT_STN 0
#define FORK_STN  1

struct innode *log_findINstn();


/* The function itself */

/*
 * Function: log_findINstn
 * Parameters: ip - interaction network to search in
 *             stn - stn to search for
 *             stn_to_use - if EVENT_STN compare stn with pr.stn
 *                          if FORK_STN compare with fork_stn
 *
 * Searches the interaction network pointed to by ip for the specified
 * stn either in the pr.stn field or the fork_stn field.  If the pr.stn
 * field is being searched for then the a pointer to the first event
 * of sub-task stn is returned.  If fork_stn is being searched for then
 * a pointer to the event which has its fork_stn equal to stn is returned.
 *
 * Note that because the graph is acyclic then this algorithm wil terminate OK
 */

        struct innode *
log_findINstn(ip, stn, stn_to_use)
        struct innode *ip;
        ipid_t stn;
        int stn_to_use;
{
	struct innode *ptr;

	if (stn_to_use == EVENT_STN && stn == ip->pr.pe_stn) {
		/*
		 * Found it.
		 */
		return ip;
	}

	for (ptr = 
}

#endif
