static char rcsid[] = "$Id: cv.c,v 1.1 1996/03/20 21:32:20 paul Exp $";

/* The program converts a Minix executable file compiled and linked by
 * the SunOS tools to a format suitable to be used within Minix.  The
 * only things that need to be converted are the file's header, the
 * symbol table format and the ordering of the sections within the
 * file. No changes are made to the text, data or relocation sections.
 * An improvement to this utility would be to perform as much
 * relocation as possible at this conversion, rather than leaving
 * it until runtime. Since we must use the '-r'option on the SunOS
 * linker, no relative relocation is performed eg. a relative branch to
 * the previous instruction is not relocated until runtime!
 *
 * Notes:
 * When using SunOS's 'ld' program, the '-r' option must be used.  'cv'
 * is a SunOS utility and must not be run under Minix.  
 */

#include "../../include/minix/config.h"
#include "../../include/minix/const.h"
#include "elf/a.out.h"
#include <stdio.h>
#include <fcntl.h>
#include <unistd.h>
#include <stdlib.h>
#include <limits.h>
#include "cv.h"

int infd, outfd;		/* SunOS and Minix files */
int stack = -1;                 /* +ve if user has specified a stack size */
struct exec shead;		/* SunOS executable file header */
struct exec_mx mhead;		/* Minix executable file header */
int num_undefined = 0;		/* Number of undefined symbols */
char *progname;


/* Read the SunOS header and build an equivalent Minix header. SunOS Minix
 * uses a 48 byte header because of the extra relocation information needed.
 * A guess is made at how much stack space should be allocated for the program
 * being converted.
 */  
static void
read_header(void)
{
    read(infd, &shead, sizeof(struct exec));
    mhead.a_magic[0] = A_MAGIC0;
    mhead.a_magic[1] = A_MAGIC1;
    mhead.a_flags = A_EXEC | A_SEP;
    mhead.a_cpu = A_SUNOS;
    mhead.a_hdrlen = sizeof(mhead);
    mhead.a_unused = 0;
    mhead.a_version = 0;
    mhead.a_text = shead.a_text;
    mhead.a_data = shead.a_data;
    mhead.a_bss = shead.a_bss;
    mhead.a_no_entry = 0;
    if (stack > 0) {    /* user has specified a gap */
	mhead.a_total = mhead.a_data + mhead.a_bss + stack;
    } else {
	mhead.a_total = mhead.a_data + mhead.a_bss + 
   	  (((mhead.a_text / 5) < CLICK_SIZE * 4) ?  CLICK_SIZE * 4 : 
          (mhead.a_text / 5));
    }
    mhead.a_syms = (shead.a_syms / sizeof (struct nlist)) *
      (sizeof(struct nlist_mx));
    mhead.a_trsize = shead.a_trsize;
    mhead.a_drsize = shead.a_drsize;
    mhead.a_tbase = 0;
    mhead.a_dbase = 0;
    write(outfd, &mhead, sizeof(struct exec_mx));
}


/* Given the size of one of the sections within the executable file, the whole
 * segment is copied from the SunOS file to the Minix file with no 
 * modifications.
 */
static void
read_segment(int size)
{
    char loadbuf[1024];
    int this_much;
    
    while (size != 0) {
  	this_much = (size < sizeof(loadbuf) ? size : sizeof(loadbuf));
  	read(infd, loadbuf, this_much);
	write(outfd, loadbuf, this_much);
	size -= this_much;
    }
}


/* This is the most complex part that converts SunOS's symbol and string 
 * tables into Minix's single symbol table. Any undefined symbols are 
 * reported here (since SunOS's 'ld -r' won't do this)
 */
static void
read_symbols(void)
{
    struct nlist *sym_base =	/* The entire SunOS symbol table */
      (struct nlist *)malloc(shead.a_syms);
    char *string_base;		/* Base of the SunOS string table */
    struct nlist_mx out_entry;	/* A single Minix symbol table entry */
    int string_size, cur_sym;
  
    if (sym_base == 0) {
  	fprintf(stderr, "can't malloc symbol size = %ld\n", shead.a_syms);
	exit(1);
    }
  
    /* read the SunOS symbol and string table into memory */  
    read(infd, sym_base, shead.a_syms);
    read(infd, &string_size, sizeof(string_size));

    string_base = (char *)malloc(string_size);
    if (string_base == 0) {
  	fprintf(stderr, "can't malloc string size = %d\n", string_size);
	exit(1);
    }
    read(infd, string_base + 4, string_size - 4);
  
    /* for each symbol, convert between formats */
    cur_sym = 0;
    while (cur_sym != shead.a_syms) {
  	register int count;
	
	/* Copy the name from SunOS string table into Minix's symbol entry */
	for (count = 0; count < 8; count++)
		out_entry.n_name[count] = '\0';
	strncpy(out_entry.n_name, &string_base[sym_base->n_un.n_strx], 8);
	out_entry.n_value = sym_base->n_value;
	
	/* Data segment starts at 0 in Minix, but not SunOS */
	switch ((sym_base->n_type) & N_TYPE){
	case N_DATA:
	case N_BSS:
	case N_COMM:
	    out_entry.n_value -= mhead.a_text;
	    break;
	default:
	    ;
	}
	
	/* Externally defined symbol */
	if ((sym_base->n_type & N_EXT == N_EXT)) {
            out_entry.n_sclass = C_EXT;
	} else {
	    out_entry.n_sclass = C_STAT;
	}
	
	/* This should be done with a switch statement, in case the type
	 * symbols are ever changed, but this is shorter.
	 */
	if ((sym_base->n_type & N_TYPE) == N_COMM) {
	    out_entry.n_sclass |= N_COMM_MX;
	} else {
	    out_entry.n_sclass |= ((sym_base->n_type & N_TYPE) >> 1);
	}

	/* Write out the finished entry */
	write(outfd, &out_entry, sizeof(out_entry));
	
	/* Was this symbol undefined ? */
	if ((out_entry.n_sclass & N_SECT_MX) == N_UNDF_MX) {
	    write(2, "Undefined symbol: ", 18);
	    write(2, out_entry.n_name, 8);
	    write(2, "\n", 1);
	    num_undefined++;
	}
	
	sym_base++;
	cur_sym += sizeof(struct nlist);
    }
}


static void
usage(void)
{
    fprintf(stderr, "Usage %s: sunfile minixfile [stack]\n", progname);
    exit(1);
}


int
main(int argc, char *argv[])
{
    progname = argv[0];

    if (argc < 3 || argc > 4) {
        usage();
    }

    if ((infd = open(argv[1], O_RDONLY)) == -1) {
	fprintf(stderr, "%s: can't open input file %s\n", argv[0], argv[1]);
	exit(1);
    }

    if ((outfd = open(argv[2], O_CREAT | O_WRONLY | O_TRUNC, 0700)) == -1) {
	fprintf(stderr, "%s: can't create output file %s\n", argv[0], 
		argv[2]);
	exit(1);
    }

    /*
     * The following piece of code was taken from the minix install command
     * where it processes a specified stack size.
     */
    if (argc == 4) {
        char *end, *p;
	int num;

	p = argv[3];
        stack = strtol(p, &end, 0);
	if (end == p || stack < 0) usage();
	p = end;
	while (*p != 0) {
	    switch (*p++) {
	    case 'm':
	    case 'M': num = 1024 * 1024; break;
	    case 'k':
	    case 'K': num = 1024; break;
	    case 'w':
	    case 'W': num = 4; break;
	    case 'b':
	    case 'B': num = 1; break;
	    default: usage();
	    }
	    if (stack > LONG_MAX / num) usage();
	    stack *= num;
	}
    }
  
    read_header();
    read_segment(mhead.a_text);
    read_segment(mhead.a_data);
    lseek(infd, shead.a_trsize + shead.a_drsize, SEEK_CUR);
    read_symbols();
    lseek(infd, shead.a_text + shead.a_data + sizeof(struct exec), SEEK_SET);
    read_segment(shead.a_trsize);
    read_segment(shead.a_drsize);
    close(infd);
    close(outfd);
    return(num_undefined != 0);
}  
