/*
 * sol2sun
 *
 * This Solaris utility program converts a Solaris object file (in ELF 
 * format) into a SunOS object file. Note that this only works for .o
 * files generated by a C compiler or with 'ld -r'
 */

#include <stdio.h>
#include <elf.h>
#include <libelf.h>
#include <fcntl.h>
#include <stdlib.h>

#include "a.out.h"

static char rcsid[] = "$Id: sol2sun.c,v 1.1 1996/03/20 21:43:34 paul Exp $";

#define align(v, base) (((v) + (base) - 1) - (((v) + (base) - 1) % (base)))

/* The list of section types that can be handled */

#define TEXT 0
#define DATA 1
#define RODATA 2
#define BSS 3
#define TRELOC 4
#define DRELOC 5
#define RODRELOC 6
#define SYM 7
#define STR 8

#define NUM_AREAS 9

/* forward definitions and prototypes */
void do_scn();
void write_file();
int write(), close(), unlink();

/* Various descriptors for use with the ELF libraries */

Elf *prog, *file;
Elf_Scn *scn;
Elf_Data *data;
Elf32_Sym *syms;

int index, secnum;
char *strings;
int debug = 0;
int ndxs[NUM_AREAS];			/* the section index for each section */
int sizes[NUM_AREAS];			/* the size of each section */
struct exec header;			/* the SunOS header being built */


/* for translating Solaris relocation codes to SunOS relocation codes */
char translate[] = {
  -1,             /* none */
  RELOC_8,
  RELOC_16,
  RELOC_32,
  RELOC_DISP8,
  RELOC_DISP16,
  RELOC_DISP32,
  RELOC_WDISP30,
  RELOC_WDISP22,
  RELOC_HI22,
  RELOC_22,
  RELOC_13,
  RELOC_LO10,
  RELOC_BASE10,
  RELOC_BASE13,
  RELOC_BASE22,
  RELOC_PC10,
  RELOC_PC22,
  -1,
  -1,
  RELOC_GLOB_DAT,
  RELOC_JMP_SLOT,
  RELOC_RELATIVE,
  -1,
  -1,
};



/*===========================================================================*
 *                                   do_scn                                  * 
 *===========================================================================*/
void do_scn(shdr)
     Elf32_Shdr *shdr;
{
  /*
   * Scan though each of the sections of the ELF file and determine its type
   * and size. The arrays 'ndx' and 'sizes' are used for recording this
   * information for later use.
   */
  
  char *name;
  char *title = "";
  
  /* get the name of this section */
  name = elf_strptr(file, index, (size_t)shdr->sh_name);
  if (debug) printf("'%s'\n", name? name: "(null)");
  
  /* get the data for this section */
  data = 0;
  if ((data = elf_getdata(scn, data)) == 0 || data->d_size == 0){
    if (debug) fprintf(stderr,"No ELF data\n\n");
  } else {
    if (debug)
      printf("Data buf = %x\nsize = %x\n\n", (unsigned)data->d_buf, data->d_size);
  }
  
  /* for each of the sections, record the appropriate size etc */
  if (!strcmp(".text", name)){
    title = "Text segment\n";
    ndxs[TEXT] = secnum;
    sizes[TEXT] = data->d_size;
    
  } else if (!strcmp(".rodata", name)){
    title = "Read only data segment\n";
    ndxs[RODATA] = secnum;
    sizes[RODATA] = data->d_size;
    
  } else if (!strcmp(".data", name)){
    title = "Data segment\n";
    ndxs[DATA] = secnum;
    sizes[DATA] = data->d_size;	
    
  } else if (!strcmp(".bss", name)){
    title = "BSS segment\n";
    ndxs[BSS] = secnum;
    sizes[BSS] = data->d_size;
    
  } else if (!strcmp(".symtab", name)){
    title = "Symbol table\n";
    ndxs[SYM] = secnum;
    sizes[SYM] = data->d_size;
    syms = (Elf32_Sym *)malloc(data->d_size);
    memcpy(syms, data->d_buf, data->d_size);
    
  } else if (!strcmp(".strtab", name)){
    title = "String table\n";
    ndxs[STR] = secnum;
    sizes[STR] = data->d_size;
    strings = (char *)malloc(data->d_size);
    memcpy(strings, data->d_buf, data->d_size);
    
  } else if (!strcmp(".shstrtab", name)){
    title = "Section header string table\n";
    
  } else if (!strcmp(".rela.text", name)){
    title = "Text relocation\n";
    ndxs[TRELOC] = secnum;
    sizes[TRELOC] = data->d_size;
    
  } else if (!strcmp(".rela.rodata", name)){
    title = "Read only Data relocation\n";
    ndxs[RODRELOC] = secnum;
    sizes[RODRELOC] = data->d_size;
    
  } else if (!strcmp(".rela.data", name)){
    title = "Data relocation\n";
    ndxs[DRELOC] = secnum;	
    sizes[DRELOC] = data->d_size;
    
  } else if (!strcmp(".comment", name)){
    title = "Comment\n";
    
  } else title = "Unknown section type\n";
  
  if (debug)
    printf("%s\n\n", title);
  
}



/*===========================================================================*
 *                                   read_scn                                * 
 *===========================================================================*/
int read_scn(num)
     int num;
{
  /* Read a particular section from the file. ELF translation is possible */
  scn = elf_getscn(prog, ndxs[num]);
  
  data = 0;
  if ((data = elf_getdata(scn, data)) == 0 || data->d_size == 0)
    return -1;
  return 0;
}



/*===========================================================================*
 *                                   panic                                   * 
 *===========================================================================*/
int read_raw_scn(num)
     int num;
{
  /* Read a particular section from the file (raw data only) */
  scn = elf_getscn(prog, ndxs[num]);
  
  data = 0;
  if ((data = elf_rawdata(scn, data)) == 0 || data->d_size == 0)
    return -1;
  return 0;
}



/*===========================================================================*
 *                                   conv_type                               * 
 *===========================================================================*/

int conv_type(sym, new_value)
     Elf32_Sym sym;
     long *new_value;
{
  /*
   * convert a Solaris section type to a SunOS section type. If necessary,
   * the value within this section may be changed. For example, Solaris's
   * RODATA and DATA sections are merged into one SunOS section
   */

  int result = 0;
  
  *new_value = 0;
  if (sym.st_shndx == 0){
    result = N_UNDF;
  } else if (sym.st_shndx == SHN_ABS){
    result = N_ABS;	    
  } else if (sym.st_shndx == ndxs[TEXT]){
    result = N_TEXT;
  } else if (sym.st_shndx == ndxs[DATA]){
    result = N_DATA;
    *new_value = align(sizes[TEXT], 8);
  } else if (sym.st_shndx == ndxs[BSS]){
    result = N_BSS;
    *new_value = align(sizes[TEXT], 8) + align(sizes[DATA], 8) +
      align(sizes[RODATA], 8);
  } else if (sym.st_shndx == ndxs[RODATA]){
    result = N_DATA;
    *new_value = align(sizes[TEXT], 8) + align(sizes[DATA], 8);
  } else if (sym.st_name != 0){
    fprintf(stderr, "Bad segment type in index\n");
    exit(1);
  }
  
  return result;
}



/*===========================================================================*
 *                                   conv_sym                                * 
 *===========================================================================*/
void conv_sym(fd)
     int fd;
{
  /* convert a Solaris symbol table entry into a SunOS table entry */

  int count, numsyms, extra;
  Elf32_Sym *solsym;
  struct nlist sunsym;
  
  if (read_scn(SYM)){
    fprintf(stderr,"Error: no symbol information\n");
    exit(1);
  }
  
  numsyms = (sizes[SYM] / sizeof(Elf32_Sym)) - 1;
  solsym = (Elf32_Sym *)((char *)(data->d_buf) + sizeof(Elf32_Sym));
  for (count = 0; count < numsyms; count++){
    
    sunsym.n_un.n_strx = solsym->st_name + sizeof(int);
    sunsym.n_type = conv_type(*solsym, &extra);
    sunsym.n_value = solsym->st_value + extra;
    
    switch (ELF32_ST_TYPE(solsym->st_info)){
    case STT_SECTION:
    case STT_FILE:
      sunsym.n_type = N_FN;
      break;
    default:
    }
    
    if (debug){
      printf("Symbol number = %d\n", count);
      printf("Name = %s\n", &strings[solsym->st_name]);
      printf("Value = %ld\n", sunsym.n_value);
      printf("Size = %ld\n", solsym->st_size);
      printf("Index = %d\n", solsym->st_shndx);
      printf("Extra = %d\n", extra);
      printf("\n");	    
    }	
    write(fd, &sunsym, sizeof(sunsym));	
    solsym++;
  }	   
}



/*===========================================================================*
 *                                   conv_reloc                              * 
 *===========================================================================*/
void conv_reloc(seg, fd, incr)
     int seg;
     int fd;
     int incr;
{
  /* convert Solaris relocation information to SunOS relocation information */

  int count, numrels, extra;
  Elf32_Rela *solrela;
  struct reloc_info_sparc sunrela;
  
  if (read_raw_scn(seg)){	/* the is no relocation for this section */
    return;
  }
  
  numrels = sizes[seg] / sizeof(Elf32_Rela);
  if (debug)
    printf("num_rels = %d, size = %d, sizeof = %d\n", numrels, sizes[seg], sizeof(Elf32_Rela));
  solrela = (Elf32_Rela *)((char *)(data->d_buf));

  for (count = 0; count < numrels; count++){
    sunrela.r_address = solrela->r_offset + incr;
    sunrela.r_index = conv_type(syms[ELF32_R_SYM(solrela->r_info)], &extra);
    
    if (sunrela.r_index == 0)
      sunrela.r_extern = 1;
    else
      sunrela.r_extern = 0;
    
    sunrela.r_type = translate[ELF32_R_TYPE(solrela->r_info)];
    if (translate[ELF32_R_TYPE(solrela->r_info)] == -1){
      fprintf(stderr,"Error: Unknown relocation type\n");
      exit(1);
    }

    if (syms[ELF32_R_SYM(solrela->r_info)].st_name != 0)
      sunrela.r_addend = syms[ELF32_R_SYM(solrela->r_info)].st_value
	+ solrela->r_addend;
    else   
      sunrela.r_addend = solrela->r_addend;
    
    sunrela.r_addend += extra;
    
    switch(sunrela.r_type){
    case RELOC_WDISP30:
    case RELOC_WDISP22:
      sunrela.r_addend -= (solrela->r_offset - extra);
      break;
    default:
    } 
    
    if (debug){
      printf("Offset = %ld\n", solrela->r_offset + incr);
      printf("Sym = %ld\n", ELF32_R_SYM(solrela->r_info));
      
      if (syms[ELF32_R_SYM(solrela->r_info)].st_name != 0)
	printf(" = %s\n",
	       &strings[syms[ELF32_R_SYM(solrela->r_info)].st_name]);
      
      printf("Type = %d\n", ELF32_R_TYPE(solrela->r_info));	
      printf("Addend = %ld, sol = %ld\n", sunrela.r_addend, solrela->r_addend);
      
      printf("Index = %d, sol = %d\n", sunrela.r_index,
	     syms[ELF32_R_SYM(solrela->r_info)].st_shndx );
      printf("\n");
    }
    
    write(fd, &sunrela, sizeof(sunrela));
    solrela ++;
    
  }
}



/*===========================================================================*
 *                                   writefile                               * 
 *===========================================================================*/

void write_file(outfile)
     int outfile;
{
  /* 
   * write out the SunOS file (header, text, data, relocation, symbols,
   * strings)
   */

  /* Create the header */
  header.a_dynamic = 0;
  header.a_toolversion = 0;
  header.a_machtype = M_SPARC;
  header.a_magic = NMAGIC;
  header.a_text = align(sizes[TEXT], 8);
  header.a_data = align(sizes[DATA], 8) + align(sizes[RODATA], 8);    
  header.a_bss = sizes[BSS];
  header.a_syms = ((sizes[SYM] / sizeof(Elf32_Sym)) - 1) * sizeof(struct nlist);
  header.a_entry = 0x2000;
  header.a_trsize = (sizes[TRELOC] / sizeof(Elf32_Rela)) 
    * sizeof(struct reloc_info_sparc);
  header.a_drsize = ((sizes[DRELOC] + sizes[RODRELOC]) / sizeof(Elf32_Rela)) 
    * sizeof(struct reloc_info_sparc); 
  write(outfile, &header, sizeof(header));
  

  /* write the text segment */

  if (read_scn(TEXT) == -1){
    fprintf(stderr,"Error: no text information\n");
    exit(1);
  } else {
    write(outfile, data->d_buf, data->d_size);
    if (data->d_size < align(data->d_size, 8)) {
      write(outfile, "\0\0\0\0\0\0\0", align(data->d_size, 8) - data->d_size);
    }
  }    
  
  /* write the data and read only data segments */
  if (read_scn(DATA)){
    fprintf(stderr,"Error: no data information\n");
    exit(1);
  } else {
    write(outfile, data->d_buf, data->d_size);
    if (data->d_size < align(data->d_size, 8)) {
      write(outfile, "\0\0\0\0\0\0\0", align(data->d_size, 8) - data->d_size);
    }
  }
  if (read_scn(RODATA)){
  } else {
    write(outfile, data->d_buf, data->d_size);
    if (data->d_size < align(data->d_size, 8)) {
      write(outfile, "\0\0\0\0\0\0\0", align(data->d_size, 8) - data->d_size);
    }
  }

  /* Read and convert all the relocation information */  
  conv_reloc(TRELOC, outfile, 0, N_TEXT);
  conv_reloc(DRELOC, outfile, 0, N_DATA);
  conv_reloc(RODRELOC, outfile, sizes[DATA], N_DATA);
  
  /* write the symbol table */
  conv_sym(outfile);    
  
  /* write the string table */    
  sizes[STR] += sizeof(int);
  write(outfile, &sizes[STR], sizeof(int));
  write(outfile, strings, sizes[STR] - sizeof(int));
}    



/*===========================================================================*
 *                                   main                                    * 
 *===========================================================================*/
int main(argc, argv)
     int argc;
     char *argv[];
{
  int fd;
  Elf32_Ehdr *ehdr;
  Elf32_Shdr *shdr;
  Elf_Cmd cmd;
  int outfile;
  char *input = "", *output = "";
  
  if (argc != 3 && argc != 4){
    fprintf(stderr, "Usage: %s [-d] solaris-filename sunos-filename\n", argv[0]);
    exit(1);
  }
  
  if (argc == 3){
    input = argv[1];
    output = argv[2];
  } else if (strcmp(argv[1], "-d") == 0){
    debug = 1;
    input = argv[2];
    output = argv[3];
  } else {
    fprintf(stderr, "%s: Bad debug flag\n", argv[0]);
    exit(1);
  }

  /* open the Solaris object file */

  if ((fd = open(input, O_RDONLY)) == -1){
    fprintf(stderr,"Can't open ELF file %s\n", argv[1]);
    exit(1);
  }
  if (elf_version(EV_CURRENT) == EV_NONE)
    {
      fprintf(stderr,"Library out of date: %s\n", argv[1]);
      exit(1);
    }    
  
  /* create the SunOS output file */

  unlink(output);
  outfile = open(output, O_RDWR | O_CREAT, 0666);
  if (outfile == -1){
    fprintf(stderr, "Can't write to %s\n", argv[2]);
    exit(1);
  };
  
  /* Read and get details about each program (normally only 1) */

  cmd = ELF_C_READ;
  file = elf_begin(fd, cmd, (Elf *)0);
  while ((prog = elf_begin(fd, cmd, file)) != 0){
    if ((ehdr = elf32_getehdr(prog)) != 0){
      index = ehdr->e_shstrndx;
      scn = (Elf_Scn *)0;
      secnum = 0;

      /* Read and get details about each section */

      while ((scn = elf_nextscn(prog, scn)) != 0) {
	secnum++;
	shdr = elf32_getshdr(scn);
	do_scn(shdr);
      }

      /* Write the SunOS object file */

      write_file(outfile);

    }

    cmd = elf_next(prog);
    elf_end(prog);
  }
  elf_end(file);
  close(outfile);
  return 0;
}


/*===========================================================================*/








