/* This is a Solaris utility for connecting with existing Minix processes in a way
 * similar to 'rlogin'. This is only a terminal program with a socket 
 * connecting this program and a Minix process for sending a user's keypresses
 * and receiving terminal output. All processing is performed by the Minix
 * process, possibly using someone elses UNIX user i.d.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/time.h>
#include <sys/socket.h>
#include <unistd.h>
#include <fcntl.h>
#include <termios.h>
#include <signal.h>
#include <stropts.h>

#define BUFSIZE 200		/* The most characters read at one time */

static char rcsid[] = "$Id: mlogin.c,v 1.1 1996/03/20 21:32:39 paul Exp $";

int pipeinfd;			/* the pipe connection */
int pipeoutfd;
int termfd;			/* the user's terminal */
struct termios old_t;		/* the save terminal configuration */
int count, ourline;
char disco[] 			/* terminal emulator escape sequence ^Q^U^I^T*/
  = {021, 025, 011, 024};
int discnt = 0;			/* how far through this sequence? */

char lock[BUFSIZE];		/* name of the lock file */
int lockfd, linecount, linemap; /* used for finding a terminal line */



/*===========================================================================*
 *                                   fatal                                   * 
 *===========================================================================*/
static void
fatal(const char *mess)
{
  /* a fatal error has occured */
  
  fprintf(stderr, mess);
  
  /* unlock this terminal line */

  if ((lockfd = open(lock, O_RDWR)) == -1){
    exit(0);	/* Minix exited before mlogin */
  }
  if (lockf(lockfd, F_LOCK, 0) != 0){
    fprintf(stderr, "Can't lock the lock file\n");
    exit(1);
  }

  /* Add this terminal line back into the map */

  read(lockfd, &linecount, sizeof(linecount));
  read(lockfd, &linemap, sizeof(linemap));
  linemap |= (1 << (ourline - 1));
  lseek(lockfd, sizeof(linecount), SEEK_SET);
  write(lockfd, &linemap, sizeof(linemap));

  /* Unlock the file */

  lockf(lockfd, F_ULOCK, 0);
  close(lockfd);

  /* Reset the UNIX keyboard characteristics */
  ioctl(termfd, TCSETS, &old_t);
  close(termfd);
  close(pipeinfd);
  close(pipeoutfd);
  exit(1);
}



/*===========================================================================*
 *                                   term                                    * 
 *===========================================================================*/
static void
term(void)
{
  /* Wait until characters are ready received from either the keyboard or
   * from Minix. Read them and send them on.
   */

  char buffer[BUFSIZE];		/* where the characters are stored */
  char quit = (char)-1;		/* Minix's hangup character */
  int length;			/* how much was read? */
  fd_set reads;
  int max_fd;
  int cnt;
  
  max_fd = termfd > pipeinfd ? termfd : pipeinfd;
  discnt = 0;

  /* forever until death do us part */
  while (1) {
    FD_ZERO(&reads);
    FD_SET(termfd, &reads);
    FD_SET(pipeinfd, &reads);

  
    /* which file descriptors are ready? */
    select(max_fd + 1, &reads, 0, 0, 0);

    /* characters typed on the terminal */
    if (FD_ISSET(termfd, &reads)){
      length = read(termfd, buffer, BUFSIZE);
    
      /* escape sequence? */
      for (cnt = 0; cnt < length; cnt++) {
        if (buffer[cnt] == disco[discnt]) {
          discnt++;
	  if (discnt == sizeof(disco)) {
	    break;
	  }
        } else {
	  discnt = 0;
        }
      }
      if (length <= 0 || discnt == sizeof(disco)){
        write(pipeoutfd, &quit, 1);
        fatal("\n\nReturning to Solaris\n");
      }
      /* pass characters on to Minix */
      write(pipeoutfd, buffer, length);
    }
  
    /* characters to display on the output */
    if (FD_ISSET(pipeinfd, &reads)){
      length = read(pipeinfd, buffer, BUFSIZE);
      if (length <= 0) {
        write(pipeoutfd, &quit, 1);
        fatal("\n\nReturning to Solaris\n");
      }
      write(termfd, buffer, length);
    }
  }
}



/*===========================================================================*
 *                                   main                                    * 
 *===========================================================================*/
int
main(int argc, char *argv[])
{
  struct termios t;		/* terminal setup */
  char host_name[BUFSIZE];	/* host we are connecting to */
  
  /* get the user's terminal and remember it parameters */
  termfd = open("/dev/tty", O_RDWR);
  if (termfd < 0) {
    fprintf(stderr, "Can't get terminal\n");
    exit(1);
  }
  ioctl(termfd, TCGETS, &old_t);
  
  /* find the directory relating to the desired host name */
  if (argc != 2) fatal("Usage: mlogin <host name>\n");
  sprintf(lock, "/tmp/%s/lock", argv[1]);

  if ((lockfd = open(lock, O_RDWR)) == -1){
    fatal("Can't locate host \n");
  }
  if (lockf(lockfd, F_LOCK, 0) != 0)
    fatal("Can't lock the lock file\n");

  /*
   * Get the map of free terminal lines and allocate ourselves one
   */

  read(lockfd, &linecount, sizeof(linecount));
  read(lockfd, &linemap, sizeof(linemap));
  ourline = 0;
  for (count = 0; count != linecount; count++)
    if ((linemap & (1 << count)) != 0) { 	/* found a free terminal */
      ourline = count + 1;
      break;
    }
    
  if (ourline == 0) fatal("Out of terminal lines\n");

  linemap &= ~(1 << count);
  lseek(lockfd, sizeof(linecount), SEEK_SET);
  write(lockfd, &linemap, sizeof(linemap));

  lockf(lockfd, F_ULOCK, 0);
  close(lockfd);

  
  /* open the named pipe with the correct name and set its parameters */

  sprintf(host_name, "/tmp/%s/out.%d", argv[1], ourline);
  if ((pipeinfd = open(host_name, O_RDONLY)) == -1){
    extern int errno;
    printf("errno = %d\n", errno);
    fatal("Can't get named pipe\n");
  }
  sprintf(host_name, "/tmp/%s/in.%d", argv[1], ourline);
  if ((pipeoutfd = open(host_name, O_RDWR)) == -1){
    extern int errno;
    printf("errno = %d\n", errno);
    fatal("Can't get named pipe\n");
  }

  fcntl(pipeinfd, F_SETFL, O_NDELAY | O_RDWR);
  fcntl(pipeoutfd, F_SETFL, O_NDELAY | O_RDWR);

  /* setup the user terminal mode (no echo, no signals, non-blocking etc) */

  t = old_t;
  t.c_lflag &= ~(ECHO | ICANON | ISIG );
  t.c_iflag &= ~(IXON | IXOFF);
  t.c_cc[VMIN] = 1;
  t.c_cc[VTIME] = 0;
  ioctl(termfd, TCSETS, &t);
  fcntl(termfd, F_SETFL, O_NDELAY | O_RDWR);
  
  /* ignore all signals */

  for (count = 1; count < 32; count++)
    signal(count, SIG_IGN);
  
  write(termfd, "Connected to host : ", 20);
  write(termfd, argv[1], strlen(argv[1]));
  write(termfd, "\n", 1);
  
  term();
  return 0;
}



/*===========================================================================*/
