static char rcsid[] = "$Id: config.c,v 1.1 1996/03/20 21:42:53 paul Exp $";

/* This code is responsible for reading a Minix configuration file and opening
 * any files that are required.  Also, any other functions that access the
 * various descriptors used by smx for emulated devices.
 */

#include <stdio.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <fcntl.h>
#include <sys/stat.h>
#include <sys/termios.h>
#include <sys/file.h>
#include <stropts.h>
#include <sys/conf.h>
#include <sys/param.h>
#include <string.h>
#undef HZ
#include "../../../include/minix/config.h"
#undef NULL /* Defined again in const.h */
#include "../../../include/minix/const.h"

#define NAMESIZE 	15	/* Maximum length of an option name */
#define COMMENT_CHAR	'#'	/* lines the startup file that start with */
				/* COMMENT_CHAR are ignored */
#define NUM_TERMS	3	/* number of external terminals */

/* Note that all of these FD's have to be manually kept in sync with those in
 * kernel/const.h
 */

#define TERMINAL_FD	20	/* first SunOS descriptor for terminals */
#define DISK_FD		40	/* first SunOS descriptor for disks */
#define SOCKET_FD	50	/* the SunOS descriptor for socket */
#define LOGGING_FD      51      /* SunOS descriptor for log file */


static char *skip(char *pos);
static char *getstring(char *src, char *dest);


void
config(const char *confname, long *mem_size, char *image_name, int *prot_lev,
       int *debug, char **host)
{
    /*
     * This hideously long function reads the Minix configuration file
     * and deals with each possible option. All files (and sockets)
     * needed for Minix are opened. The desired memory size and the
     * kernel image file name are returned.
     */
  
    int fd;			    /* general purpose file descriptor */
    struct sockaddr sockname;	    /* our socket name */
    FILE *config;		    /* the configuration file */
    char buffer[MAXPATHLEN];	    /* a config file line */
    static char host_name[MAXPATHLEN]; /* the host name */
    char name[NAMESIZE];	    /* an option name */
    char *ptr;
    int line;
  
    if (confname != NULL){
	if ((config = fopen(confname, "r")) == NULL) {
	    fprintf(stderr, "Minix: can't open file: %s\n", confname);
	    exit(1);
	}
    } else if ((config = fopen(".minix", "r")) == NULL) {
	strcpy(buffer, (char *)getenv("HOME"));
	strcat(buffer, "/.minix");
	config = fopen(buffer, "r");
    }
  
    if (config == NULL) {
	fprintf(stderr, "Minix: can't open .minix file\n");
	exit(1);
    }
    
    host_name[0] = '\0';
  
    /* foreach line in the config file, separate the option name from the 
     * option's value.
     */
    while(fgets(buffer, sizeof(buffer), config), !feof(config)) {
	
	/* skip blank lines */
	if ((ptr = skip(buffer)) == NULL) continue;
    
	/* skip comment lines */
	if (*ptr == COMMENT_CHAR) continue;
    
	/* get the option name */
	ptr = getstring(ptr, name);
	if ((ptr = skip(ptr)) == NULL) {
	    fprintf(stderr,
		    "Minix config file: missing value for option '%s'\n",name);
	    exit(1);
	}
	ptr[strlen(ptr) - 1] = '\0';
    
	/* OPTION: memory */
	if (strcmp("memory", name) == 0){
	    *mem_size = atoi(ptr);
	    if (*mem_size < 512 || *mem_size > 32768){
		fprintf(stderr, "Minix config file: invalid memory size: %d\n",
			*mem_size);
		exit(1);
	    }
	    *mem_size *= 1024;
      
	    /* OPTION: image */
	} else if (strcmp("image", name) == 0){
	    strncpy(image_name, ptr, MAXPATHLEN);
      
	    /* OPTION: host */
	} else if (strcmp("host", name) == 0){
	    strncpy(host_name, "/tmp/" , 5);
	    strncpy(host_name + 5, ptr, sizeof(host_name) - 5);
      
	    /* OPTION: hdx0, ..., hdx7 */
	} else if (strncmp("hdx", name, 3) == 0){
	    int disk_num;
	    
	    disk_num = (int)(name[3] - '0');
	    if (disk_num < 0 || disk_num > 7) {
		fprintf(stderr, "Minix config file: invalid disk number: %c\n",
			name[2]);
		exit(1);
	    }
	    if (((fd = open(ptr, O_RDWR)) < 0) &&
		((fd = open(ptr, O_RDONLY)) < 0)) {
		fprintf(stderr, "Minix config file: can't find disk: '%s'\n",
			ptr);
		exit(1);
	    }
	    if (dup2(fd, DISK_FD + disk_num) < 0) {
		fprintf(stderr, "Minix config file: dup error\n");
		exit(1);
	    }  
	    close(fd);
            
	} else if (strcmp("protect", name) == 0){
	    *prot_lev = atoi(ptr);
	    if (*prot_lev < 0 || *prot_lev > 2){
		fprintf(stderr, "Minix config file: Bad protection level\n");
		exit(1);
	    }
	    
	} else if (strcmp("debug", name) == 0){
	    if (strcmp("on", ptr) == 0) 
		*debug = 1;
	    else if (strcmp("off", ptr) == 0)
		*debug = 0;
	    else {
		fprintf(stderr, "Minix config file: Bad debug flag\n");
		exit(1);
	    }
      
	} else if (strcmp("logfile", name) == 0) {
	    if ((fd = open(ptr, O_WRONLY | O_CREAT | O_TRUNC, 0666)) < 0) {
		fprintf(stderr,"Minix config file: can't open logfile: '%s'\n",
			ptr);
		exit(1);
	    }
	    if (dup2(fd, LOGGING_FD) < 0) {
		fprintf(stderr, "Minix config file: dup error\n");
		exit(1);
	    }  
	    close(fd);
	    
	} else {
	    fprintf(stderr, "Minix config file: unknown config option '%s'\n",
		    name);
	    exit(1);
	}  /* end of checking for the various config options */
    
    } /* end of while loop for reading lines */
  
    /* Open and setup the main console's terminal device. Terminals must be
     * non-blocking and asynchronous (causes SIGIO) with no echo, flow control
     * or special control characters or signals.
     */
    fd = open("/dev/tty", O_RDWR);
    if( (dup2(fd, TERMINAL_FD) < 0) ||  (dup2(fd, TERMINAL_FD + 1) < 0)) {
	fprintf(stderr, "Couldn't get main terminal\n");
	exit(1);
    }
    close(fd);
    fclose(config);

    
    /* Possibly set up a socket so that extra logins are
     * possible. This requires that a host name is specified in the
     * config file.
     */
    *host = host_name;
    if (host_name[0] != '\0'){  

	/* 
	 * Within Solaris it is necessary to create a directory containing
	 * 2 named pipes per terminal line. A lock file is also included
	 * so that the 'mlogin' process can correctly allocate themselves
	 * a terminal line.
	 */
    
	if (mkdir(host_name, 0755) == -1){
	    fprintf(stderr, "Can't create named pipe directory\n");
	    exit(1);
	}

	for (line = 1; line < 1 + NUM_TERMS; line++){
	    sprintf(buffer, "%s/in.%d", host_name, line);
	
	    if (mknod(buffer, 0666 | S_IFIFO, 0) == -1){
		extern int errno;
		fprintf(stderr, "Can't create input pipe %d\n", errno);
		exit(1);
	    }
	    
	    fd = open(buffer, O_RDWR );
	    if (dup2(fd, TERMINAL_FD + line * 2) < 0) {
		fprintf(stderr, "Couldn't dup input pipe\n");
		exit(1);
	    }
	    close(fd);

	    sprintf(buffer, "%s/out.%d", host_name, line);
	    if (mknod(buffer, 0666 | S_IFIFO, 0) == -1){
		fprintf(stderr, "Can't create output pipe\n");
		exit(1);
	    }
	    fd = open(buffer, O_RDWR );
	    if (dup2(fd, TERMINAL_FD + line * 2 + 1) < 0) {
		fprintf(stderr, "Couldn't dup output pipe\n");
		exit(1);
	    }
	    close(fd);
	    
	}

	sprintf(buffer, "%s/lock", host_name);
	if ((fd = open(buffer, O_CREAT | O_RDWR, 0644 )) == -1){
	    fprintf(stderr, "Couldn't create lock file\n");
	    exit(1);
	}
	line = NUM_TERMS;
	write(fd, &line, sizeof(line));
	line = (1 << NUM_TERMS) - 1;
	write(fd, &line, sizeof(line));
	close(fd);
    }
}


static char *
skip(char *pos)
{
/* Skip through a buffer until a non-blank or end of line character is found */

    while(*pos == ' ' || *pos == '\t') {
  	pos++;
    }
    if (*pos == '\n') return (NULL);
    return(pos);
}


static char *
getstring(char *src, char *dest)
{
/* Fetch a single token (eg. name) from a buffer.*/

    while(*src != ' ' && *src != '\t' && *src != '\n')
  	*dest++ = *src++;
    *dest = '\0';
    return(src);
}


/*
 * Saving and restoring of tty
 */

static struct termios old_set;

void
set_sun_tty(void)
{
    struct termios t;
    int fd;

    ioctl(TERMINAL_FD, TCGETS, &old_set);
    t = old_set;
    t.c_lflag &= ~(ECHO | ICANON | ISIG);
    t.c_iflag &= ~(IXON | IXOFF);
    t.c_cc[VMIN] = 1;
    t.c_cc[VTIME] = 0;
    ioctl(TERMINAL_FD, TCSETS, &t);
    
    for (fd = 0; fd < 2 * (1 + NUM_TERMS); fd++) {
	fcntl(TERMINAL_FD + fd, F_SETFL, O_NDELAY | O_RDWR);
    }
}


void
reset_sun_tty(void)
{
    ioctl(TERMINAL_FD, TCSETS, &old_set);
}


/*
 * Setup all Minix terminals to send interrupts on input.
 */
void
enable_tty_interrupts(void)
{
    int fd;

    for (fd = 0 ; fd < 2 * (1 + NUM_TERMS) ; fd += 2) {
	ioctl(TERMINAL_FD + fd, I_SETSIG, S_RDNORM);      
    }
}
