static char rcsid[] = "$Id: build.c,v 1.1 1996/03/20 21:42:50 paul Exp $";

/* This program is responsible for booting Solaris Minix. The
 * configuration file is examined and the appropriate files and the
 * desired amount of memory is allocated. Each of the four sections of
 * the Minix image file (kernel, mm, fs, init) are read into memory
 * and relocated. This is a Solaris utility and should not be executed
 * under Minix.
 */
					     
#include <stdio.h>
#include <fcntl.h>
#include <malloc.h>
#include <unistd.h>
#include <sys/wait.h>
#include <sys/param.h>
#include <signal.h>
#include "../../../include/minix/config.h"
#undef HZ                       /* Defined again in minix/const.h */
#undef NULL                     /* Defined again in minix/const.h */
#include "../../../include/minix/const.h"

#include "../elf/a.out.h"
#include "config.h"
#include "relocate.h"

#define NUM_PROGS 	4	/* number of separate programs in 'image' */
#define KERNEL_PROG 	0
#define MM_PROG 	1
#define FS_PROG 	2
#define DEF_MEM_SIZE	2048	/* default nunmber of kbytes of memory */
#define DEF_PROTECT	1	/* default protection level */
#define NUM_ARGS	2	/* number of possible command line options */

#define USAGE "%s [-d] [-m[none|half|full]] [config-file]\n"

extern char end ;
#ifdef RDEBUG
extern char etext, edata ; 
#endif

static int stack_size[] = { 0, 10, 4, 4};

char *memory_base;			/* base of MINIX's memory */
      
unsigned short *sizes;			/* this will actually be part of the */
					/* kernel's data segment */

/* pointers to their data segments */
unsigned short *fs_data,*mm_data;
char *progname;

int
main(int argc, char *argv[])
{
    int prog, fd;
    long core_size;
    char *tbase, *base, image_name[MAXPATHLEN], file[MAXPATHLEN];
    long request, size, t_size, d_size, b_size, b_base;
    int debug, protect, configdebug, configprotect;
    int num_args;
    char *config_file;
    unsigned long gaps[NUM_PROGS];
    struct nlist nl[3];
    int *debug_addr, *prot_addr;
    char *hostname;
    int status;
    sigset_t sigs;

    progname = argv[0];
    argv++; argc--;

    /*
     * Check that the click size is a multiple of the pagesize - otherwise
     * protection of segments using mprotect(2) will fail.
     */
    if (CLICK_SIZE % sysconf(_SC_PAGESIZE) != 0) {
	(void) fprintf(stderr,
     "%s: Minix aborted - clicksize (%d) is not a multiple of pagesize (%d)\n",
		       progname, CLICK_SIZE, sysconf(_SC_PAGESIZE));
	exit(1);
    }

    /*
     * Process options. Only the debug (-d) and the three memory
     * protection levels (-mnone, -mhalf, -mfull) are accepted
     */
    config_file = 0;
    debug = 0;
    protect = -1;
    for (num_args = 0; num_args < NUM_ARGS; num_args++){
	if (argc > 0) {
	    if (strcmp(*argv, "-d") == 0) {
		debug = 1;
		argc--; argv++;
	    }
	    else if (strcmp(*argv, "-mnone") == 0) {
		protect = 0;
		argc--; argv++;
	    }
	    else if (strcmp(*argv, "-mhalf") == 0) {
		protect = 1;
		argc--; argv++;
	    }
	    else if (strcmp(*argv, "-mfull") == 0) {
		protect = 2;
		argc--; argv++;
	    }
	}
    }
    if (argc > 1) {
	(void) fprintf(stderr, USAGE, progname);
	exit(1);
    }
    if (argc == 1) {
	config_file = *argv;
    }
  

    /*
     * open all disk files etc. and get the desired memory size, image file
     * name, protection level and debug mode.
     */
    core_size = DEF_MEM_SIZE * 1024;
    configprotect = DEF_PROTECT;
    configdebug = 0;
    config(config_file, &core_size, image_name, &configprotect, &configdebug,
	   &hostname);
  
    /*
     * Command line protection and debug options overide the configuration
     * file options.
     */
    if (!debug) {
        debug = configdebug;
    }
    if (protect == -1) {
        protect = configprotect;
    }

    set_sun_tty();        /* Setup the terminal for use by smx */
    
    switch (fork()){
    case -1:      /* error */
	reset_sun_tty();
	fprintf(stderr, "%s: Unable to fork secondary Minix process\n",
		progname);
	exit(1);
    
    case 0:       /* child */
	break;
    
    default:      /* parent */
	sigfillset(&sigs);
	sigdelset(&sigs, SIGILL);
	sigdelset(&sigs, SIGTRAP);
	sigdelset(&sigs, SIGEMT);
	sigdelset(&sigs, SIGFPE);
	sigdelset(&sigs, SIGBUS);
	sigdelset(&sigs, SIGSEGV);
	sigdelset(&sigs, SIGSYS);
	(void) sigprocmask(SIG_SETMASK, &sigs, NULL);

	wait(&status);

	/*
	 * Tidy up after smx has finished---delete terminal named pipes
	 * and restore terminal settings.
	 */
	if (hostname[0] != '\0') {
	    sprintf(file, "rm -rf %s", hostname);
	    system(file);
	}
	reset_sun_tty();
	exit(0);
    } /* fork() switch */

    /* Round up core size to the next click */
    core_size = (core_size + CLICK_SIZE - 1) & ~(CLICK_SIZE-1);
  
    /*
     * Use malloc() to allocate Minix's core memory and determine the
     * starting click
     */
    request = (long) core_size + (long) CLICK_SIZE;
    if ((tbase = malloc(request)) == NULL) {
  	fprintf(stderr, "%s: Unable to allocate enough memory\n", progname);
	exit(1);
    }
    memory_base = (char *)(((int)tbase + CLICK_SIZE - 1) & ~(CLICK_SIZE - 1));
  
  
    if (debug) {
	(void) fprintf(stderr, "Core size = 0x%x\nMemory base = 0x%x\n",
		       (unsigned) core_size, (unsigned long) memory_base);
    }  

    /*
     * Open the image file.
     */
    if ((fd = open(image_name, O_RDONLY)) < 0){
  	fprintf(stderr, "%s: Unable to open image file\n", progname);
	exit(1);
    }
  
    /*
     * Foreach program within the image file, load it, relocate it and
     * store its size in the size's array (in the kernel's data
     * segment).
     */
    
    base = memory_base;
    for (prog = 0; prog < NUM_PROGS; prog++)
    {
	gaps[prog] = relocate(fd, base, &t_size, &d_size);
  	if (prog == KERNEL_PROG)
	    sizes = (unsigned short *)(memory_base + t_size);
	else if (prog == MM_PROG)
	    mm_data = (unsigned short *)( base + t_size);
	else if (prog == FS_PROG)
	    fs_data = (unsigned short *)(base + t_size);
	
	d_size += stack_size[prog] << CLICK_SHIFT;

	sizes[prog * 2] = t_size >> CLICK_SHIFT;
	sizes[prog * 2 + 1] = (unsigned short)(d_size >> CLICK_SHIFT);
	
	if (debug) {
	    (void) fprintf(stderr,
			   "Program %d, base 0x%x, t_xize 0x%x, d_size 0x%x\n",
			   prog, base, t_size, d_size);
	}
	base += t_size + d_size;
    }

    sizes[10] = debug;
    sizes[11] = protect;

    /* This extra value is used by MM (via do_mem) to build its hole table */
    *((long *) &sizes[8]) = (long)(core_size);
  
    /* is MM valid? */
    if (*mm_data != 0xdada){
  	fprintf(stderr,"Bad MM marker\n");
	exit(1);
    }
    mm_data[1] = (long)memory_base >> CLICK_SHIFT;
  
    /* is FS valid? */
    if (*fs_data != 0xdada){
  	fprintf(stderr,"Bad FS marker\n");
	exit(1);
    }
    fs_data[2] = (long)(base - t_size - d_size)>>CLICK_SHIFT; /*init's origin*/
    fs_data[3] = t_size>>CLICK_SHIFT;			/* init's text size */
    fs_data[4] = d_size>>CLICK_SHIFT;			/* init's data size */
  
    enable_tty_interrupts();

    /* start the kernel running */
    ((void (*)()) memory_base)();
}



