static char rcsid[] = "$Id: relocate.c,v 1.1 1996/03/20 21:42:58 paul Exp $";

/* This program will load an object file linked with 'ld -r -dc *.o' and 
 * perform all necessary relocation. This is part of the SunOS 'build' program.
 */

#include <fcntl.h>
#include <malloc.h>
#include <stdio.h>
#include <unistd.h>
#include "../../../include/minix/config.h"
#undef NULL /* Defined again in const.h */
#include "../../../include/minix/const.h"

/*#define RDEBUG*/


#include "../elf/a.out.h"
#include "relocate.h"
					
#define  MASK(n) ((1<<(n))-1)	
	

/* to help save typing */
typedef struct reloc_info_sparc ris;

/* for storing the file's header */
struct exec header;		

/* various segment pointers */
char *t_base, *d_base, *b_base, *tr_base, *dr_base;

/* these sizes are rounded up to the click size. Note, d_size includes bss */
unsigned long t_size,d_size,tot_size,tr_size,dr_size;


/* given the length of the segment (from the load file), round the length
 * up to the nearest click boundary
 */
unsigned long
adjust(unsigned long length)
{
    return (length + CLICK_SIZE - 1) & ~(CLICK_SIZE - 1);
}


/* given the name of the load file, open it, read all the segments into memory
 * and collect pointers to them. Only the text, data, text reloc and data reloc
 * parts are loaded.
 */ 
unsigned long
loadit(int fd, char *base)
{
    int stringsize;
    unsigned long bytesread; 

    /* get the size information from the file header */
    read(fd, &header, sizeof(header));
    t_size = adjust(header.a_text);
    d_size = adjust(header.a_data + header.a_bss);
    tr_size = header.a_trsize;
    dr_size = header.a_drsize;
    tot_size = t_size + d_size;  

    /* allocate the final memory, then load in the segments */
    /* we assume that no read errors will occur */
    t_base = base;
    base += tot_size;
    if (bytesread = read(fd, t_base, header.a_text) != header.a_text){
	printf("error reading text  base %08x  wanted %08x  got %08x\n",
	       t_base, header.a_text, bytesread);
	exit(2);
    }

    d_base = t_base + t_size;
    if (bytesread = read(fd, d_base, header.a_data) != header.a_data){
	printf("error reading data  base %08x  wanted %08x  got %08x\n",
	       d_base, header.a_data, bytesread);
	exit(3);
    }
    b_base = d_base + header.a_data;	/* bss is not aligned */  
    {
	int count;

	for (count = 0; count < header.a_bss; count++)
	    *(((char *)b_base) + count) = '\0';
    }
	
    /* read in the text and data relocation info */
    tr_base = base;
    base += tr_size;
    dr_base = base;
    base += dr_size;
    read(fd, tr_base, tr_size);
    read(fd, dr_base, dr_size);
    lseek(fd, header.a_syms, SEEK_CUR);	/* skip symbols */
    read(fd, &stringsize, sizeof(stringsize));
    lseek(fd, stringsize-4, SEEK_CUR);   /* skip string table */

    /* the entire file is now in memory */
#ifdef RDEBUG
    printf("++++++++Program now in memory++++++++++\n");
    printf("          base      size\n");
/*        name__  XXXXXXXX  XXXXXXXX */
    printf("text    %08X  %08X\n",t_base,t_size);
    printf("data    %08X  %08X\n",d_base,header.a_data);
    printf("bss     %08X  %08X\n",b_base,header.a_bss);
    printf("total data size   %08X\n",d_size);
    printf("treloc  %08X  %08X\n",tr_base,tr_size);
    printf("dreloc  %08X  %08X\n",dr_base,dr_size);
#endif

  return t_size - header.a_text;
}


/* This code is responsible for performing the actual relocations. 'rp' is the
 * relocation info entry from the load file, 'where' is the destination of the
 * change, 'what' is either the value to be inserted into 'where', or a 
 * displacement from 'where'.
 */
void
upd_reloc(ris *rp, unsigned long *where, unsigned long what)
{
    switch (rp->r_type) {
    case RELOC_RELATIVE:
	what += *where << (32-22);
	*where = (*where & ~MASK(22)) | ((what >> (32-22)) & MASK(22));
	where++;
	what += *where & MASK(10);
	*where = (*where & ~MASK(10)) | (what & MASK(10));
	break;

    case RELOC_8:
    case RELOC_DISP8:
	what += *where & MASK(8);
	*where = what;
	break;

    case RELOC_LO10:
    case RELOC_BASE10:
	what += *where & MASK(10);
	*where = (*where & ~MASK(10)) | (what & MASK(10));
	break;

    case RELOC_BASE13:
    case RELOC_13:
	what += *where & MASK(13);
	*where = (*where & ~MASK(13)) | (what & MASK(13));
	break;

    case RELOC_16:
    case RELOC_DISP16:
	what += *where & MASK(16);
	*(short *)where = what;
	break;

    case RELOC_22:
    case RELOC_BASE22:
	what += *where & MASK(22);
	*where = (*where & ~MASK(22)) | (what & MASK(22));
	break;

    case RELOC_HI22:
	what += (*where<<32-22) & MASK(22);
	*where = (*where & ~MASK(22)) | ((what>>(32-22)) & MASK(22));
	break;
	
    case RELOC_WDISP22:
	what += *where & MASK(22);
	what >>= 2;
	*where = (*where & ~MASK(22)) | (what & MASK(22));
	break;
		
    case RELOC_WDISP30:
	what += *where & MASK(30);
	what >>= 2;
	*where = (*where & ~MASK(30)) | (what&MASK(30));
	break;

    case RELOC_32:
    case RELOC_GLOB_DAT:
    case RELOC_DISP32:
	what += *where;
	*where = what;
	break;

    default:
	fprintf(stderr, "unknown relocation type %d", rp->r_type);
	break;
    }
}


/* perform all relocation for the given segment (text or data). 'rel_base' is
 * the base of the load file relocation info, 'rel_size' is the length of the
 * reloc table and 'seg_base' gives the base of the current segment (text or
 * data).
 */
void
reloc(char *rel_base, int rel_size, unsigned long seg_base)
{
    ris *rp = (ris *) rel_base;
    unsigned long offset, dest_value;
    int rel_num = 0;

    if (rel_size == 0) return;
    do 
    {
#ifdef RDEBUG
  	printf("relocation element %d\n", rel_num);
	printf("address = %d  index = %d", rp->r_address, rp->r_index);
	printf("  extern = %d  type = %d", rp->r_extern, rp->r_type);
	printf("  addend  = %d\n\n", rp->r_addend);
#endif
	
	/* check for relocations that rely on symbols */
	if (rp->r_extern == 1)
	{
	    fprintf(stderr, "Undefined symbols still remain\n");
	    exit(1);
	}

	/* decide which segment contains the destination of the relocation */
	switch(rp->r_index)
	{
	case N_UNDF:			/* all symbols must be defined */
	    fprintf(stderr, "Undefined symbols still remain\n");
	    exit(1);

	case N_ABS:
	    offset = 0;
	    break;

	case N_TEXT:			/* rel. to base of text */
	    offset = (unsigned long)t_base;
	    break;

	case N_DATA:			/* rel. to base of data */
	    offset = (unsigned long)d_base - header.a_text;
	    break;

	case N_BSS:			/* rel. to base of bss */
	    offset = (unsigned long)d_base - header.a_text;
	    break;

	default:
	    fprintf(stderr, "Invalid relocation index found\n");
	    exit(1);
	}
	
	/* decide how the destination should be referenced (rel. or abs.) */
	switch(rp->r_type)
	{
	case RELOC_DISP8:
	case RELOC_DISP16:
	case RELOC_DISP32:
	case RELOC_WDISP30:
	case RELOC_WDISP22:
	    dest_value = rp->r_addend;
	    break;

	default:
	    dest_value = offset + rp->r_addend;
	    break;
	}
	
	/* perform the relocation */
	upd_reloc(rp, (unsigned long *)(rp->r_address + seg_base), dest_value);
/*	upd_reloc(rp, (rp->r_address + seg_base), (dest_value + 1) & ~0x1);*/
    } while (rel_num += sizeof(ris), rp++, rel_num != rel_size);
}


unsigned long
relocate(int fd, char *base, unsigned long *ptext_size,
	 unsigned long *pdata_size)
{
    unsigned long gap;
    
    gap = loadit(fd, base);
  
    /* relocate both the text and data segments */
    reloc(tr_base, tr_size, (unsigned long) t_base);
    reloc(dr_base, dr_size, (unsigned long) d_base);
    
    /* the code is now ready to be executed */
    *ptext_size = t_size;
    *pdata_size = d_size;
    
    /*
     * Return the gap between the text and data segments.
     */
    return gap;
}
