static char rcsid[] = "$Id";

/* This file contains the main program of MINIX.  The routine main()
 * initializes the system and starts the ball rolling by setting up the proc
 * table, interrupt vectors, and scheduling each task to run to initialize
 * itself.
 *
 * The entries into this file are:
 *   main:		MINIX main program
 *   panic:		abort MINIX due to a fatal error
 */

#include "kernel.h"
#include <unistd.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "proc.h"
#if (MACHINE == SUN)
#include <minix/boot.h>
#include <sun/syscall.h>
#include <sun/signal.h>
#include "logging.h"
#endif

#if (MACHINE == SUN)
so_sigset_t sigs;
void * vectors[_SIGRTMAX + 1];
#endif



/*===========================================================================*
 *                                   main                                    *
 *===========================================================================*/
PUBLIC void main()
{
/* Start the ball rolling. */

  register struct proc *rp;
  register int t;
  int sizeindex;
  phys_clicks text_base;
  vir_clicks text_clicks;
  vir_clicks data_clicks;
  phys_bytes phys_b;
  reg_t ktsb;			/* kernel task stack base */
  struct memory *memp;
  struct tasktab *ttp;
#if (MACHINE == SUN)
  void protect_init();  
  
  struct so_sigaction sig;
  struct so_sigaltstack sstk;

  if (sizeof(so_sigset_t) != SO_SIGSET_BYTES) {
      panic("SO_SIGSET_BYTES has the wrong value---change it!", 0);
  }

  /* fetch debug and prot_lev values that were placed by build */
  debug = sizes[10];
  prot_lev = sizes[11];

  sunsigemptyset(&sigs);
  sunsigaddset(&sigs, SO_SIGIO);
  sunsigaddset(&sigs, SO_SIGALRM);

  SunOS(SYS_sigprocmask, SO_SIG_SETMASK, &sigs, (so_sigset_t *)0 );
#endif


#if (CHIP == INTEL)
  /* Initialize the interrupt controller. */
  intr_init(1);
#endif

  /* Interpret memory sizes. */
  mem_init();

#if (MACHINE == SUN)
 /*
  * For protection purposes, the size of the kernel stack must be a
  * multiple of the click size.  The following is designed to cause
  * the compilation of this file to fail if the kernel stack is
  * not a multiple of the click size.
  */
#if K_STACK_BYTES % CLICK_SIZE != 0
  compilererror;
#endif

  debug_str("Starting kernel - debugging on\n");
#endif
  /* Clear the process table.
   * Set up mappings for proc_addr() and proc_number() macros.
   */
  for (rp = BEG_PROC_ADDR, t = -NR_TASKS; rp < END_PROC_ADDR; ++rp, ++t) {
	rp->p_flags = P_SLOT_FREE;
#if (MACHINE == SUN)
	rp->p_stn = INVALID_STN;
#endif
	rp->p_nr = t;		/* proc number from ptr */
        (pproc_addr + NR_TASKS)[t] = rp;        /* proc ptr from number */
  }

  /* Set up proc table entries for tasks and servers.  The stacks of the
   * kernel tasks are initialized to an array in data space.  The stacks
   * of the servers have been added to the data segment by the monitor, so
   * the stack pointer is set to the end of the data segment.  All the
   * processes are in low memory on the 8086.  On the 386 only the kernel
   * is in low memory, the rest if loaded in extended memory.
   */

  /* Task stacks. */
  ktsb = (reg_t) t_stack;

  for (t = -NR_TASKS; t <= LOW_USER; ++t) {
	rp = proc_addr(t);			/* t's process slot */
	ttp = &tasktab[t + NR_TASKS];		/* t's task attributes */
	strcpy(rp->p_name, ttp->name);
	if (t < 0) {
		if (ttp->stksize > 0) {
			rp->p_stguard = (reg_t *) ktsb;
			*rp->p_stguard = STACK_GUARD;
		}
		ktsb += ttp->stksize;
		rp->p_reg.sp = ktsb;
		text_base = code_base >> CLICK_SHIFT;
					/* tasks are all in the kernel */
		sizeindex = 0;		/* and use the full kernel sizes */
		memp = &mem[0];		/* remove from this memory chunk */
	} else {
		sizeindex = 2 * t + 2;	/* MM, FS, INIT have their own sizes */
	}
	rp->p_reg.pc = (reg_t) ttp->initial_pc;
	rp->p_reg.psw = istaskp(rp) ? INIT_TASK_PSW : INIT_PSW;

	text_clicks = sizes[sizeindex];
	data_clicks = sizes[sizeindex + 1];
	rp->p_map[T].mem_phys = text_base;
	rp->p_map[T].mem_len  = text_clicks;
	rp->p_map[D].mem_phys = text_base + text_clicks;
	rp->p_map[D].mem_len  = data_clicks;
	rp->p_map[S].mem_phys = text_base + text_clicks + data_clicks;
	rp->p_map[S].mem_vir  = data_clicks;	/* empty - stack is in data */

#if (CHIP == SPARC)
	if (!istaskp(rp) && !isidlehardware(proc_number(rp))) {
   	        rp->p_reg.pc = ((reg_t) text_base) << CLICK_SHIFT;
   	        rp->p_reg.sp = rp->p_map[S].mem_phys << CLICK_SHIFT;
        }
	rp->p_map[T].mem_vir = rp->p_map[T].mem_phys;
	rp->p_map[D].mem_vir = rp->p_map[D].mem_phys;
	rp->p_map[S].mem_vir = rp->p_map[S].mem_phys;
	rp->p_map[S].mem_len = 0;

	rp->p_reg.npc = rp->p_reg.pc + 4;
	rp->p_reg.sp -= INIT_SP;         /*Must leave room for a stack frame*/
#endif /* CHIP == SPARC */
debug_str("Process "); debug_int(t);
debug_str(": pc = "); debug_int(rp->p_reg.pc);
debug_str(", sp = "); debug_int(rp->p_reg.sp);
debug_str("\n");
	text_base += text_clicks + data_clicks;	/* ready for next, if server */

	memp->size -= (text_base - memp->base);
	memp->base = text_base;			/* memory no longer free */

#if (CHIP == INTEL)
	if (t >= 0) {
		/* Initialize the server stack pointer.  Take it down one word
		 * to give crtso.s something to use as "argc".
		 */
		rp->p_reg.sp = (rp->p_map[S].mem_vir +
				rp->p_map[S].mem_len) << CLICK_SHIFT;
		rp->p_reg.sp -= sizeof(reg_t);
	}

#if _WORD_SIZE == 4
	/* Servers are loaded in extended memory if in 386 mode. */
	if (t < 0) {
		memp = &mem[1];
		text_base = 0x100000 >> CLICK_SHIFT;
	}
#endif
#endif /* CHIP == INTEL */
	if (!isidlehardware(t)) lock_ready(rp);	/* IDLE, HARDWARE neveready */
	rp->p_flags = 0;

#if (CHIP == INTEL)
	alloc_segments(rp);
#endif
  }

  proc[NR_TASKS+INIT_PROC_NR].p_pid = 1;/* INIT of course has pid 1 */
  bill_ptr = proc_addr(IDLE);		/* it has to point somewhere */
  lock_pick_proc();

#if (MACHINE == SUN)
  /* tell SunOS where the 'sigcontext' information about the currently 
   * executing process should be put.
   */
   
  sstk.ss_sp = (char *)getksp() - K_STACK_BYTES;
  sstk.ss_size = K_STACK_BYTES;
  sstk.ss_flags = 0;
  SunOS(SYS_sigaltstack, &sstk, (so_stack_t *)0);

  debug_str("Return value of getksp(): ");
  debug_int((int)sstk.ss_sp);
  debug_char('\n');

  signal_mask = ~(sunsigmask(SO_SIGILL) | sunsigmask(SO_SIGTRAP) |
	sunsigmask(SO_SIGEMT) | sunsigmask(SO_SIGFPE) | sunsigmask(SO_SIGBUS) |
	sunsigmask(SO_SIGSEGV) | sunsigmask(SO_SIGSYS)); 
  
  /* each of SunOS's signals (except USR2) go through a special handler that
   * takes the current Minix process's context and stores it in the proc table
   */
  sig.sa_flags = SUN_SA_ONSTACK | SUN_SA_SIGINFO;
  sig.sa_handler = SunOSsig;
  sunsigemptyset(&sigs);
  sunsigmakeset(&sigs, signal_mask);
  sig.sa_mask = sigs;
  for (t = 1; t <= _SIGRTMAX; t ++) 
	SunOS(SYS_sigaction, t, &sig, (struct sigaction *) 0);
	
  /* SIGUSR2 must be done separately since its purpose is to start a process
   * running with a new context, without saving the current context. This is
   * used by a process when it returns from the Minix signal handling code.
   */
  sig.sa_handler = restart2;
  
  SunOS(SYS_sigaction, SO_SIGUSR2, &sig, (struct sigaction *)0);

  /* the special signal handler uses 'vectors' to decide which function should
   * be called to handle special signals (eg. SIGIO).
   */
  for(t = 0; t <= _SIGRTMAX; t++)
  	vectors[t] = exception;

  vectors[SO_SIGUSR1] = s_call;
  vectors[SO_SIGIO] = keyboard;
  vectors[SO_SIGALRM] = clock_handler;

  /*
   * Setup the boot information.
   */
  boot_parameters.bp_rootdev = DEV_FD0;
  boot_parameters.bp_ramimagedev = 0;
  boot_parameters.bp_ramsize = 0;
  boot_parameters.bp_processor = CHIP;

  /* enable all SunOS signals */
  unlock();

  protect_init(code_base, sizes[0], sizes[1], mem_bytes);

  debug_str("About to call log_init\n");

  log_init();
#endif /* MACHINE == SUN */

  debug_str("In kernel main---about to call restart\n");

  /* Now go to the assembly code to start running the current process. */
  restart();
}


/*===========================================================================*
 *                                   panic                                   *
 *===========================================================================*/
PUBLIC void panic(s,n)
_CONST char *s;
int n;
{
/* The system has run aground of a fatal error.  Terminate execution.
 * If the panic originated in MM or FS, the string will be empty and the
 * file system already syncked.  If the panic originates in the kernel, we are
 * kind of stuck.
 */

  if (*s != 0) {
	printf("\nKernel panic: %s",s);
	if (n != NO_NUM) printf(" %d", n);
	printf("\n");
  }
#if (MACHINE == SUN)
  /* a useful register and context dump */
  do_dump();
#endif
  wreboot(RBT_PANIC);
}
