static char rcsid[] = "$Id: sunconsole.c,v 1.1 1996/03/20 21:27:59 paul Exp $";

/* The file contains the code for transferring Minix's terminal output into
 * SunOS. Also, multiple terminal handling is performed here.
 */
 
#include "kernel.h"
#include <termios.h>
#include "proc.h"
#include <sgtty.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "tty.h"
#include <signal.h>
#include <sun/syscall.h>

#define CONSOLE		0

FORWARD _PROTOTYPE( void cons_write, (tty_t *tp)                             );
FORWARD _PROTOTYPE( void cons_echo, (tty_t *tp, int c)                       );


/*===========================================================================*
 *				cons_write				     *
 *===========================================================================*/
PRIVATE void cons_write(tp)
tty_t *tp;
{
/* As much data as possible is copied from the user's address space into
 * SunOS's buffers. A limited amount of flow control is implemented in order
 * to stop output as soon as CTRL-S or CTRL-C is pressed. Due to the large
 * number of buffers between Minix and some Unix terminals, it may not work
 * too well.
 */
     
    int remaining, count;
    char buf[64];
    phys_bytes user_phys;

    /* Check quickly for nothing to do, so this can be called often without
     * unmodular tests elsewhere.
     */
    if ((count = tp->tty_outleft) == 0 || tp->tty_inhibited) return;
  
    do {
	if (count > sizeof(buf)) count = sizeof(buf);
	user_phys = proc_vir2phys(proc_addr(tp->tty_outproc), tp->tty_out_vir);
	phys_copy(user_phys, vir2phys(buf), (phys_bytes) count);

	SunOS(SYS_write, TERMINAL_FD + 1 + tty_line(tp) * 2,
	      buf, count);
	
	tp->tty_out_vir += count;
	tp->tty_outcum += count;
	tp->tty_outleft -= count;

    } while ((count = tp->tty_outleft) != 0 && !tp->tty_inhibited);

    /* Reply to the writer if all output is finished. */
    if (tp->tty_outleft == 0) {
	tty_reply(tp->tty_outrepcode, tp->tty_outcaller, tp->tty_outproc,
		  tp->tty_outcum);
	tp->tty_outcum = 0;
    }
}
 

/*===========================================================================*
 *                              cons_echo                                    *
 *===========================================================================*/
PRIVATE void cons_echo(tp, c)
tty_t *tp;             /* pointer to tty struct */
int c;                          /* character to be echoed */
{
  out_char(tp, c);
}


/*===========================================================================*
 *				out_char 				     *
 *===========================================================================*/
PUBLIC void out_char(tp, c)
tty_t *tp;
char c;
{
/* Output a single character to a terminal. */

  SunOS(SYS_write, TERMINAL_FD + 1 + tty_line(tp) * 2, &c, 1);
}


/*===========================================================================*
 *				putk					     *
 *===========================================================================*/
PUBLIC void putk(c)
char c;				/* character to print */
{
/* This procedure is used by the version of printf() that is linked with
 * the kernel itself.  The one in the library sends a message to FS, which is
 * not what is needed for printing within the kernel.  This version just queues
 * the character and starts the output.
 */
    if (c) {
	if (c == '\n') out_char(&tty_table[CONSOLE], (int) '\r');
	out_char(&tty_table[CONSOLE], (int) c);
    }
}


/*===========================================================================*
 *				hangup					     *
 *===========================================================================*/
PUBLIC void hangup(connum)
int connum;
{
    struct tty *tp = &tty_table[connum];

/* When a remote user exits from their terminal program, a special code is
 * sent and we must close the two file descriptors and kill all their 
 * processes.
 */
	printf("Hanging up console %d\n", connum);
	scr_init(tp);
	sigchar(tp, SIGKILL);
}

	   	   
/*===========================================================================*
 *				scr_init				     *
 *===========================================================================*/
void scr_init(tp)
tty_t *tp;
{

  /* Initialize the keyboard driver. */
  kb_init(tp);

  /* Output functions. */
  tp->tty_devwrite = cons_write;
  tp->tty_echo = cons_echo;
}
