static char rcsid[] = "$Id: sunfloppy.c,v 1.1 1996/03/20 21:28:05 paul Exp $";

/* This file contains a driver for "floppy".
 *
 * The driver supports the following operations (using message format m2):
 *
 *    m_type      DEVICE    PROC_NR     COUNT    POSITION  ADRRESS
 * ----------------------------------------------------------------
 * |  DEV_READ  | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DEV_WRITE  | device  | proc nr |  bytes  |  offset | buf ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DEV_IOCTL  | device  |         |         |         |         |
 * |------------+---------+---------+---------+---------+---------|
 * |SCATTERED_IO| device  | proc nr | requests|         | iov ptr |
 * |------------+---------+---------+---------+---------+---------|
 * | DISK_RO	| device  |         |         |         |         |
 * ----------------------------------------------------------------
 * 
 *
 * The file contains one entry point:
 *
 *   floppy_task:	main entry when system is brought up
 *
 */

#include "kernel.h"
#include <minix/callnr.h>
#include <minix/com.h>
#include <minix/boot.h>
#include <sun/syscall.h>
#include <sys/ioctl.h>
#include "driver.h"

#define NR_DEVICES 8

PRIVATE int f_device;		/* current device */

FORWARD _PROTOTYPE( struct device *f_prepare, (int device) );
FORWARD _PROTOTYPE( int f_schedule, (int proc_nr, struct iorequest_s *iop) );
FORWARD _PROTOTYPE( int f_do_open, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( int f_do_ioctl, (struct driver *dp, message *m_ptr) );
FORWARD _PROTOTYPE( void f_init, (void) );


/* Entry points to this driver. */
PRIVATE struct driver f_dtab = {
  no_name,	/* current device's name */
  f_do_open,	/* open or mount */
  do_nop,	/* nothing on a close */
  f_do_ioctl,	/* ioctl */
  f_prepare,	/* prepare for I/O on a given minor device */
  f_schedule,	/* do the I/O */
  nop_finish,	/* schedule does the work, no need to be smart */
  nop_cleanup	/* nothing's dirty */
};


/*===========================================================================*
 *				floppy_task				     * 
 *===========================================================================*/
PUBLIC void floppy_task()
{
  driver_task(&f_dtab);
}

/*===========================================================================*
 *				f_prepare				     *
 *===========================================================================*/
PRIVATE struct device *f_prepare(device)
int device;
{
/* Prepare for I/O on a device. */

  if (device < 0 || device >= NR_DEVICES) return(NIL_DEV);
  f_device = device;

  return (struct device *)1;     /* A bit ugly */
}


/*===========================================================================*
 *				f_schedule				     * 
 *===========================================================================*/
PRIVATE int f_schedule(proc_nr, iop)
int proc_nr;			/* process doing the request */
struct iorequest_s *iop;	/* pointer to read or write request */
{
  int opcode;
  int r, errors = 0;
  phys_bytes user_phys;
  so_sigset_t savemask;

  /* Type of request */
  opcode = iop->io_request & ~OPTIONAL_IO;

  /* Determine address where data is to go or to come from. */
  user_phys = numap(proc_nr, (vir_bytes) iop->io_buf,
		    (vir_bytes) iop->io_nbytes);
  if (user_phys == 0) return(iop->io_nbytes = EINVAL);

  /* With SunOS disks (disk files) we can simply lseek() to the correct
   * position.
   */
  if ( (r = SunOS(SYS_lseek, DISK_FD + f_device, iop->io_position, 0)) == -1)
  	return(iop->io_nbytes = EIO);

  /* Next we can read or write a block by using SunOS's read and write. lock()
   * is needed incase the read or write gets interrupted by an alarm signal.
   */
  set_protect(user_phys, iop->io_nbytes, PROT_READ_WRITE);          
  lock(&savemask);	
  if (opcode == DEV_READ)	
  	r = SunOS(SYS_read, DISK_FD + f_device, user_phys, iop->io_nbytes);
  else
  	r = SunOS(SYS_write, DISK_FD + f_device, user_phys, iop->io_nbytes);
  restore(&savemask);
  set_protect(user_phys, iop->io_nbytes, PROT_NONE);
  
  /* This extra case is for when we wish to make the SunOS disk files
   * read only. Even though they may mounted as read only from Minix's point
   * of view, it will still be written to when updating times etc.
   */
  if (r != iop->io_nbytes && !(opcode == DEV_WRITE && errno == 9)) {
  	 return iop->io_nbytes = EIO;
  }
  return(iop->io_nbytes = OK);
}


PRIVATE int f_do_open(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
/* Check device number on open.  
 */

  if (f_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

  return(OK);
}


PRIVATE int f_do_ioctl(dp, m_ptr)
struct driver *dp;
message *m_ptr;
{
  int fcntl_ret;
  unsigned int ro;
  phys_bytes get_phys;

  if (f_prepare(m_ptr->DEVICE) == NIL_DEV) return(ENXIO);

  if (m_ptr->REQUEST == FIOISRDONLY) {
        fcntl_ret = SunOS(SYS_fcntl, DISK_FD + f_device, 3 /*F_GETFL*/, 0);
	if (fcntl_ret == -1) {
	        return(EIO);
	}

	ro = ((fcntl_ret & 03) == 0);   /* Read only if write bit not set */

        get_phys = numap(m_ptr->m_source, (vir_bytes) m_ptr->ADDRESS,
			 sizeof(ro));
        if (get_phys == 0) return(EINVAL);
        phys_copy(vir2phys(&ro), get_phys, (phys_bytes) sizeof(ro));

  } else {
        /* Not implemented. */
        return(ENOTTY);
  }
  return(OK);
}
