static char rcsid[] = "$Id$";

/* This file contains the keyboard driver for SunOS Minix. Input is received
 * when SunOS generates SIGIO. A pair of SunOS file descriptors is associated
 * with each Minix terminal, one for input and one for output. These file
 * descriptors may be either the local terminal or a socket for remote logins.
 */
 
#include "kernel.h"
#include <termios.h>
#include <sgtty.h>
#include <minix/callnr.h>
#include <minix/com.h>
#include "tty.h"
#include "proc.h"
#include <sun/syscall.h>
#include <sun/signal.h>

#define KB_IN_BYTES	 1024	/* size of keyboard input buffer */

#define min(x,y) ((x) < (y) ? (x) : (y))

#define kb_addr(tp)	   (&kb_lines[tty_line(tp)])

/* Keyboard structure, 1 per window. */
struct kb_s {
    char *ihead;			/* next free spot in input buffer */
    char *itail;			/* scan code to return to TTY */
    int icount;			        /* # codes in buffer */

    char ibuf[KB_IN_BYTES];	        /* input buffer */
};


PRIVATE struct kb_s kb_lines[NR_CONS];


FORWARD _PROTOTYPE( void kb_read, (tty_t *tp)                       );
FORWARD _PROTOTYPE( int hw_read, (tty_t *tp, char *buffer, int len) );
FORWARD _PROTOTYPE( void input_flush, (tty_t *tp)                   );
FORWARD _PROTOTYPE( int func_key, (tty_t *tp, int scode)            );
FORWARD _PROTOTYPE( void reboot, (void)                             );


/*==========================================================================*
 *				kb_read					    *
 *==========================================================================*/
PRIVATE void kb_read(tp)
tty_t *tp;
{
/* Process characters from the circular keyboard buffer. */

    struct kb_s *kb;
    int scode;
    char ch;
    so_sigset_t savemask;
    
    kb = kb_addr(tp);
    
    while (kb->icount > 0) {
	scode = *kb->itail++;			/* take one key scan code */
	if (kb->itail == kb->ibuf + KB_IN_BYTES) kb->itail = kb->ibuf;
	lock(&savemask);
	kb->icount--;
	restore(&savemask);

	/* Function keys are being used for debug dumps. */
	if (func_key(tp, scode)) continue;

	ch = scode;
	(void) in_process(tp, &ch, 1);
    }
}


/*===========================================================================*
 *				keyboard			     	     *
 *===========================================================================*/
PUBLIC void keyboard()
{
    /* The function is called upon receiving a SunOS SIGIO signal. We
     * must locate which file descriptor caused to signal and read its
     * input. It is possible that more than one character is available
     * from a file descriptor or that more than one file descriptor is
     * ready. A simple polling method is used at the moment, although
     * using SunOS's select() system call may be an advantage.  */
   
    struct kb_s *kb;
    tty_t *tp;
    int num_read, poll;
    int to_read;
    so_sigset_t savemask;

    lock(&savemask);

    for (poll = 0; poll < NR_CONS; poll++) {
	tp = &tty_table[poll];
	kb = kb_addr(tp);  
	if (kb->icount < KB_IN_BYTES) {
	    to_read = min(KB_IN_BYTES - kb->icount,
			  KB_IN_BYTES - (kb->ihead - kb->ibuf));
	    num_read = hw_read(tp, kb->ihead, to_read);
	    if (num_read > 0) {
		tp->tty_events = 1;
		force_timeout();
		kb->icount += num_read;
		kb->ihead += num_read;
		if (kb->ihead == kb->ibuf + KB_IN_BYTES) {
		    kb->ihead = kb->ibuf;
		    if (kb->icount < KB_IN_BYTES) {
			num_read = hw_read(tp, kb->ihead,
					   KB_IN_BYTES - kb->icount);
			if (num_read > 0) {
			    kb->icount += num_read;
			    kb->ihead += num_read;
			    /*
			     * Read at least 1 byte the first time; ihead
			     * cannot wraparound this time.
			     */
			}
		    } /* if space in input buffer for a second read */
		} /* if first read wraps buffer */
	    } /* if first read returns at least one char */
	} /* if space in input buffer */
	input_flush(tp);
    } /* for loop checking consoles */
    
    restore(&savemask);
}


PRIVATE int hw_read(tp, buffer, len)
tty_t *tp;
char *buffer;
int len;
{
    return SunOS(SYS_read, TERMINAL_FD + tty_line(tp) * 2, buffer, len);
}


PRIVATE void input_flush(tp)
tty_t *tp;
{
    static char flushbuf[32];

    while (hw_read(tp, flushbuf, sizeof(flushbuf))) {
	;
    }
}


/*===========================================================================*
 *				kb_init					     *
 *===========================================================================*/
PUBLIC void kb_init(tp)
tty_t *tp;
{
/* Initialize the keyboard driver. */
    register struct kb_s *kb;

    /* Input function. */
    tp->tty_devread = kb_read;
    kb = kb_addr(tp);

    kb->ihead = kb->itail = kb->ibuf;
    kb->icount = 0;
}


/*===========================================================================*
 *				func_key				     *
 *===========================================================================*/
PRIVATE int func_key(tp, scode)
tty_t *tp;
int scode;			/* scan code for a function key */
{
/* This procedure traps function keys for debugging and control purposes. */

    switch(scode) {
/* The first 5 options are commented out since they use valuable key codes. */
/*
   	case 1:	 p_dmp();	 break;
        case 2:	 map_dmp();	 break;
        case 4:  tty_dmp(); 	 break;
        case 5:  sigchar(&tty[CONSOLE], SIGINT);
		 break;
        case 6:  sigchar(&tty[CONSOLE], SIGKILL);
		 break;
*/
	case 0x1f: /* ^_, I'm desperate */
	         p_dmp(); map_dmp(); break;
	/* this is a special case for handling terminals that hangup */
	case 0xff: hangup(tty_line(tp));
		 break;
	default: return (FALSE);
    }
    return(TRUE);
}


/*==========================================================================*
 *				reboot					    *
 *==========================================================================*/
PRIVATE void reboot()
{
    SunOS(SYS_exit,0);
}


/*==========================================================================*
 *				wreboot					    *
 *==========================================================================*/
PUBLIC void wreboot(how)
int how;      /* Unused in smx */
{
    printf("Returning to SunOS\n");
    reboot();
}



