/*
 * This file contains the majority of the code for handling
 * SunOS Minix's memory protection. There are three levels
 * of protection: level 0 ('none') - no protection, level 1 
 * ('half') - read only text segments, level 2 ('full') - full
 * protection. With full protection, a process may not access memory
 * outside its own address space, with the exception of a small part
 * of the kernel (eg. the interrupt stack). Level 1 is
 * the default, whereas level 2 should only be used to 
 * detect the situation where a process is corrupting
 * memory. 
 * The file mpxSUN.s has been altered slightly so that
 * memory protection is switched when an interrupt (context
 * switch etc) occurs. Also, an extra message type for use
 * between MM and the system task has been added to allow
 * MM to relocate a program. Similar changes have been made
 * to copySUN.s and sunfloppy.c.
 *
 * The main entry points are:
 *	protect_init:		initialise protection info
 *	set_protect:		set protection on a region
 *	mprotect:		wrapping for mprotect(2)
 *	entering_kernel:	called when switching kernel prot on
 *	leaving_kernel:		called when switching kernel prot off
 *	flipclicks:		wrapper for the real flipclicks
 *	zeroclicks:		wrapper for the real zeroclicks
 *	copyclicks:		wrapper for the real copyclicks
 *	phys_copy:		wrapper for the real phys_copy
 *	set_mm_protect:		extends MM address space.
 */
 
#include "kernel.h"
#include "const.h"
#include <minix/com.h>
#include <sun/syscall.h>

#include "proc.h"

#define max(x,y) ((x) > (y) ? (x) : (y))

/* SunOS's protection codes for mprotect(2) */

#define PROT_READ        0x1       /* page can be read */
#define PROT_WRITE       0x2       /* page can be written */
#define PROT_EXEC        0x4       /* page can be executed */
#define PROT_NONE        0x0       /* page can not be accessed */
#define PROT_ALL (PROT_READ | PROT_WRITE | PROT_EXEC)
#define PROT_READ_EXEC (PROT_READ | PROT_EXEC)

#define SEGMENTS 10

/*
 * The following structures are for storing information about the
 * various segments of memory and their appropriate protection.
 * With full protection, it is necessary to identify the parts
 * of the kernel that can be safely made nonaccessable
 * along with those that must remain read/write/executable
 * in case an interrupt (a SunOS signal) occurs. These areas
 * include the interrupt handling code, the kernel stack and
 * the following 'pinfo' structure.
 */
 
struct seg_info {
	phys_bytes begin, end;
	int prot;
};

struct prot_info {
	phys_bytes begin;
	int len;
	int prot;
};

struct protect_info {
	phys_bytes kernel_base;
	phys_bytes kernel_text;
	phys_bytes kernel_data;
	struct prot_info prot_info[SEGMENTS];
	int numsegs;
	int in_kernel;
	phys_bytes mm_ext_base;
	phys_bytes mm_ext_len;
	int mm_ext_prot;
} pinfo;

int prot_lev;               /* the protection level specified by build */
                            /* 'prot_lev' MUST FOLLOW DIRECTLY AFTER 'pinfo' */


extern void mpxSP_start(), mpxSP_end();

FORWARD _PROTOTYPE( void protect_start, (void)                             );
FORWARD _PROTOTYPE( void protect_end, (void)                               );
FORWARD _PROTOTYPE( void kernel_on, (void)                                 );
FORWARD _PROTOTYPE( void kernel_off, (void)                                );
FORWARD _PROTOTYPE( void process_on, (struct proc *p)                      );
FORWARD _PROTOTYPE( void process_off, (struct proc *p)                     );


struct seg_info segs[] = {
	{0, 0, PROT_ALL},      /* entry for kernel stack */
	{(phys_bytes)protect_start, (phys_bytes)protect_end, PROT_READ_EXEC},
	{(phys_bytes)mpxSP_start, (phys_bytes)mpxSP_end, PROT_READ_EXEC},
	{(phys_bytes)&pinfo, (phys_bytes)&pinfo + sizeof(pinfo) + sizeof(prot_lev),
	                 PROT_READ},
};

#define SEGS_SIZE (sizeof(segs) / sizeof(struct seg_info))

static char rcsid[] = "$Id: sunprotect.c,v 1.1 1996/03/20 21:28:13 paul Exp $";

/*===========================================================================*
 *                             protect_init                                  * 
 *===========================================================================*/
PUBLIC void protect_init(kernel_base, kernel_text, kernel_data, mem_len)
phys_bytes kernel_base;
phys_clicks kernel_text;
phys_clicks kernel_data;
phys_bytes  mem_len;
{
/* 
 * Determine where the various memory segments are and give them their
 * initial protection values. The actual work done will depend on the
 * user specified memory protection level.
 */
    
    phys_bytes kernel_end;      /* first click after kernel */
    phys_bytes next_click;
    int i, j;
    register struct proc *rp;
    
    debug_str("Protection level ");
    debug_int(prot_lev);
    debug_char('\n');

    if (prot_lev == 2){			
	pinfo.kernel_base = kernel_base;
	pinfo.kernel_text = kernel_text << CLICK_SHIFT;
	pinfo.kernel_data = kernel_data << CLICK_SHIFT;
	pinfo.in_kernel = 1;
	kernel_end = pinfo.kernel_base + pinfo.kernel_text + pinfo.kernel_data;
	
	/*
	 * Add seg entry on kernel stack.
	 */			
	segs[0].end = upclick(getksp());  /* Should already be click aligned */
	segs[0].begin = segs[0].end - K_STACK_BYTES;
	
	/*
	 * Align address ranges to the enclosing clicks
	 */
	for (i = 0 ; i < SEGS_SIZE ; i++) {
	    segs[i].begin = downclick(segs[i].begin);
	    segs[i].end = upclick(segs[i].end);
	}
	
	/*
	 * Sort segments by beginning address
	 */
	{
	    int done = 0;
	    struct seg_info temp;
	    
	    while (!done) {
		done = 1;
		for (i=0 ; i < SEGS_SIZE-1 ; i++) {
		    if (segs[i].begin > segs[i+1].begin) {
			temp = segs[i];
			segs[i] = segs[i+1];
			segs[i+1] = temp;
			done = 0;
		    }
		}
	    }
	}
	
	/*
	 * Create entries in pinfo based on the segments that need special
	 * treatment as described in segs.
	 */
	j = -1;      /* j is the index into pinfo */
	next_click = pinfo.kernel_base;
	for (i = 0 ; i < SEGS_SIZE ; i++) {
	    /*
	     * Each time around the loop deal with an element of segs.
	     */
	    if (segs[i].begin > next_click) {
		/*
		 * There's a gap between this special segment
		 * and the previous one - fill it with a no go area.
		 */
		j++;
		pinfo.prot_info[j].begin = next_click;
		pinfo.prot_info[j].len = segs[i].begin - next_click;
		pinfo.prot_info[j].prot = PROT_NONE;
	    }

	    if (segs[i].begin < next_click) {
		/*
		 * An overlap - merge into a single segment,
		 * with protection as weak as required.
		 */
		pinfo.prot_info[j].len = max(pinfo.prot_info[j].len,
					     segs[i].end - pinfo.prot_info[j].begin);
		pinfo.prot_info[j].prot |= segs[i].prot;
	    } else {
		/*
		 * No overlap - create a new segment
		 */
		j++;
		pinfo.prot_info[j].begin = segs[i].begin;
		pinfo.prot_info[j].len = segs[i].end - segs[i].begin;
		pinfo.prot_info[j].prot = segs[i].prot;
	    }
	    next_click = pinfo.prot_info[j].begin + pinfo.prot_info[j].len;
	}
	
	/*
	 * Is there a gap at the end?
	 */
	if (next_click != kernel_end) {
	    j++;
	    pinfo.prot_info[j].begin = next_click;
	    pinfo.prot_info[j].len = kernel_end - next_click;
	    pinfo.prot_info[j].prot = PROT_NONE;
	}

	pinfo.numsegs = j+1;
	
	/*
	 * Dump the pinfo struct if debugging is on
	 */
	debug_str("kbase: "); debug_int(pinfo.kernel_base);
	debug_str("\nktext: "); debug_int(pinfo.kernel_text);
	debug_str("\nkdata: "); debug_int(pinfo.kernel_data);
	for (i=0 ; i < pinfo.numsegs ; i++) {
	    debug_str("\nbegin: "); debug_int(pinfo.prot_info[i].begin);
	    debug_str("\nlen: "); debug_int(pinfo.prot_info[i].len);
	    debug_str("\nprot: "); debug_int(pinfo.prot_info[i].prot);
	}
	debug_char('\n');

	/*
	 * Now that the data structures are set up, setup the current
	 * protection so that there is normal access to the kernel, and
	 * no access to the rest of the SunOS Minix address space.
	 */
	kernel_on();
	mprotect(kernel_end, mem_len - (pinfo.kernel_text + pinfo.kernel_data),
		 PROT_NONE);
		 
    } else if (prot_lev == 1) {
    
    /* 
     * For half protection, we need to make the kernel, MM, FS and init text
     * segments read-exec only and the remainder of the address space above init
     * completely non accessable.
     */
     
        mprotect(kernel_base,  kernel_text << CLICK_SHIFT, PROT_READ_EXEC);
        for (rp = BEG_SERV_ADDR; rp <= BEG_USER_ADDR; rp++){
            mprotect(rp->p_map[T].mem_phys << CLICK_SHIFT,
		     rp->p_map[T].mem_len << CLICK_SHIFT,
		     PROT_READ_EXEC);
        }
        kernel_end = (BEG_USER_ADDR->p_map[S].mem_phys + 
		      BEG_USER_ADDR->p_map[S].mem_len) << CLICK_SHIFT;
        mprotect(kernel_end, mem_len + code_base - kernel_end, PROT_NONE);	
    }       
}



/*===========================================================================*
 *                                set_protect                                * 
 *===========================================================================*/
PUBLIC void set_protect(start, len, prot)
phys_bytes start;
int len;
int prot;
{
/*
 * The the protection for an area of memory. This may involve rounding to a
 * click/page boundary.
 */
    phys_bytes prot_start;
    int prot_len;
    
    if (prot_lev == 2){ /* only for full protection */
	/*
	 * If the area is part of the kernel then just quietly ignore this
	 * request as the required access should already be possible.
	 */
	if (start > pinfo.kernel_base && start < pinfo.kernel_base +
	    pinfo.kernel_text + pinfo.kernel_data) {
	    return;
	}
	
	/*
	 * Click align the area whose protection is to be changed.
	 */
	prot_start = downclick(start);
	prot_len = upclick(start + len) - prot_start;
	mprotect(prot_start, prot_len, prot);
    }
}



/*===========================================================================*
 *                            protect_start                                  * 
 *===========================================================================*/

/*
 * DO NOT MOVE THIS FUNCTION.
 */
 
static void protect_start() {}



/*===========================================================================*
 *                                 mprotect                                  * 
 *===========================================================================*/
PUBLIC void mprotect(addr, len, prot)
phys_bytes addr;
int len;
int prot;
{
/*
 * This is a wrapper for the SunOS mprotect(2) system call.
 */
    
    typedef char *caddr_t;

    if (SunOS(SYS_mprotect, (caddr_t) addr, len, prot | 8) == -1) {
	printk("mprotect failed: address 0x%x, len 0x%x, prot 0%o\n",
	       addr, len, prot);
    }
}


/*===========================================================================*
 *                              entering_kernel                              * 
 *===========================================================================*/
PUBLIC void entering_kernel()
{
/*
 * 'entering_kernel' is called from within 'mpxSUN.s' to make the entire
 * kernel address space accessable and the process's address space 
 * inaccessable.
 */
 
    if (prot_lev == 2){
	/*
	 * Note - only need to change protection if we were outside the kernel.
	 */
	if (!pinfo.in_kernel) {
	    kernel_on();
	    process_off(proc_ptr);
	    pinfo.in_kernel = 1;
	    
	    /* 
	     * if the MM has an extended address space, make it 
	     * inaccessable.
	     */
	    
	    if (proc_number(proc_ptr) == MM_PROC_NR && 
		pinfo.mm_ext_prot != PROT_NONE)
		mprotect(pinfo.mm_ext_base, pinfo.mm_ext_len, PROT_NONE);
	    
	}
    }
}



/*===========================================================================*
 *                             leaving_kernel                                * 
 *===========================================================================*/
PUBLIC void leaving_kernel()
{
/*
 * 'leaving_kernel' is called from within 'mpxSUN.s' to make the entire
 * kernel address space inaccessable and the process's address space 
 * accessable.
 */
    if (prot_lev == 2){
	/*
	 * Note - only change protection if execution is leaving the kernel.
	 */
	if (!istaskp(proc_ptr) && !isidlehardware(proc_number(proc_ptr))) {
	    pinfo.in_kernel = 0;
	    process_on(proc_ptr);
	
	    /* 
	     * if the MM has an extended address space, make it 
	     * accessable.
	     */
	    if (proc_number(proc_ptr) == MM_PROC_NR && 
		pinfo.mm_ext_prot != PROT_NONE)
		mprotect(pinfo.mm_ext_base, pinfo.mm_ext_len, 
			 pinfo.mm_ext_prot);
		
	    kernel_off();
	}
    }
}



/*===========================================================================*
 *                                 kernel_on                                 * 
 *===========================================================================*/
PRIVATE void kernel_on()
{
    /*
     * Make the text segment rx, and the data segment rwx
     */
    mprotect(pinfo.kernel_base, pinfo.kernel_text, PROT_READ_EXEC);
    mprotect(pinfo.kernel_base + pinfo.kernel_text, pinfo.kernel_data,
	     PROT_ALL);
}



/*===========================================================================*
 *                                 kernel_off                                * 
 *===========================================================================*/
PRIVATE void kernel_off()
{
    int i;

    /*
     * Prepare for leaving the kernel - change protection as per
     * the info in pinfo
     */
    for (i = 0 ; i < pinfo.numsegs ; i++) {
	if (pinfo.prot_info[i].prot != PROT_READ_EXEC &&
	    pinfo.prot_info[i].prot != PROT_READ_WRITE){
	    mprotect(pinfo.prot_info[i].begin, pinfo.prot_info[i].len,
		     pinfo.prot_info[i].prot);
	}
    }
}



/*===========================================================================*
 *                                process_on                                 * 
 *===========================================================================*/
PRIVATE void process_on(p)
struct proc *p;
{
    /*
     * Text segment must be executable, data segment (contains stack)
     * must be writable.
     */
    mprotect(p->p_map[T].mem_phys << CLICK_SHIFT,
	     p->p_map[T].mem_len << CLICK_SHIFT, PROT_READ_EXEC);
    if (p->p_map[S].mem_len != 0) {
	/*
	 * Assume data, gap, and stack are contiguous
	 */
	mprotect(p->p_map[D].mem_phys << CLICK_SHIFT,
		 (p->p_map[S].mem_phys << CLICK_SHIFT) -
		 (p->p_map[D].mem_phys << CLICK_SHIFT) +
		 (p->p_map[S].mem_len << CLICK_SHIFT), PROT_ALL);
    } else {
	/*
	 * No separate stack
	 */
	mprotect(p->p_map[D].mem_phys << CLICK_SHIFT,
		 p->p_map[D].mem_len << CLICK_SHIFT, PROT_ALL);
    }					 		
}


/*===========================================================================*
 *                              process_off                                  * 
 *===========================================================================*/
PRIVATE void process_off(p)
struct proc *p;
{
    /*
     * Turn off access to text segment and data segment.
     */
    mprotect(p->p_map[T].mem_phys << CLICK_SHIFT,
	     p->p_map[T].mem_len << CLICK_SHIFT, PROT_NONE);
    if (p->p_map[S].mem_len != 0) {
	/*
	 * Assume data, gap, and stack are contiguous
	 */
	mprotect(p->p_map[D].mem_phys << CLICK_SHIFT,
		 (p->p_map[S].mem_phys << CLICK_SHIFT) -
		 (p->p_map[D].mem_phys << CLICK_SHIFT) +
		 (p->p_map[S].mem_len << CLICK_SHIFT), PROT_NONE);
    } else {
	/*
	 * No separate stack
	 */
	mprotect(p->p_map[D].mem_phys << CLICK_SHIFT,
		 p->p_map[D].mem_len << CLICK_SHIFT, PROT_NONE);
    }			 		
}



/*===========================================================================*
 *                              protect_end                                  * 
 *===========================================================================*/

/*
 * DO NOT MOVE THIS FUNCTION.
 */

static void protect_end() {}



/*===========================================================================*
 *                              flipclicks                                   * 
 *===========================================================================*/
PUBLIC void flipclicks(addr1, addr2, numclicks)
phys_clicks addr1, addr2, numclicks;
{
/*
 * Wrapper for the copying function in copySUN.s, to ensure that
 * the protections are appropriate during copying.
 */

    set_protect(addr1 << CLICK_SHIFT, numclicks << CLICK_SHIFT,
		PROT_READ_WRITE);
    set_protect(addr2 << CLICK_SHIFT, numclicks << CLICK_SHIFT,
		PROT_READ_WRITE);
    real_flipclicks(addr1, addr2, numclicks);
    set_protect(addr1 << CLICK_SHIFT, numclicks << CLICK_SHIFT,
		PROT_NONE);
    set_protect(addr2 << CLICK_SHIFT, numclicks << CLICK_SHIFT,
		PROT_NONE);
}



/*===========================================================================*
 *                               zeroclicks                                  * 
 *===========================================================================*/
PUBLIC void zeroclicks(addr, numclicks)
phys_clicks addr, numclicks;
{
/* 
 * Wrapper for the copying function in copySUN.s, to ensure that
 * the protections are appropriate during copying.
 */ 
    set_protect(addr << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_WRITE);
    real_zeroclicks(addr, numclicks);
    set_protect(addr << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_NONE);
}



/*===========================================================================*
 *                                copyclicks                                 * 
 *===========================================================================*/
PUBLIC void copyclicks(src, dest, numclicks)
phys_clicks src, dest, numclicks;
{
/*
 * Wrapper for the copying function in copySUN.s, to ensure that
 * the protections are appropriate during copying.
 */
    set_protect(src << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_READ);
    set_protect(dest << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_WRITE);
    real_copyclicks(src, dest, numclicks);
    set_protect(src << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_NONE);
    set_protect(dest << CLICK_SHIFT, numclicks << CLICK_SHIFT, PROT_NONE);
}



/*===========================================================================*
 *                                phys_copy                                  * 
 *===========================================================================*/
PUBLIC void phys_copy(src, dest, len)
phys_bytes src, dest, len;
{
/*
 * Wrapper for the copying function in copySUN.s, to ensure that
 * the protections are appropriate during copying.
 */
    set_protect(src, len, PROT_READ);
    set_protect(dest, len, PROT_WRITE);
    real_phys_copy(src, dest, len);
    set_protect(src, len, PROT_NONE);
    set_protect(dest, len, PROT_NONE);
}

/*===========================================================================*
 *                                set_mm_ext                                 * 
 *===========================================================================*/
PUBLIC void set_mm_ext(addr, len, prot)
phys_bytes addr;
int len;
int prot;
{
    pinfo.mm_ext_base = addr;
    pinfo.mm_ext_len = len;
    pinfo.mm_ext_prot = prot;
}



/*===========================================================================*/
