static const char *config_c_rcsid = "$Id: parse_config.c,v 1.2 1993/09/08 02:50:41 bjaspan Exp $";

#include <stdio.h>
#include <string.h>
#include <sys/types.h>
#include "pcmcia.h"
#include "pcmciad.h"
#include "lex_config.h"

struct devtype_name {
     char *name;
     enum devtypes type;
};

static struct devtype_name devtype_names[] = {
     "modem", MODEM,
     "ethernet", ETHERNET,
};
static int num_devtype_names = sizeof(devtype_names) /
	sizeof(struct devtype_name);

struct prod_map *prod_map;
struct dev_action *dev_actions[MAX_DEVTYPE];

static int parse_action(struct dev_action *action);
static int parse_prodmap(struct prod_map *devmap);
static int parse_devtype(char *s);

void syntax_error(char *msg, ...)
{
     fprintf(stderr, "syntax error, line %d: %s\n", config_lineno, msg);
}

int parse_configfile(FILE *f)
{
     extern FILE *yyin;
     int tok, devtype;
     struct dev_action *act, *act_tails[MAX_DEVTYPE];
     struct prod_map *pmap, *pmap_tail;

     yyin = f;

     config_lineno = 0;
     while ((tok = yylex()) != 0) {
	  switch (tok) {
	  case DEVICE:
	       pmap = (struct prod_map *) malloc(sizeof(struct prod_map));
	       if (pmap == NULL) {
		    fprintf(stderr, "Out of memory alocating prod_map.\n");
		    exit(1);
	       }
	       pmap->next = NULL;
	       if (prod_map == NULL)
		    prod_map = pmap_tail = pmap;
	       else {
		    pmap_tail->next = pmap;
		    pmap_tail = pmap;
	       }
	       if (parse_prodmap(pmap) < 0)
		    goto syntax_error;
	       break;
	  case IDENTIFIER:
	       devtype = parse_devtype(tokval.s);
	       if (devtype < 0)
		    goto syntax_error;

	       act = (struct dev_action *) malloc(sizeof(struct dev_action));
	       if (act == NULL) {
		    fprintf(stderr, "Out of memory allocating dev_action.\n");
		    exit(1);
	       }
	       act->busy = 0;
	       act->devtype = devtype;
	       act->next = NULL;
	       if (dev_actions[devtype] == NULL)
		    dev_actions[devtype] = act_tails[devtype] = act;
	       else {
		    act_tails[devtype]->next = act;
		    act_tails[devtype] = act;
	       }
	       
	       if (parse_action(act) < 0)
		    goto syntax_error;
	       break;
	  default:
	       syntax_error("Expected `device', device type, or EOF");
	       goto syntax_error;
	       break;
	  }
     }

     /* copy tmp to current config info */

     return 0;

syntax_error:
     return -1;
}

#define GETTOK(type,errmsg) if ((tok = yylex()) != type) { \
     syntax_error(errmsg); \
     goto syntax_error; \
     }

static int parse_action(struct dev_action *act)
{
     enum tokens tok;
     int io, mem;

     io = mem = 0;
     
     GETTOK(EQ, "expected `='");
     GETTOK(IRQ, "expected `irq'");
     GETTOK(INT, "expected integer");
     act->cfent.irq = tokval.n;
     GETTOK(COMMA, "expected `,'");

     GETTOK(IO, "expected `io'");
     while ((tok = yylex()) != COMMA) {
	  switch (tok) {
	  case NONE: /* what a hack */
	       continue;
	  case INT:
	       act->cfent.io_addrs[io] = tokval.n;
	       GETTOK(DASH, "expected `-'");
	       GETTOK(INT, "expected integer");
	       act->cfent.io_lens[io] = tokval.n -
		    act->cfent.io_addrs[io] + 1;
	       io++;
	       break;
	  default:
	       syntax_error("expected integer or `none'");
	       goto syntax_error;
	  }
     }
     
     GETTOK(MEM, "expected `mem'");
     while ((tok = yylex()) != COMMA) {
	  switch (tok) {
	  case NONE: /* what a hack */
	       continue;
	  case INT:
	       act->cfent.mem_caddrs[mem] = tokval.n;
	       GETTOK(DASH, "expected `-'");
	       GETTOK(INT, "expected integer");
	       act->cfent.mem_lens[mem] = tokval.n -
		    act->cfent.mem_caddrs[mem] + 1;
	       GETTOK(AT, "expected `@'");
	       GETTOK(INT, "expected integer");
	       act->cfent.mem_haddrs[mem] = tokval.n;
	       mem++;
	       break;
	  default:
	       syntax_error("expected integer or `none'");
	       goto syntax_error;
	  }
     }
     
     GETTOK(STRING, "expected string");
     act->init = strdup(tokval.s);
     GETTOK(COMMA, "expected `,'");
     GETTOK(STRING, "expected string");
     act->reset = strdup(tokval.s);

     act->cfent.ios = io;
     act->cfent.mems = mem;

     return 0;

syntax_error:
     if (act->init)
	  free(act->init);
     if (act->reset)
	  free(act->reset);
     return -1;
}

static int parse_prodmap(struct prod_map *devmap)
{
     enum tokens tok;

     GETTOK(STRING, "expected string");
     devmap->manufacturer = strdup(tokval.s);
     GETTOK(COMMA, "expected `,'");
     GETTOK(STRING, "expected string");
     devmap->prod_name = strdup(tokval.s);
     GETTOK(COMMA, "expected `,'");
     GETTOK(STRING, "expected string");
     devmap->addl_info1 = strdup(tokval.s);
     GETTOK(COMMA, "expected `,'");
     GETTOK(STRING, "expected string");
     devmap->addl_info2 = strdup(tokval.s);
     GETTOK(EQ, "expected `='");
     GETTOK(IDENTIFIER, "expected device type");
     devmap->devtype = parse_devtype(tokval.s);
     if (devmap->devtype < 0)
	  goto syntax_error;

     return 0;

syntax_error:
     if (devmap->manufacturer)
	  free(devmap->manufacturer);
     if (devmap->prod_name)
	  free(devmap->prod_name);
     if (devmap->addl_info1)
	  free(devmap->addl_info1);
     if (devmap->addl_info2)
	  free(devmap->addl_info2);
     return -1;
}

static int parse_devtype(char *s)
{
     int i;

     for (i = 0; i < num_devtype_names; i++)
	  if (strcmp(s, devtype_names[i].name) == 0)
	       return devtype_names[i].type;

     syntax_error("unknown device type %s", s);
     return -1;
}

char *unparse_devtype(enum devtypes type)
{
     int i;

     for (i = 0; i < num_devtype_names; i++)
	  if (type == devtype_names[i].type)
	       return devtype_names[i].name;

     return "UNKNOWN DEVICE TYPE";
}
