#include <stdio.h>
#include <sys/types.h>

#include "pcmcia.h"
#include "pcmciad.h"

static const char *parse_tuples_c = "$Id: parse_tuples.c,v 1.3 1993/09/07 20:52:52 bjaspan Exp $";

void parse_device(u_char *, int, struct pcic_sock *, int);
void parse_ver1(u_char *, int, struct pcic_sock *, int);
void parse_config(u_char *, int, struct pcic_sock *, int);
void parse_cfent(u_char *, int, struct pcic_sock *, struct
		 pcmcia_cfentry *, int);
void read_extended_speed(u_char *, int *);
const char *tuple_name(int), *dtype_name(int), *dsize_name(int),
     *dspeed_name(int);

#ifndef KERNEL
#define READ(ptr, len) read_tuple_data(mem_fd, ptr, len)

extern int mem_fd;
#endif

void read_tuple_data(int fd, u_char *ptr, int len)
{
     int i;
     u_short buf[BUFSIZ], *p;
     
     i = read(fd, (char *) buf, len*2);
     if (i < 0) {
	  perror("reading tuple memory");
	  exit(2);
     } else if (i != len*2) {
	  fprintf(stderr, "read %d of %d bytes\n", i, len*2);
	  exit(3);
     }

     p = buf;
     while (len--)
	  *ptr++ = (u_char) (*p++ & 0xff);
}

void parse_tuples(struct pcic_sock *sock, int print_tuples)
{
     u_char code, len, tbuf[256];
     int done;
     struct pcmcia_cfentry *cfent = sock->configs;
     
     done = 0;
     while (!done) {
	  READ(&code, 1);
	  if (code == CIS_NULL) {
	       if (print_tuples)
		    printf("NULL tuple\n");
	       continue;
	  }

	  READ(&len, 1);
	  if (print_tuples)
	       printf("%s (%d):\n", tuple_name(code), len);

	  if (code != CIS_NULL)
	       READ(tbuf, len);
	  switch (code) {
	  case CIS_NULL:
	       if (print_tuples)
		    printf("NULL\n");
	       break;
	  case CIS_END:
	       done = 1;
	       break;
	  case CIS_DEVICE:
	  case CIS_DEVICE_A:
	       if (!print_tuples)
		    break;
	       parse_device(tbuf, len, sock, print_tuples);
	       break;
	  case CIS_VER1:
	       parse_ver1(tbuf, len, sock, print_tuples);
	       break;
	  case CIS_CFG_INFO:
	       parse_config(tbuf, len, sock, print_tuples);
	       break;
	  case CIS_CFG_ENT:
	       if (cfent - sock->configs < MAX_CONFIGS)
		    parse_cfent(tbuf, len, sock, cfent, print_tuples);
	       cfent++;
	       break;
	  default:
	       if (print_tuples)
		    printf("\tskpping\n");
	       break;
	  }
     }
}

void parse_device(u_char *tbuf, int tlen, struct pcic_sock *sock,
		  int print_tuples)
{
     int i, idx, addr_units;
     
     i = 0;
     while (i < tlen) {
	  /* last info structure? */
	  if (tbuf[i] == 0xff)
	       break;
	  
	  /* device id */
	  idx = (tbuf[i] & CIS_DEVICE_TYPE) >> CIS_DEVICE_TYPE_SHIFT;
	  printf("\tType %s, ", dtype_name(idx));
	  printf("WPS %s, ", tbuf[i] & CIS_DEVICE_WPS ? "set" : "clear");
	  idx = tbuf[i] & CIS_DEVICE_SPEED;
	  printf("Speed %s, ", dspeed_name(idx));
	  
	  /* device size */
	  i++;
	  if (tbuf[i] != 0xff) {
	       addr_units = ((tbuf[i] & CIS_DEVICE_ADDRS) >>
			     CIS_DEVICE_ADDRS_SHIFT) + 1;
	       idx = tbuf[i] & CIS_DEVICE_SIZE;
	       printf("Size %s * %d", dsize_name(idx), addr_units);
	  } else {
	       printf("IGNORED");
	       /* ignore this device info entry */
	  }
	  
	  printf("\n");
	  
	  i++;
     }
}

void parse_ver1(u_char *tbuf, int len, struct pcic_sock *sock, int print_tuples)
{
     int i;
     
     i = 0;
     if (tbuf[i++] != 0x04) {
	  sock->valid = -1;
	  if (print_tuples)
	       fprintf(stderr, "Major version != 0x04\n");
	  return;
     }
     if (tbuf[i++] != 0x01) {
	  sock->valid = -1;
	  if (print_tuples)
	       fprintf(stderr, "Minor version != 0x01\n");
	  return;
     }
     strncpy(sock->manufacturer, &tbuf[i], 
	     sizeof(sock->manufacturer));
     i += strlen(&tbuf[i]) + 1;
     strncpy(sock->prod_name, &tbuf[i], sizeof(sock->prod_name));
     i += strlen(&tbuf[i]) + 1;
     strncpy(sock->addl_info1, &tbuf[i], sizeof(sock->addl_info1));
     i += strlen(&tbuf[i]) + 1;
     strncpy(sock->addl_info2, &tbuf[i], sizeof(sock->addl_info2));
     i += strlen(&tbuf[i]) + 1;
     if (print_tuples) {
	  printf("\tManufacturer: %s\n", sock->manufacturer);	    
	  printf("\tProduct name: %s\n", sock->prod_name);
	  printf("\tAddl info 1: %s\n", sock->addl_info1);
	  printf("\tAddl info 2: %s\n", sock->addl_info2);
     }
     if (tbuf[i] != 0xff) {
	  sock->valid = -1;
	  if (print_tuples)
	       fprintf(stderr, "Tuple not ended by 0xff!\n");
	  return;
     }
}

void parse_config(u_char *tbuf, int len, struct pcic_sock *sock, int
		  print_tuples)
{
     int i, rasz, rmsz;
     
     i = 0;
     rasz = (tbuf[i] & TPCC_RASZ) >> TPCC_RASZ_SHIFT;
     rmsz = (tbuf[i] & TPCC_RMSZ) >> TPCC_RMSZ_SHIFT;
     if (print_tuples)
	  printf("\tRASZ %d, RMSZ %d, ", rasz+1, rmsz+1);
     
     i++;
     sock->config_midx = (tbuf[i] & TPCC_LAST) >> TPCC_LAST_SHIFT;
     if (print_tuples)
	  printf("last idx %d, ", sock->config_midx);
     
     i++;
     sock->base_addr = 0;
     switch (rasz) {
     case 3:
	  sock->base_addr |= (tbuf[i+3] << 24);
     case 2:
	  sock->base_addr |= (tbuf[i+2] << 16);
     case 1:
	  sock->base_addr |= (tbuf[i+1] << 8);
     case 0:
	  sock->base_addr |= tbuf[i];
     }
     if (print_tuples)
	  printf("base addr 0x%08x\n", sock->base_addr);
     
     i += rasz + 1;
     sock->regmask[0] = sock->regmask[1] = 0;
     sock->regmask[2] = sock->regmask[3] = 0; 
     switch (rmsz) {
     case 15:
	  sock->regmask[3] |= (tbuf[i+15] << 24);
     case 14:
	  sock->regmask[3] |= (tbuf[i+14] << 16);
     case 13:
	  sock->regmask[3] |= (tbuf[i+13] << 8);
     case 12:
	  sock->regmask[3] |= tbuf[i+12];
     case 11:
	  sock->regmask[2] |= (tbuf[i+11] << 24);
     case 10:
	  sock->regmask[2] |= (tbuf[i+10] << 16);
     case 9:
	  sock->regmask[2] |= (tbuf[i+9] << 8);
     case 8:
	  sock->regmask[2] |= tbuf[i+8];
     case 7:
	  sock->regmask[1] |= (tbuf[i+7] << 24);
     case 6:
	  sock->regmask[1] |= (tbuf[i+6] << 16);
     case 5:
	  sock->regmask[1] |= (tbuf[i+5] << 8);
     case 4:
	  sock->regmask[1] |= tbuf[i+4];
     case 3:
	  sock->regmask[0] |= (tbuf[i+3] << 24);
     case 2:
	  sock->regmask[0] |= (tbuf[i+2] << 16);
     case 1:
	  sock->regmask[0] |= (tbuf[i+1] << 8);
     case 0:
	  sock->regmask[0] |= tbuf[i+0];
	  break;
     }
     if (print_tuples)
	  printf("\treg mask 0x%04x%04x%04x%04x, ",
		 sock->regmask[3], sock->regmask[2],
		 sock->regmask[1], sock->regmask[0]); 
     
     i += rmsz + 1;
     if (print_tuples)
	  printf("\n\t%d bytes in subtuples\n", len - i);
}

void parse_cfent(u_char *tbuf, int len, struct pcic_sock *sock, struct
		 pcmcia_cfentry *cfent, int print_tuples)
{
     int i, j, k, intface, ftrs;
     int pwr_desc, wait_scale, rdy_scale, rsv_scale;
     int io_block_len, io_block_size;
     int host_addr_p, addr_size, len_size;
     
     i = 0;
     intface = (tbuf[i] & TPCE_INDX_INT);
     cfent->idx = (tbuf[i] & TPCE_INDX_ENTRY);
     cfent->defp = (tbuf[i] & TPCE_INDX_DEF);
     if (print_tuples)
	  printf("\tEntry %d, %sdefault, %sinterface\n", cfent->idx,
		 cfent->defp ? "" : "not ", intface ? "" : "no ");
     if (intface) {
	  i++;
	  if (print_tuples)
	       printf("\ttype %d, BVD %d, WP %d, RdyBsy %d, "
		      "wait sig %d\n",
		      tbuf[i] & TPCE_IF_TYPE,
		      !!tbuf[i] & TPCE_IF_BVD,
		      !!tbuf[i] & TPCE_IF_WP,
		      !!tbuf[i] & TPCE_IF_RDYBSY,
		      !!tbuf[i] & TPCE_IF_MWAIT);
     }
     i++;
     
     ftrs = tbuf[i++];
     if (print_tuples)
	  printf("\tfeatures 0x%02x\n", ftrs);
     
     /* XXX skip all power description structures */
     for (j = 0; j < (ftrs & TPCE_FS_PWR); j++) {
	  pwr_desc = tbuf[i++];
	  /* for each struct, skip all parameter defns */
	  for (k = 0; k < 8; pwr_desc >>= 1, k++) {
	       if (pwr_desc & 0x01) {
		    /* skip bytes until non-ext found */
		    while (tbuf[i++] & 0x80)
			 ;
	       }
	  }
     }
     
     if (ftrs & TPCE_FS_TD) {
	  wait_scale = tbuf[i] & TPCE_FS_TD_WAIT;
	  rdy_scale = (tbuf[i] & TPCE_FS_TD_RDY) >>
	       TPCE_FS_TD_RDY_SHIFT;
	  rsv_scale = (tbuf[i] & TPCE_FS_TD_RSV) >>
	       TPCE_FS_TD_RSV_SHIFT;
	  i++;
	  if (wait_scale != 3) {
	       read_extended_speed(tbuf, &i);
	       if (print_tuples)
		    printf("\twait scale %d\n", wait_scale);
	  }
	  if (rdy_scale != 7) {
	       read_extended_speed(tbuf, &i);
	       if (print_tuples)
		    printf("\tReady/Busy scale %d\n", rdy_scale);
	  }
	  if (rsv_scale != 7) {
	       read_extended_speed(tbuf, &i);
	       if (print_tuples)
		    printf("\tReserved scale %d\n", rsv_scale);
	  }
     }
     
     if (ftrs & TPCE_FS_IO) {
	  cfent->iop = 1;
	  cfent->io_16 = tbuf[i] & TPCE_FS_IO_BUS16;
	  if (print_tuples)
	       printf("\tIO lines %x, bus8 %d, bus16 %d, range %d\n",
		      (tbuf[i] & TPCE_FS_IO_LINES),
		      !!(tbuf[i] & TPCE_FS_IO_BUS8),
		      !!cfent->io_16,
		      !!(tbuf[i] & TPCE_FS_IO_RANGE));
	  i++;
	  io_block_len = (tbuf[i] & TPCE_FS_IO_LEN) >>
	       TPCE_FS_IO_LEN_SHIFT;
	  io_block_size = (tbuf[i] & TPCE_FS_IO_SIZE) >>
	       TPCE_FS_IO_SIZE_SHIFT;
	  cfent->ios = (tbuf[i] & TPCE_FS_IO_NUM) + 1;
	  if (print_tuples)
	       printf("\t# ranges %d, length size %d, "
		      "addr size %d\n", cfent->ios,
		      io_block_len, io_block_size); 
	  i++;
	  for (j = 0; j < cfent->ios; j++) {
	       cfent->io_addrs[j] = cfent->io_lens[j] = 0;
	       switch (io_block_size) {
	       case 3:
		    cfent->io_addrs[j] |= tbuf[i+3] << 24;
		    cfent->io_addrs[j] |= tbuf[i+2] << 16;
	       case 2:
		    cfent->io_addrs[j] |= tbuf[i+1] << 8;
	       case 1:
		    cfent->io_addrs[j] |= tbuf[i];
		    break;
	       }
	       i += io_block_size + (io_block_size == 3 ? 1
				     : 0);
	       switch (io_block_len) {
	       case 3:
		    cfent->io_lens[j] |= tbuf[i+3] << 24;
		    cfent->io_lens[j] |= tbuf[i+2] << 16;
	       case 2:
		    cfent->io_lens[j] |= tbuf[i+1] << 8;
	       case 1:
		    cfent->io_lens[j] |= tbuf[i];
		    break;
	       }
	       cfent->io_lens[j]++;
	       i += io_block_len + (io_block_len == 3 ? 1
				    : 0);
	       
	       if (print_tuples)
		    printf("\taddr %08x, len %d\n",
			   cfent->io_addrs[j],
			   cfent->io_lens[j]);
	  }
     }
     
     if (ftrs & TPCE_FS_IRQ) {
	  cfent->irqp = 1;
	  if (print_tuples)
	       printf("\tIRQ: share %d, pulse %d, level %d, ",
		      !!(tbuf[i] & TPCE_FS_IRQ_SHARE),
		      !!(tbuf[i] & TPCE_FS_IRQ_PULSE),
		      !!(tbuf[i] & TPCE_FS_IRQ_LEVEL));
	  if (tbuf[i] & TPCE_FS_IRQ_MASK) {
	       cfent->irq_mask = (tbuf[i+2] << 8) + tbuf[i+1];
	       if (print_tuples)
		    printf("VEND %d, BERR %d, IOCK %d, NMI %d\n"
			   "\t    mask 0x%04x\n",
			   !!(tbuf[i] & TPCE_FS_IRQ_VEND),
			   !!(tbuf[i] & TPCE_FS_IRQ_BERR),
			   !!(tbuf[i] & TPCE_FS_IRQ_IOCK),
			   !!(tbuf[i] & TPCE_FS_IRQ_NMI),
			   cfent->irq_mask);
	       i += 2;
	  } else {
	       cfent->irq = tbuf[i] & TPCE_FS_IRQ_IRQN;
	       if (print_tuples)
		    printf("IRQ %d\n", cfent->irq);
	  }
	  
	  i++;
     }
     
     if (ftrs & TPCE_FS_MEM) {
	  cfent->memp = 1;
	  switch ((ftrs & TPCE_FS_MEM) >> TPCE_FS_MEM_SHIFT) {
	  case 1:
	       cfent->mems = 1;
	       cfent->mem_lens[0] = (tbuf[i+1] << 8) + tbuf[i];
	       cfent->mem_lens[0] <<= 8;
	       printf("\tmem: len %d\n", cfent->mem_lens[0]);
	       
	       break;
	  case 2:
	       cfent->mems = 1;
	       cfent->mem_lens[0] = (tbuf[i+1] << 8) + tbuf[i];
	       cfent->mem_caddrs[0] = cfent->mem_haddrs[0] =
		    (tbuf[i+3] << 8) + tbuf[i+2];

	       cfent->mem_lens[0] <<= 8;
	       cfent->mem_caddrs[0] <<= 8;
	       
	       if (print_tuples)
		    printf("\tmem: len %d, addr %d\n",
			   cfent->mem_lens[0],
			   cfent->mem_caddrs[0]); 
	       break;
	  case 3:
	       host_addr_p = tbuf[i] & TPCE_FS_MEM_HOST;
	       addr_size = (tbuf[i] & TPCE_FS_MEM_ADDR) >>
		    TPCE_FS_MEM_ADDR_SHIFT;
	       len_size = (tbuf[i] & TPCE_FS_MEM_LEN) >>
		    TPCE_FS_MEM_LEN_SHIFT;
	       cfent->mems = (tbuf[i] & TPCE_FS_MEM_WINS) + 1;
	       if (print_tuples)
		    printf("\tmem (%x): host %d, addr size %d, len "
			   "size %d, # wins %d\n", tbuf[i],
			   !!host_addr_p, addr_size,
			   len_size, cfent->mems); 
	       i++;
	       for (j = 0; j < cfent->mems; j++) {
		    cfent->mem_lens[j] = 0;
		    cfent->mem_caddrs[j] = 0;
		    cfent->mem_haddrs[j] = 0;  
		    switch (len_size) {
		    case 3:
			 cfent->mem_lens[j] |= (tbuf[i+2] << 16);
		    case 2:
			 cfent->mem_lens[j] |= (tbuf[i+1] << 8);
		    case 1:
			 cfent->mem_lens[j] |= tbuf[i];
		    }
		    i += len_size;
		    switch (addr_size) {
		    case 3:
			 cfent->mem_caddrs[j] |= (tbuf[i+2] << 16);
		    case 2:
			 cfent->mem_caddrs[j] |= (tbuf[i+1] << 8);
		    case 1:
			 cfent->mem_caddrs[j] |= tbuf[i];
		    }
		    i += addr_size;
		    if (host_addr_p) {
			 switch (addr_size) {
			 case 3:
			      cfent->mem_haddrs[j] |=
				   (tbuf[i+2] << 16); 
			 case 2:
			      cfent->mem_haddrs[j] |=
				   (tbuf[i+1] << 8); 
			 case 1:
			      cfent->mem_haddrs[j] |=
				   tbuf[i]; 
			 }
			 i += addr_size;
		    }

		    cfent->mem_lens[j] <<= 8;
		    cfent->mem_caddrs[j] <<= 8;
		    cfent->mem_haddrs[j] <<= 8;
		    
		    if (print_tuples)
			 printf("\t\twin %d: len %d, card addr "
				"%x, host addr %x\n", j,
				cfent->mem_lens[j],
				cfent->mem_caddrs[j],
				cfent->mem_haddrs[j]);
	       }
	  }
     }
}

void read_extended_speed(u_char *tbuf, int *i)
{
     *i += 1;
     /* fprintf(stderr, "\tXXX read_extended_speed not implemented!\n"); */
}

const char *tuple_name(int code)
{
#define MAX_TUPLE_NAME (sizeof(tuple_names) / sizeof(char *))
     static const char *tuple_names[] = {
	  "NULL", "DEVICE",
	  "reserved", "reserved", "reserved", "reserved",
	  "reserved", "reserved", "reserved", "reserved",
	  "reserved", "reserved", "reserved", "reserved",
	  "reserved", "reserved",
	  "CHECKSUM", "LONGLINK_A", "LONGLINK_C", "LINKTARGET",
	  "NO_LINK", "VERS_1", "ALTSTR", "DEVICE_A",
	  "JEDEC_C", "JEDEC_A", "CONFIG", "CFTABLE_ENTRY",
	  "DEVICE_OC", "DEVICE_OA",
     };

     return (code < MAX_TUPLE_NAME) ? tuple_names[code] : "UNKNOWN";
}

const char *dtype_name(int idx)
{
#define MAX_DTYPE_NAME (sizeof(dtype_names) / sizeof(char *))
     static const char *dtype_names[] = {
	  "NULL", "ROM", "OTPROM", "EPROM",
	  "EEPROM", "FLASH", "SRAM", "DRAM",
     };

     return (idx < MAX_DTYPE_NAME) ? dtype_names[idx] : "INVALID";
}

const char *dspeed_name(int idx)
{
#define MAX_DSPEED_NAME (sizeof(dspeed_names) / sizeof(char *))
     static const char *dspeed_names[] = {
	  "NULL", "250NS", "200NS", "150NS",
	  "100NS", "reserved", "reserved", "extended",
     };

     return (idx < MAX_DSPEED_NAME) ? dspeed_names[idx] : "INVALID";
}

const char *dsize_name(int idx)
{
#define MAX_DSIZE_NAME (sizeof(dsize_names) / sizeof(char *))
     static const char *dsize_names[] = {
	  "512b", "2k", "8k", "32k", "128k", "512k", "2M", "reserved",
     };
     return (idx < MAX_DSIZE_NAME) ? dsize_names[idx] : "INVALID";
}
