/* Support code for the Databook TCIC/2 PCMCIA interface chip.  For
   more information, see the Databook Hardware Design Guide for
   this chip.

       Databook Incorporated
       Babcock Hall
       112 Prospect Street
       Ithaca, N.Y. 14850
       (607) 277-4817 (phone)
       (607) 273-8803 (FAX)

   This support code was written by Robert Bedichek at the University
   of Washington, robertb@cs.washington.edu.
*/

static char *version = "tcic2.c:v0.99-14 based (robertb@cs.washington.edu)\n";

/* Slight dependence on the Ethernet driver for the DL650. */

extern int ei_debug;

#include <linux/config.h>
#include <linux/kernel.h>
#include <linux/sched.h>
#include <linux/errno.h>
#include "dev.h"
#include <asm/system.h>
#include <asm/io.h>

/* Default base address of the Databook TCIC/2 chip. */

#define	TCIC_BASE	(0x240)

/* Offsets of TCIC/2 registers for the TCIC/2 base address. */

#define OFF_DATA		(0)
#define	OFF_ADDR		(2)
#define OFF_SCTRL		(6)
#define OFF_SSTAT		(7)
#define OFF_MODE		(8)
#define OFF_PWR			(9)
#define OFF_EDC			(10)
#define OFF_ICSR		(12)
#define OFF_IENA		(13)
#define OFF_AUX			(14)

/* Codes that are put in the top three bits of the R_MODE register
   to select which auxiliary register is visiable at OFF_AUX. */

#define AUX_TCTL 		(0<<5)
#define AUX_PCTL		(1<<5)
#define AUX_WCTL		(2<<5)
#define AUX_EXTERN 		(3<<5)
#define AUX_PDATA 		(4<<5)
#define AUX_SYSCFG		(5<<5)
#define AUX_ILOCK		(6<<5)
#define AUX_TEST		(7<<5)

/* Top 16 bits of the address register. */

#define	ADDR_H_INDIRECT		(0x0800)

/* Bit definitions for selected fields (like just those that we use). */

#define	SSTAT_CARD		(0x80)	/* Card installed. */

/* Socket control register */

#define	SCTRL_EN		(0x01)	/* Enable card access to selected soc */

/* Power control register */

#define	PWR_CURRENTL		(0x40)	/* Enable current limiting */
#define	PWR_VCCSEL0		(0x01)	/* 5 Volt supply control for sock 0 */
#define	PWR_VCCSEL1		(0x02)	/* 5 Volt supply control for sock 1 */

/* I/O map control register */

#define	IOCTL_EN		(0x8000) /* Enable this map */
#define	IOCTL_8BIT		(0x0800)
#define	IOCTL_16BIT		(0x0400)
#define	IOCTL_QUIET		(0x0040) /* Make the buffers quieter */
#define	IOCTL_ONEK		(0x0080) /* This map is 1k or less */

/* Interrupt control/status register */

#define	ICSR_WRITEALL		(3)	/* Write all bits 7:2 in CSR */

/* Interrupt enable register */

#define	IENA_SSTAT		(0x40)	/* Interrupt on any changed to SSTAT*/
#define	IENA_OPENDRAIN		(0x01)	/* Make STKIRQ output open drain */

/* Mode register */

#define	MODE_WORD		(0x10)	/* Mode register, word access */

/* Memory map control register */

#define	MCTL_EN			(0x8000) /* Mem map ctl reg, enable */
#define	MCTL_QUIET		(0x0040) /* Make accesses use quiet mode */

/* Memory map map register. */

#define	MMAP_ATTRIBUTE		(0x8000) /* Map this to card attribute space */

/* System configuration register 1 */

#define	SCF1_STATIO		(0x0800)

/* In de650.c, delays 'len' clock ticks. */

void delay(int len);

/* 16-bit read. */
static int in_p(int addr) 
{ 
  return (inb_p(addr+1)<<8) | inb_p(addr);
}

/* 16-bit write. */
static void out_p(int val, int addr) 
{ 
  outb_p(val&0xff,addr); 
  outb_p((val >> 8)&0xff, addr+1);
}

/* Map TCIC/2 I/O map number 'map' (range 0..3) to host address 'iobase' 
   for 32 bytes.  'tcic_base' is the base address of the TCIC/2 itself. */

static void set_tcic_io_map(int tcic_base, int map, int iobase, int control)
{
	/* Put the address register into indirect mode. */
	out_p(ADDR_H_INDIRECT, tcic_base + OFF_ADDR + 2);

	/* Low 16 bits of address register, we select the I/O base address of
	  the appropriate map. */
	out_p(0x200 + (map << 2), tcic_base + OFF_ADDR);

	/* Load I/O base register. */
	out_p(iobase, tcic_base + OFF_DATA);

	/* Low 16 bits of address register, we select the I/O control register of
	  the appropriate map. */
	out_p(0x202 + (map << 2), tcic_base + OFF_ADDR);

	/* Load I/O control register. */
	out_p(control, tcic_base + OFF_DATA);
}

/* Map TCIC/2 memory map number 'map' (range 0..7) to host address
   'membase' for 4k bytes.  'tcic_base' is the base address of the 
   TCIC/2 itself. */

static void set_tcic_mem_map(int tcic_base, int map, int membase, int control)
{
	membase >>= 12;

	/* Put the address register into indirect mode. */
	out_p(ADDR_H_INDIRECT, tcic_base + OFF_ADDR + 2);

	/* Low 16 bits of address register, we select the memory base address 
	register of the specified map. */
	out_p(0x102 + (map << 3), tcic_base + OFF_ADDR);

	/* Load memory base register.  Make the window 16k bytes. */
	out_p(membase | 2, tcic_base + OFF_DATA);

	/* Low 16 bits of address register, we select the memory window
	  map register of the specified map.   We want to map the host
	  address to location zero of the card's attribute memory.  To
	  do this we load the memory map register with a value that when
	  added to the host address will overflow out of bit 25 and will 
	  be zero.  So we put the two's complement of the host address
	  into the window map register.  */

	out_p(0x104 + (map << 3), tcic_base + OFF_ADDR);
	out_p(MMAP_ATTRIBUTE|-membase, tcic_base + OFF_DATA);

	/* Low 16 bits of address register, we select the memory 
	   control register of the specified map. */
	out_p(0x106 + (map << 3), tcic_base + OFF_ADDR);

	/* Load memory control register. */
	out_p(control, tcic_base + OFF_DATA);
}

/* Set up a pair of TCIC/2 socket configuration registers for socket
   'socket'. */

static void set_tcic_socket_config(int tcic_base, int socket, int cf1, int cf2)
{
	/* Put the address register into indirect mode. */
	out_p(ADDR_H_INDIRECT, tcic_base + OFF_ADDR + 2);

	/* Load low 16 bits of address register, we select the 
	   socket configuration register 1 of the specified socket. */
	out_p(socket << 3, tcic_base + OFF_ADDR);

	/* Load socket configuration register 1. */
	out_p(cf1, tcic_base + OFF_DATA);

	/* Load low 16 bits of address register, we select the 
	   socket configuration register 2 of the specified socket. */
	out_p((socket << 3) | 2, tcic_base + OFF_ADDR);

	/* Load socket confiuration register 2. */
	out_p(cf2, tcic_base + OFF_DATA);
}

/* See if this laptop is using Databook TCIC/2 PCMCIA controller.  If
   it is, configure it for use with the DL650 Ethernet controller and
   return 0.  Otherwise, return ENODEV. */

int probe_for_tcic2(struct device *dev)
{
	/* This will probe for a TCIC/2 at the standard location. */
	int ioaddr, socket, base = TCIC_BASE;
	int addr = base + OFF_ADDR;

	if (ei_debug)
	  printk(version);

	/* We probe for the TCIC/2 by writing the address register with
	   zeros, making sure it is zero, writing it with a non-zero
	   pattern and then checking to see that that pattern was written. */
	out_p(0, addr);
	out_p(0, addr+2);
	if (in_p(addr) | in_p(addr+2)) return ENODEV;

	out_p(0xbed1, addr);
	if (in_p(addr) != 0xbed1) return ENODEV;

	if (inb_p(base+OFF_SSTAT) & SSTAT_CARD) {
		printk("Found card in TCIC/2 (0x%x) in socket 0.\n", base);
		socket = 0;
	} else {
		outb_p(addr+3,0x10); /* Select socket 1 */
		if (inb_p(base+OFF_SSTAT) & SSTAT_CARD) {
			printk("Found card in TCIC/2 (0x%x) in socket 1.\n", base);
			socket = 1;
		} else {
			printk("Found TCIC/2 (0x%x), but no card.\n", base);
			return ENODEV;
		}
	}
        /* Shut down, then turn on the card */
        outb_p(PWR_CURRENTL, base+OFF_PWR);
        delay(1);
        outb_p(PWR_CURRENTL | (socket ? PWR_VCCSEL1 : PWR_VCCSEL0), 
							base+OFF_PWR);
        delay(2);
        outb_p(SCTRL_EN,base+OFF_SCTRL);    
    #if defined(DL_SHMEM)
        dev->mem_start = DL_SHMEM;
    #else
        if (dev->mem_start == 0) dev->mem_start = 0xD6000L;
    #endif
        ioaddr = dev->base_addr;
    
        /* Map 32 bytes of I/O space starting at 'ioaddr' to the
           socket specified by 'socket'.  Use 8-bit mode.
           Use the quiet mode to reduce power consumption, wait state 
           value of 7. */

        set_tcic_io_map(base, 0, ioaddr | 0x10, 
                 IOCTL_EN|(socket << 12)|IOCTL_8BIT|IOCTL_ONEK|IOCTL_QUIET|7);

        /* Set up memory window 0 for the PROM */
        set_tcic_mem_map(base, 0, dev->mem_start, 
                           MCTL_EN | (socket << 12)|MCTL_QUIET);

        set_tcic_socket_config(base, socket, SCF1_STATIO | dev->irq, 0x7a);

        /* Load the interrupt enable register with 0 to make sure
           that we don't get an IRQ from the TCIC/2 because of
           something other than the card requesting an interrupt. */

        outb_p(0, base+OFF_IENA);

        /* Load wait-state control auxiliary register.*/

	outb_p(AUX_WCTL|MODE_WORD, base+OFF_MODE); /* Select WCTL register */
        out_p(0x6067, base+OFF_AUX);

        /* Load the system configuration auxiliary register*/
	outb_p(AUX_SYSCFG|MODE_WORD, base+OFF_MODE); /* Select SYSCFG register */
        out_p(0x66d0|dev->irq, base+OFF_AUX);

        /* Load the interlock control/status auxiliary register */
	outb_p(AUX_ILOCK|MODE_WORD, base+OFF_MODE); /* Select WCTL register */
        out_p(0x2e8, base+OFF_AUX);

	return 0;
}
