/*
 * XDU - X Window System Interface.
 *
 * We hide all of the X hieroglyphics inside of this module.
 *
 * Phillip C. Dykstra
 * <phil@arl.army.mil>
 * 4 Sep 1991.
 * 
 * Copyright (C)	Phillip C. Dykstra	1991, 1993
 * Permission to use, copy, modify, distribute, and sell this software
 * and its documentation for any purpose is hereby granted without fee,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation, and that the authors name not be used in
 * advertising or publicity pertaining to distribution of the software
 * without specific, written prior permission.  The author makes no
 * representations about the suitability of this software for any purpose.
 * It is provided "as is" without express or implied warranty.
 */
#include <stdio.h>
#include <X11/Intrinsic.h>
#include <X11/StringDefs.h>
#include <X11/Core.h>

/* IMPORTS: routines that this module vectors out to */
extern int press();
extern int reset();
extern int repaint();
extern int reorder();
extern int setorder();
extern int nodeinfo();
extern int helpinfo();
extern int ncols;

/* EXPORTS: routines that this module exports outside */
extern int xsetup();
extern int xmainloop();
extern int xclear();
extern int xrepaint();
extern int xrepaint_noclear();
extern int xdrawrect();

static String fallback_resources[] = {
"*window.width:		600",
"*window.height:	480",
"*order:		first",
NULL
};

/* Application Resources */
typedef struct {
	Pixel	foreground;
	Pixel	background;
	XFontStruct *font;
	int	ncol;
	Boolean	showsize;
	char	*order;
} res_data, *res_data_ptr;
static res_data res;

static XtResource application_resources[] = {
	{ XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
		XtOffset(res_data_ptr,foreground), XtRString, XtDefaultForeground},
	{ XtNbackground, XtCBackground, XtRPixel, sizeof(Pixel),
		XtOffset(res_data_ptr,background), XtRString, XtDefaultBackground},
	{ XtNfont, XtCFont, XtRFontStruct, sizeof(XFontStruct *),
		XtOffset(res_data_ptr,font), XtRString, XtDefaultFont },
	{ "ncol", "Ncol", XtRInt, sizeof(int),
		XtOffset(res_data_ptr,ncol), XtRString, "5"},
	{ "showsize", "ShowSize", XtRBoolean, sizeof(Boolean),
		XtOffset(res_data_ptr,showsize), XtRString, "True"},
	{ "order", "Order", XtRString, sizeof(String),
		XtOffset(res_data_ptr,order), XtRString, "first"}
};

/* Command Line Options */
static XrmOptionDescRec options[] = {
	{"-c",		"*ncol",	XrmoptionSepArg,	NULL},
	{"+s",		"*showsize",	XrmoptionNoArg,		"True"},
	{"-s",		"*showsize",	XrmoptionNoArg,		"False"},
	{"-n",		"*order",	XrmoptionNoArg,		"size"},
	{"-rn",		"*order",	XrmoptionNoArg,		"rsize"},
	{"-a",		"*order",	XrmoptionNoArg,		"alpha"},
	{"-ra",		"*order",	XrmoptionNoArg,		"ralpha"}
};

/* action routines */
static void a_goto();
static void a_reset();
static void a_quit();
static void a_reorder();
static void a_size();
static void a_ncol();
static void a_info();
static void a_help();

static XtActionsRec actionsTable[] = {
	{ "reset",	a_reset },
	{ "goto",	a_goto },
	{ "quit",	a_quit },
	{ "reorder",	a_reorder },
	{ "size",	a_size },
	{ "ncol",	a_ncol },
	{ "info",	a_info },
	{ "help",	a_help }
};

static char defaultTranslations[] = "\
<Key>Q:	quit()\n\
<Key>Escape: quit()\n\
<Key>/:	reset()\n\
<Key>S:	size()\n\
<Key>I:	info()\n\
<Key>H: help()\n\
<Key>A:	reorder(alpha)\n\
<Key>N:	reorder(size)\n\
<Key>F:	reorder(first)\n\
<Key>L:	reorder(last)\n\
<Key>R:	reorder(reverse)\n\
<Key>1:	ncol(1)\n\
<Key>2:	ncol(2)\n\
<Key>3:	ncol(3)\n\
<Key>4:	ncol(4)\n\
<Key>5:	ncol(5)\n\
<Key>6:	ncol(6)\n\
<Key>7:	ncol(7)\n\
<Key>8:	ncol(8)\n\
<Key>9:	ncol(9)\n\
<Key>0:	ncol(10)\n\
<Btn1Down>: goto()\n\
<Btn2Down>: reset()\n\
<Btn3Down>: quit()\n\
";

/*  action routines  */

static void a_quit(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	XtDestroyApplicationContext(XtWidgetToApplicationContext(w));
	exit(0);
}

static void a_goto(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	press(event->xbutton.x, event->xbutton.y);
}

static void a_reset(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	reset();
}

static void a_reorder(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	if (*num_params != 1) {
		fprintf(stderr, "xdu: bad number of params to reorder action\n");
	} else {
		reorder(*params);
	}
}

static void a_size(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	if (res.showsize)
		res.showsize = 0;
	else
		res.showsize = 1;
	xrepaint();
}

static void a_ncol(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	int	n;

	if (*num_params != 1) {
		fprintf(stderr, "xdu: bad number of params to ncol action\n");
		return;
	}
	n = atoi(*params);
	if (n < 1 || n > 1000) {
		fprintf(stderr, "xdu: bad value to ncol action\n");
		return;
	}
	ncols = res.ncol = n;
	xrepaint();
}

static void a_info(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	nodeinfo();
}

static void a_help(w, event, params, num_params)
Widget w;
XEvent *event;
String *params;
Cardinal *num_params;
{
	helpinfo();
}

/* callback routines */

static void c_resize(w, data, event)
Widget w;
caddr_t data;
XEvent *event;
{
	/*printf("Resize\n");*/
	xrepaint();
}

static void c_repaint(w, data, event)
Widget w;
caddr_t data;
XEvent *event;
{
	/*printf("Expose\n");*/
	xrepaint_noclear();
}

/* X Window related variables */
static Cursor WorkingCursor;
static Display *dpy;
static int screen;
static Visual *vis;
static Window win;
static GC gc;
static GC cleargc;
static XtAppContext app_con;

/*  External Functions  */

int
xsetup(argcp, argv)
int *argcp;
char **argv;
{
	XtTranslations trans_table;
	Widget toplevel;
	Widget w;
	XGCValues gcv;

	/* Create the top level Widget */
	toplevel = XtAppInitialize(&app_con, "XDu",
			options, XtNumber(options),
			argcp, argv,
			fallback_resources, NULL, 0);

	XtGetApplicationResources(toplevel, (XtPointer)&res,
		application_resources, XtNumber(application_resources),
		NULL, 0 );

	XtAppAddActions(app_con, actionsTable, XtNumber(actionsTable));
	trans_table = XtParseTranslationTable(defaultTranslations);

	/* Create a simple Core class widget to draw in */
	w = XtCreateManagedWidget("window", coreWidgetClass, toplevel,
		NULL, 0);

	/* events */
	XtAddEventHandler(w, ExposureMask, FALSE, c_repaint, NULL);
	XtAddEventHandler(w, StructureNotifyMask, FALSE, c_resize, NULL);
	XtAugmentTranslations(w, trans_table);

	XtRealizeWidget(toplevel);

	/* We need these for the raw Xlib calls */
	win = XtWindow(w);
	dpy = XtDisplay(w);
	screen = DefaultScreen(dpy);
	vis = DefaultVisual(dpy,screen);

	gcv.foreground = res.foreground;
	gcv.background = res.background;
	gcv.font = res.font->fid;
	gc = XCreateGC(dpy, win, (GCFont|GCForeground|GCBackground), &gcv);

	setorder(res.order);
	ncols = res.ncol;
}

xmainloop()
{
	XtAppMainLoop(app_con);
	return(0);
}

xclear()
{
	XClearWindow(dpy, win);
}

xrepaint()
{
	XWindowAttributes xwa;

	XClearWindow(dpy, win);
	XGetWindowAttributes(dpy, win, &xwa);
	repaint(xwa.width, xwa.height);
}

xrepaint_noclear()
{
	XWindowAttributes xwa;

	XGetWindowAttributes(dpy, win, &xwa);
	repaint(xwa.width, xwa.height);
}

xdrawrect(name, size, x, y, width, height)
char *name;
int size;
int x, y, width, height;
{
	int	textx, texty;
	char	label[1024];
	XCharStruct overall;
	int	ascent, descent, direction;
	int	cheight;

	/*printf("draw(%d,%d,%d,%d)\n", x, y, width, height );*/
	XDrawRectangle(dpy, win, gc, x, y, width, height);

	if (res.showsize) {
		sprintf(label,"%s (%d)", name, size);
		name = label;
	}

	XTextExtents(res.font, name, strlen(name), &direction,
		&ascent, &descent, &overall);
	cheight = overall.ascent + overall.descent;
	if (height < (cheight + 2))
		return;

	/* print label */
	textx = x + 4;
	texty = y + height/2.0 + (overall.ascent - overall.descent)/2.0 + 1.5;
	XDrawString(dpy, win, gc, textx, texty, name, strlen(name));
}
