/*
    yamm, Yet Another Micro Monitor
    curses.c
    Copyright (C) 1994  Riccardo Facchetti && Andrea Marangoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include "define.h"
#include "extern.h"

#ifndef NO_CURSES
#if !defined(NCURSES)
#include <curses.h>
#else
#include <termios.h>
#include "ncurses.h"
#endif /* !NCURSES */
#include <signal.h>
#include <string.h>
#include <time.h>
#include <termios.h>
#include <sys/ioctl.h>
#if defined(linux)
#include <linux/time.h>
#include <sys/time.h>
#include <sys/types.h>
#include <unistd.h>
#endif /* linux */

#if defined(NCURSES)
# if !defined(FGC)
# define FGC COLOR_WHITE
# endif /* !FGC */
# if !defined(BGC)
# define BGC COLOR_BLACK
# endif /* !BGC */

int fgc, bgc;

#endif /* NCURSES */

#if defined(linux) && !defined(NCURSES)
#define GETSTR(x) \
	nocbreak(); \
	echo(); \
	gets ( x ); \
	noecho(); \
	cbreak(); \
	refresh_all()
#endif /* linux */

#if defined(hpux) || defined(NCURSES)
#define GETSTR(x) \
	nocbreak(); \
	echo(); \
	getstr ( x ); \
	noecho(); \
	cbreak(); \
	refresh_all()
#endif /* hpux */

#if !defined(GETSTR)
# error You must define one macro called GETSTR
#endif /* !GETSTR */

#define CTRL(x) ((x) & 037)
#define PF (void)printw

typedef struct termios MYSGTTY;
static MYSGTTY _mytty;

#if defined(SIGWINCH) && defined(OWN_WINCH)
void do_chsize(void)
{
/*
 * End curses package
 */
	endwin();

/*
 * Tell the parent task (the shell)  that the window is changed.
 */
	kill(getppid(), SIGWINCH);

/*
 * This thing works well: restart yamm from scratch because restart only (n)curses
 * seems not work very well ... (coredumps and the like).
 * Anyway if you want the SIGWINCH trapped and not ignored, this is the only solution
 * for now: i have not time nor the will to do the things better.
 * If anyone of you would like to write down the code that handle the do_chsize
 * function in a better way, (s)he's welcome!! :)
 */
#if 0
	init_curses();
#else
	execv(yammav[0], yammav);
#endif /* 0 */
}
#endif /* SIGWINCH && OWN_WINCH */

void quit ( sig )
int sig;
{
	move ( LINES -1 , 0 );
	clrtoeol();
	curses_refresh();
	endwin();

#if defined(MOD)

/*
 * close the yammfd before exit
 */
	close(yammfd);

#endif /* MOD */

/*
 * reset the screen attribs
 */
	tcsetattr(fileno(stdin), TCSADRAIN, &_mytty);

	if ( sig > 0 ) {
		fprintf ( stderr, "Trapped signal %d\n", sig );
		exit ( sig );
	}
	exit ( sig );
}

void init_curses()
{
	int i;

/*
 * save screen attribs
 */
	tcgetattr(fileno(stdin), &_mytty);
	initscr();
	erase ();
	clear();
	curses_refresh();
	cbreak();
	noecho();

	if ( COLS < 80 ) {
		echo();
		fprintf ( stderr, "You must have at least 80 cols.\n" );
		quit ( 0 );
	}

	if ( LINES < 8 ) {
		echo();
		fprintf ( stderr, "You must have at least 8 lines.\n" );
		quit ( 0 );
	}

#if defined(NCURSES) || !defined(linux)
	keypad(stdscr,TRUE);
#endif /* NCURSES || !linux */

	for ( i = 1; i < NSIG; ++i ) {

#if defined(SIGWINCH)
/*
 * trap SIGWINCH for use in X11 windows when resizing a window in which yamm run
 */
		if ( i == SIGWINCH ) {
#if defined(OWN_WINCH)
			signal(SIGWINCH, winch_func);
#endif /* OWN_WINCH */
			continue;
		}
#endif /* SIGWINCH */

#if defined(PROFILING)
/*
 * If profiling the code, do not trap the SIGPROF
 */
		if ( i == SIGPROF )
			continue;
#endif /* PROFILING */

		if ( i == SIGCLD )
			continue;
		( void ) signal ( i , quit );
	}

	if ( Num_display_proc > LINES - 6 || Num_display_proc < 0 )
		Num_display_proc = LINES - 6 ;

#if defined(NCURSES)
	if (has_colors()) {
		start_color();
		fgc = FGC;
		bgc = BGC;
		init_pair(1, fgc, bgc);
		attron(COLOR_PAIR(MAIN_PAIR));
	}
#endif /* NCURSES */

	curses_refresh();
}

void clear_screen()
{
	erase();
	move ( 0, 0 );
}

int look_lines()
{
	return ( LINES );
}

int look_cols ()
{
	return ( COLS );
}

void update_first_line( nproc, dinfo )
int nproc;
struct pst_dynamic *dinfo;
{
	extern struct tm *localtime();
	struct tm actual;
	time_t mclock = time ( ( time_t *)0 );

	long user_processes = 0L, t_size = 0L, t_rss = 0L;

	clear_screen();

	actual = *localtime ( &mclock );
	count_status ( nproc );

	PF ( "[%2d:%.2d%:%.2d] Load averages: %.2f  %.2f  %.2f.  Pages: %d - %s Idle\n" ,
		actual.tm_hour, actual.tm_min, actual.tm_sec,
		dinfo->psd_avg_1_min, dinfo->psd_avg_5_min,
		dinfo->psd_avg_15_min, pages,
		info_cpu ( dinfo->psd_cpu_time ));

	PF ( "%d Processes: %d Sleeping, %d Running, %d Stopped, %d Zombie, %d Others.\n",
		nproc, status_sleep(), status_run(), status_stop(),
		status_zombie(), status_other() );

	PF ( "Memory: %s  ", get_dimension ( dinfo->psd_rm) );
	PF ( "( %s ) real,  ", get_dimension ( dinfo->psd_arm ) );
	PF ( "%s  ", get_dimension ( dinfo->psd_vm ) );
	PF ( "( %s ) virtual, ", get_dimension ( dinfo->psd_avm ) );
	PF ( "%s free\n", get_dimension ( dinfo->psd_free ) );

	if ( look_uid >= 0 ) {
		get_user_sum( nproc, &user_processes, &t_size, &t_rss);
		curses_in_reverse();
		PF ( "UID %d: %ld processes,", look_uid, user_processes );
		PF ( "  %s  TOTAL SIZE,", get_dimension ( t_size ) );
		PF ( "  %s  TOTAL RSS.", get_dimension ( t_rss ) );
		curses_out_reverse();
	}
		
}

void curses_help()
{
	static int first = 1;
	int page = 0, car = 0;

	if ( LINES < 24 ) {
		curses_error ( "Sorry, you must have at least 24 lines." );
		return;
	}

	if ( !first )
		car = ( int )'f';  /* The first time start page 0 else 1 */
	else
		car = ( int )'B';
	
	first = 0;

	do {
		switch ( car ) {
		case 'f':
		case ' ':
#if !defined(NCURSES)
			if ( page < 2 )
#else
			if ( page < 3 )
#endif /* !NCURSES */
				++page;
		break;
		case 'b':
			if ( page > 0 )
				--page;
		break;
		case 'B':
			page = 0;
		break;	
		default:
		break;
		}

		switch ( page ) {
		case 0:
		clear_screen();
		PF ( "YAMM ( Yet Another Micro Monitor ) " YAMM_VERSION 
#if defined(MOD)
			" ( Modular )"
#endif /* MOD */
			" Jul 1994.\n" );
		PF ( "By Riccardo Facchetti.\n" );
		PF ( "E-mail: riccardo@cdc8g5.cdc.polimi.it.\n");
		PF ( "Derived from V2.4 Jun 1992 by Marangoni Andrea.\n\n" );
		PF ( "Explanation of fields:\n\n" );
#if !defined(linux)
		PF ( "S           - Status of process.\n");
		PF ( "    'S' Sleep, 'R' Running or waiting for Cpu, 'T' Stopped,\n");
		PF ( "    'Z' Zombie, 'I' Being created, 'O' Other ( forking, exiting, etc. ).\n");
		PF ( "\nUSER        - Name of user.\n" );
		PF ( "SUID        - Effective user id ( an '*' means that euid != uid ).\n");
#else
		PF ( "S           - Status of process.\n");
		PF ( "    'S' Sleep, 'R' Running or waiting for Cpu, 'T' Stopped,\n");
		PF ( "    'Z' Zombie, 'D' Swapping or uninterruptible.\n");
		PF ( "\nUSER        - Name of user.\n" );
		PF ( "UID         - User id.\n");
#endif /* linux */
		PF ( "PID         - Process id.\n" );
		PF ( "NI          - Nice.\n" );
		PF ( "PRI         - Priority.\n" );
		PF ( "SIZE        - Size of process ( data + text + stack ).\n");
		PF ( "RSS         - Resident set size ( private pages only ).\n");
		PF ( "STIME       - Start time.\n");
		PF ( "\"u|s| \"TIME - ( 'TIME' Total time, 'uTIME' User time, 'sTIME' System time ).\n" );
		PF ( "CMD         - Command.\n" );
		move ( LINES - 1 , 0 );
		PF ( "Type 'f' to next page or <RETURN> to exit." );
		break;

		case 1:
		clear_screen();
		PF ( "YAMM ( Yet Another Micro Monitor ) " YAMM_VERSION " Jul 1994.\n\n" );
		PF ( "\"f\" or PAGEDOWN Forward  1 page.\n" );
		PF ( "\"b\" or PAGEUP Backward 1 page.\n" );
		PF ( "\"B\" or HOME Go to page 0.\n\n" );
		PF ( "\"s\" Sort by 's'tart time.\n" );
		PF ( "\"t\" Sort by process 't'ime.\n" );
		PF ( "\"c\" Sort by process %%cpu time, default.\n" ); /* HV */
		PF ( "\"d\" Sort by process size ( 'd'ata + text + stack ).\n" );
		PF ( "\"D\" Sort by process resident set size ( private pages ).\n" );
		PF ( "\"i\" Invert sorting order.\n");
		PF ( "\"<SPACE>\" Disable sorting.\n\n" );
		PF ( "\"m\" Display only 'm'y processes. Press \"m\" again to return to previous state.\n");
		PF ( "\"r\" Allow display of 'r'oot processes ( default = off ).\n" );
		PF ( "\"E\" Show processes with UID != EUID. ( \"E\" again to return to previous state ).\n" );
		PF ( "\"w\" Like \"who\". ( \"w\" again or <RETURN> to return to previous state ).\n" );
		PF ( "\"e\" Display the entries in utmp.\n\n" );
		PF ( "\"P\" or DELETE Kill process ( After choose it with slide bar ).\n" );
		PF ( "\"K\" Kill user ( After choose it with slide bar ).\n " );
		move ( LINES - 1 , 0 );
		PF ( "Type 'f' to next page, 'b' to previous page or <RETURN> to exit." );
		break;

		case 2:
		clear_screen();
		PF ( "YAMM ( Yet Another Micro Monitor ) " YAMM_VERSION " Jul 1994.\n\n" );
		PF ( "\"u\" Restrict to a specific 'u'id ( You may choose it with slidebar ).\n" );
		PF ( "    Type \"m\"  to return to previous state.\n" );
		PF ( "    Type \"I\" for full user info.\n\n" );
		PF ( "\"S\" Display only System time in field \"TIME\".\n");
		PF ( "\"U\" Display only User time in field \"TIME\".\n\n");
		PF ( "\"%%\" Switch between  STIME-TIME and WCPU-WCHAN fields.\n" );
		PF ( "    If you have >= 90 cols both fields are displayed.\n" );
		PF ( "\"N\" Display only \"number\" processes.\n" );
		PF ( "\"p\" Analyze a single process ( Type <SPACE> to return to previous state ).\n" );
		PF ( "\"C\" Display configuration of machine.\n" );
		PF ( "\"V\" Virtual memory info.\n" );
		PF ( "\"W\" Loop forever waiting \"second\" for each loop.\n" );
		PF ( "\"/\" Search \"string\" in command line. NULL string or <SPACE> to annull.\n" );
		PF ( "\"R\" Renice process ( You must be root :-) ).\n" );
		PF ( "\"G\" Renice process group ( You must be root :-) ).\n" );
		PF ( "\"A\" Renice user ( You must be root :-) ).\n" );
		PF ( "\"j\" or DOWN ARROW Move slide bar down.\n" );
		PF ( "\"k\" or UP ARROW Move slide bar up.\n\n" );
		PF ( "\"Q\" or \"CTRL-D\" or ESCAPE to exit.\n" );
		move ( LINES - 1 , 0 );
#if !defined(NCURSES)
		PF ( "Type 'b' to previous page or <RETURN> to exit." );
#else
		PF ( "Type 'f' to next page, 'b' to previous page or <RETURN> to exit." );
#endif /* !NCURSES */
		break;

#if defined(NCURSES)
		case 3:
		clear_screen();
		PF ( "YAMM ( Yet Another Micro Monitor ) " YAMM_VERSION " Jul 1994.\n\n" );
		PF ( "\"-\" or LEFT ARROW to backward cycle the foreground color.\n" );
		PF ( "\"=\" or RIGHT ARROW to forward cycle the foreground color.\n" );
		PF ( "\"_\" to backward cycle the background color.\n" );
		PF ( "\"+\" to forward cycle the foreground color.\n" );
		move ( LINES - 1 , 0 );
		PF ( "Type 'b' to previous page or <RETURN> to exit." );
		break;
#endif /* NCURSES */

		default:
		break;
	}
	curses_refresh();
	} while ( ( car = getch() ) != 'q' && car != '\n' ); 

#if defined(linux)
	if (car == '\n')
		refresh_all();
#endif /* linux */
}

void curses_refresh()
{
	refresh();
}

void refresh_all ()
{
	clear();
	touchwin( stdscr );
	curses_refresh();
}

int curses_wait( second )
int second;
{
	struct timeval timeout;
	fd_set input_set;

	timeout.tv_sec = ( second <= 0 ) ?  1L : ( long )second;
	timeout.tv_usec = 0L;

	FD_ZERO ( &input_set );
	FD_SET ( fileno ( stdin ), &input_set );

#if !defined(linux)
	if ( select ( ( size_t )(fileno ( stdin ) + 1), ( int *)&input_set,
		( int *)0, ( int *)0, &timeout ) == -1 ) {
#else
	if ( select ( ( size_t )(fileno ( stdin ) + 1), &input_set,
		( fd_set * )0, ( fd_set * )0, &timeout ) == -1 ) {
#endif /* !linux */
			curses_error( "Select" );
		return ( 0 );
	}


	if ( FD_ISSET ( 0, &input_set ) ) {
		int i = getch();

		if ( i == 'q' || i == 'Q' || i == CTRL('D') || i == 27 )
			quit ( 0 );

		if ( i == CTRL('L') || i == CTRL('R') ) {
			refresh_all();
			return ( 0 );
		}


		if ( i == '?' || i == 'h' )
			curses_help();

		mpreprocess ( i, NULL, NULL );
		return ( i );
	}
	return ( 0 );
}

#define BUF 128

long ask_curses ( ask )
char *ask;
{
#if !defined(linux)
	extern long atol();
#endif /* !linux */
	char buf [ BUF ];
	register int i;

	move ( LINES - 1 , 0 );
	clrtoeol();
	PF ( ask );
	curses_refresh();
	GETSTR ( buf );

	/* skip space and tab */
	for ( i = 0; buf [ i ] == ' ' || buf [ i ] == '\t'; ++i );

	if ( buf[ i ] == '\n' || buf[ i ] == '\0' )
		return ( -1L );

	return ( atol ( buf + i ) );
}

char *curses_get_string ( ask )
char *ask;
{
	static char buf [ BUF ];
	register int i;

	move ( LINES - 1 , 0 );
	clrtoeol();
	PF ( ask );
	curses_refresh();
	GETSTR ( buf );

	/* skip space and tab */
	for ( i = 0; buf [ i ] == ' ' || buf [ i ] == '\t'; ++i );

	if ( buf[ i ] == '\n' || buf [ i ] == '\0' )
		return ( NULL );

	return ( buf + i  );
}


void curses_error ( message )
char *message;
{
	move ( 3, 0 );
	clrtoeol();
	curses_in_reverse();
	PF ( message );
	curses_out_reverse();
	curses_refresh();
	( void )sleep ( 1 ); /* refresh too fast */
}

/* if reverse don't work comment "standout()" and "standend()" */
void curses_in_reverse()
{
#if !defined(NCURSES)
	standout();
#else
	attron(A_STANDOUT);
#endif /* !NCURSES */
}

void curses_out_reverse()
{
#if !defined(NCURSES)
	standend();
#else
	attroff(A_STANDOUT);
	attron(COLOR_PAIR(MAIN_PAIR));
#endif /* !NCURSES */
}


#define HOSTNAMELEN 65 /* 64 + 1 */
void curses_print_help()
{
	static int yet = 0;
	static char host_name [ HOSTNAMELEN ];

	if ( !yet ) {
		host_name [ 0 ] = '\0';
		if ( gethostname ( host_name, ( size_t ) HOSTNAMELEN ) == -1 )
			host_name [ 0 ] = '\0';
		yet = 1;
		host_name [ HOSTNAMELEN - 1 ] = '\0';
	}

	move ( LINES - 1 , 0 );
	clrtoeol();

	if ( host_name [ 0 ] )
		PF ( "[ %s ] Type <?> for help." , host_name );
	else
		PF ( "Type < ? > for help." );
}

#else /* NO_CURSES */

long ask_curses( msg )
char *msg;
{
	return ( -1L ); 
}

char *curses_get_string( msg )
char *msg;
{
	return ( NULL );
}

int look_lines()
{
	return ( 0 ); /* Not necessary only for lint */
}

int look_cols()
{
	if ( long_format )
		return ( 90 ); 
	else
		return ( 80 );
}

#endif

