/*
    yamm, Yet Another Micro Monitor
    get_user.c
    Copyright (C) 1994  Riccardo Facchetti && Andrea Marangoni

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

#include <pwd.h>
#include <grp.h>
#include <sys/stat.h>
#include <string.h>
#if defined(NCURSES)
# include "ncurses.h"
#else
# include <curses.h>
#endif /* NCURSES */
#include "define.h"
#include "extern.h"

static struct _myuser {
	long uid, procs, t_size, t_rss;
	char name [ MAX_LOGIN + 1 ];
} user [ MAX_PROC ];

static int actu = 0;   /* actual user in system */

void set_users ()
{
	register int i;

	for ( i = 0; i < MAX_PROC; ++i ) {
		user [ i ].name[ 0 ] = '\0';
		user [ i ].uid = -1L;
		user [ i ].procs = user [ i ].t_size = user [ i ].t_rss = 0L;
	}
}

char *get_user ( index )
int index;
{
	register int i,uid = pbuf [ index ].pst_uid;

	for ( i = 0 ; i < actu && user [ i ].uid != uid; ++i );

	if ( i == actu ) {   /* new_user in system */
		struct passwd *buf;

		if ( ( buf = getpwuid ( uid ) ) == NULL ) /* problem */
			sprintf ( user [ actu ].name, "(%d)", uid );
		else
			strncpy ( user [ actu ].name,  buf->pw_name, MAX_LOGIN );
			
		user [ actu ].name[ MAX_LOGIN ] = '\0';
		user [ actu ].uid = uid;
		++actu;
	}
				
	return ( user [ i ].name );
}

long get_user_id ( name )
char *name;
{
	struct passwd *pwd;
	long uid;

	if ( !name )
		return ( -1L );

	if ( actu ) {
		register int i;

		for ( i = 0; i < actu && strcmp ( user [ i ].name, name); ++i );
		if ( i != actu )
			return ( user [ i ].uid );
	}

	if ( ( pwd = getpwnam ( name ) ) == NULL )
		uid = -1L;
	else
		uid = pwd->pw_uid;
	
	return ( uid );
}

void user_info()
{
	struct passwd *buf;
	struct group *grp;
	struct stat home;
	
#ifndef NO_CURSES
	if ( look_lines() < 24 ) {
		curses_error( "Sorry, you must have at least 24 lines." );
		return;
	}

	clear_screen();
#define PF (void)printw
#else
#define PF (void)printf
#endif

	PF  ( "\nTry to get information about uid %ld.\n", look_uid );

	if ( ( buf = getpwuid ( look_uid ) ) == NULL ) { /* problem */
		PF  ( "\n\nProblems...! Can't get struct passwd of uid %ld\n",
			look_uid );
	} else {
		grp = getgrgid ( buf->pw_gid );


		PF  ( "\n\nName    : %s\n", buf->pw_name );
		PF  ( "Passwd  : \"%s\"", buf->pw_passwd );

#if !defined(linux)
		if ( buf->pw_passwd [ 0 ] == '*' && 
			buf->pw_audid == -1 && buf->pw_audid == -1 )
			PF ( " ( Probably You don't have permission to read encrypted password )\n");
		else
#endif /* !linux */
			PF ("\n");

		PF  ( "Uid     : %d ( %s )\n", buf->pw_uid, buf->pw_name );
		PF  ( "Gid     : %d ( %s )\n", buf->pw_gid,
			grp ? grp->gr_name : "???" );
#if !defined(linux)
		PF  ( "Age     : %s\n", buf->pw_age );
		PF  ( "Comment : %s\n", buf->pw_comment );
#endif /* !linux */
		PF  ( "Gecos   : %s\n", buf->pw_gecos );
		PF  ( "Dir     : %s", buf->pw_dir );

		if ( stat ( buf->pw_dir, &home ) == -1 )
			PF ( " ( Can't read permission )\n" );
		else
			PF ( " ( %c%c%c%c%c%c%c%c%c )\n",
				home.st_mode & S_IRUSR ? 'r' : '-',
				home.st_mode & S_IWUSR ? 'w' : '-',
				home.st_mode & S_IXUSR ? 'x' : '-',
				home.st_mode & S_IRGRP ? 'r' : '-',
				home.st_mode & S_IWGRP ? 'w' : '-',
				home.st_mode & S_IXGRP ? 'x' : '-',
				home.st_mode & S_IROTH ? 'r' : '-',
				home.st_mode & S_IWOTH ? 'w' : '-',
				home.st_mode & S_IXOTH ? 'x' : '-' );

		PF  ( "Shell   : %s\n", buf->pw_shell );
#if !defined(linux)
		PF  ( "Audid   : %ld\n", buf->pw_audid );
		PF  ( "Audflg  : %ld\n", buf->pw_audflg );
#endif /* !linux */
	}

#ifndef NO_CURSES
	( void )curses_get_string ( "Type <RETURN> to return to previous state." );
#endif
}

void get_user_sum ( nproc, processes, t_size, t_rss )
int nproc;
long  *processes, *t_size, *t_rss;
{
	register int i;

	*processes = *t_size = *t_rss = 0L;

	for ( i = 0; i < nproc; ++i ) { 
		if ( pbuf [ i ].pst_uid != look_uid )
			continue;
		(*processes)++;
#if !defined(linux)
		*t_size += pbuf [ i ].pst_dsize + pbuf [ i ].pst_tsize + pbuf [ i ].pst_ssize;
#else
		*t_size += pbuf [ i ].pst_vsize;
#endif /* linux */
		*t_rss += pbuf [ i ].pst_rssize;
	}
}

void Print_Who( nproc )
int nproc;
{
	register int i,j;

	/********** setup_user *********/

	for ( i = 0; i < MAX_PROC; ++i ) 
		user [ i ].procs = user [ i ].t_size = user [ i ].t_rss = 0L;

	for ( i = 0; i < nproc; ++i )
		for ( j = 0; j < actu; ++j ) {
			if ( pbuf [ i ].pst_uid != user [ j ].uid )
				continue;
			user [ j ].procs++;
#if !defined(linux)
			user [ j ].t_size += pbuf [ i ].pst_dsize + pbuf [ i ].pst_tsize + pbuf[ i ].pst_ssize;
#else
			user [ j ].t_size += pbuf [ i ].pst_vsize;
#endif /* !linux */
			user [ j ].t_rss += pbuf [ i ].pst_rssize;
		}

	/******************************/


#ifndef NO_CURSES
#define PF (void)printw
#else
#define PF (void)printf
#endif

	/* this loop is similar to the main loop */
	PF ( "\nUSER        UID    NPROCS    TOTAL_SIZE        TOTAL_RSS\n" );

	for ( i = j = displayed = 0; i < actu; ++i ) {

#ifndef NO_CURSES
		if ( j++ < pages*Num_display_proc )
			continue;
#endif
		if ( user [ i ].procs == 0 )   /* user logout */
			continue;

		if ( Num_display_proc >= 0 && displayed >= Num_display_proc )
			break;

		displayed++;

		PF ( "%-*s  %5ld    %4ld        %s", MAX_LOGIN,
			user [ i ].name,
			user [ i ].uid,
			user [ i ].procs,
			get_dimension ( user [ i ].t_size ));
		PF ( "          %s\n",
			get_dimension ( user [ i ].t_rss ));
	}
}
