/*
 * man.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 *
 * Wed Dec 23 13:26:03 1992: Rik Faith (faith@cs.unc.edu) applied bug fixes
 * supplied by Willem Kasdorp (wkasdo@nikhefk.nikef.nl)
 *
 * Sat Apr 30 11:54:08 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk)
 * Extensively rewritten, ANSI-fied, and upgraded to include self maintaining 
 * global database of available man pages, and to (optionally) adhere to the 
 * new filesystem standard (FSSTND). Also use newer glob routines, culled from
 * the bash-1.13.5 distribution.
 * New code mandb.c added to initialise the global database. 
 */

#ifdef FSSTND
#define CATPATH(path)	catpath(path)
char *catpath (char *path);
#else
#define CATPATH(path)	path
#endif

#ifndef EXIT_FAILURE
#define EXIT_FAILURE 1
#endif

#include <stdio.h>
#include <ctype.h>
#include <string.h>
#include <pwd.h>
#include <sys/file.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <signal.h>
#include <stdlib.h>
#include <unistd.h>
#include <time.h>
#include <termios.h>

#include "config.h"
#include "mydbm.h"
#include "gripes.h"
#include "util.h"
#include "statdir.h"
#include "manp.h"
#include "newman.h"
#include "dbver.h"
#include "convert_name.h"
#include "db_management.h"

#ifdef HAS_GETOPT_LONG
#  include <getopt.h>
#endif /* HAS_GETOPT_LONG */

/* version.c */
void ver(void);

/* man.c */
void usage(void);
char **add_dir_to_mpath_list(char **mp, char *p);
void man_getopt(int argc, char *argv[]);
char *is_section(char *name);
void do_extern(char *prog, char *name);
char **glob_for_file(char *path, char *section, char *name, int cat);
char **make_name(char *path, char *section, char *name, int cat);
int display_cat_file(char *file);
void add_directive(int *first, char *d, char *file, char *buf);
int parse_roff_directive(char *cp, char *file, char *buf);
char *make_roff_command(char *file);
char **get_section_list (void);
int man (char *name);
int format_and_display_nosave (char *name);
int do_prompt (char *name);

extern char **glob_vector ();
extern char **glob_filename ();
extern uid_t ruid;
extern uid_t euid;

/* externals */
int debug;
short quiet = 1;
short db = 1;
short findall;
short dups = DUPS;
char *manp;
char *manpathlist[MAXDIRS];
char *alt_system_name = NULL;
char *prognam;
char *database;
char *pager;
MYDBM_FILE dbf; 

/* locals */
static char **section_list;
static char *section;
static char *colon_sep_section_list;
static char *roff_directive;
static short apropos;
static short whatis;
static short print_where;
static short catman;
static short man_file = 0;

#ifdef SECURE_MAN_UID
static roff_command_secure;
#endif

static int troff = 0;

#ifdef HAS_TROFF
static char *troff_device = NULL;
#endif /* HAS_TROFF */

int main (int argc, char *argv[])
{
	int status = 0;
	char *nextarg;
	char *tmp;
	extern int optind;

	prognam = mkprogname (argv[0]);
	ruid = getuid ();
	euid = geteuid ();
/*
 * if the user happens to be root and man is installed setuid, forget all
 * of the priv dropping etc. Just make sure that any cat files produced
 * are owned by MAN_OWNER
 */
#ifdef SECURE_MAN_UID
	if (!ruid)
#  if defined(POSIX_SAVED_IDS)
		seteuid(ruid);
#  elif defined(SYSV_SAVED_IDS)
		setuid(ruid);
#  elif defined(HAVE_SETREUID)
		setreuid(ruid, ruid);
#  endif
#endif

	man_getopt (argc, argv);

	if (optind == argc)
		gripe_no_name ((char *) NULL);

	if (man_file) {
		while (optind < argc) {
			format_and_display_nosave (argv[optind]);
			optind++;
		}
		return 0;
	}

	section_list = get_section_list ();

	if (optind == argc - 1){
		tmp = is_section (argv[optind]);

		if (tmp != NULL)
			gripe_no_name (tmp);
	}

	while (optind < argc){
		nextarg = argv[optind++];

		/*
     		 * See if this argument is a valid section name.  If not,
      		 * is_section returns NULL.
      		 */

		tmp = is_section (nextarg);

		if (tmp != NULL){
			section = tmp;

			if (debug)
				fprintf (stderr, "\nsection: %s\n", section);

			continue;
		}

		if (apropos)
			do_extern (BINDIR "/apropos", nextarg);
		else if (whatis)
			do_extern (BINDIR "/whatis", nextarg);

		/* this is where we actually start looking for the man page */

		else {
			status = (int) do_manual(nextarg);

			if (status == 0)
				gripe_not_found (nextarg, section, alt_system_name);
			else
				if (debug)
					fprintf(stderr,
						"\nFound %d man pages\n", status);
		}
	}
	return 0;
}

void usage (void)
{
#ifdef HAS_TROFF
	printf("usage: %s [-adfhkwlV] [section] [-M path] [-P pager] [-S list]\n"
               "           [-m system] [-p string] [-t [-T device]] name/file ...\n", prognam);
#else
	printf("usage: %s [-adfhkwlV] [section] [-M path] [-P pager] [-S list]\n"
               "           [-m system] [-p string] name/file ...\n", prognam);
#endif

	printf("-a --all                    find all matching entries.\n"
               "-d --debug                  produce debugging info.\n"
               "-f --whatis                 same as whatis(1).\n"
               "-k --apropos                same as apropos(1).\n"
               "-w --where --location       print location of man page(s) only.\n"
               "-l --local-file             use manpage argument as local filename.\n"
#ifdef HAS_TROFF
	       "-t --troff                  use troff to format pages to stdout.\n"
	       "-T --troff-device device    use troff with selected device.\n"
#endif
               "-M --manpath path           set search path for manual pages to `path'.\n"
               "-P --pager pager            use program `pager' to display pages.\n"
               "-S --sections list          use colon separated section list.\n"
	       "-m --systems system         search for man pages on alternate system(s).\n"
	       "-p --preprocessor string    string tells which preprocessors to run.\n"
               "                              e - [n]eqn(1)   p - pic(1)    t - tbl(1)\n"
               "                              g - grap(1)     r - refer(1)  v - vgrind(1)\n"
               "-V --version                show version.\n"
               "-h --help                   show this usage message.\n");
}

char **add_dir_to_mpath_list (char **mp, char *p)
{
	int status;
	char cwd[BUFSIZ];

	status = is_directory (p);

	if (status < 0){
		fprintf (stderr, "Warning: couldn't stat file %s!\n", p);
	}
	else if (status == 0){
		fprintf (stderr, "Warning: %s isn't a directory!\n", p);
	}
	else {
		/* deal with relative paths */

		if (*p != '/') {
			if (getcwd(cwd, BUFSIZ - 2 - strlen(p)) == NULL) {
				perror("man: getcwd():");
				exit(1);
			}
			strcat(cwd, "/");
			*mp = strdup(strcat(cwd, p));
		} else 
			*mp = strdup (p);

		if (strstr(*mp, "/man") == NULL && strstr(*mp, "/MAN") == 0) {

			/* not really a man tree after all */

			fprintf (stderr, 
			  "Warning: %s does not have a man tree component\n", 
			  *mp);
			free(*mp);
		} else {
			if (debug)
				fprintf (stderr, 
				  "adding %s to manpathlist\n", 
				  *mp);
			mp++;
		}
	}
	return mp;
}

/*
 * Get options from the command line and user environment.
 */

#ifdef HAS_GETOPT_LONG
static struct option long_options[] =
{
    {"local-file", no_argument, 	0, 'l'},
    {"manpath", optional_argument, 	0, 'M'},
    {"pager", optional_argument, 	0, 'P'},
    {"sections", optional_argument, 	0, 'S'},
    {"all", no_argument, 		0, 'a'},
    {"debug", no_argument, 		0, 'd'},
    {"whatis", no_argument, 		0, 'f'},
    {"help", no_argument, 		0, 'h'},
    {"apropos", no_argument, 		0, 'k'},
    {"version", no_argument, 		0, 'V'},
    {"system", optional_argument, 	0, 'm'},
    {"preprocessor", optional_argument,	0, 'p'},
    {"location", no_argument, 		0, 'w'},
    {"where", no_argument,		0, 'w'},
#endif /* HAS_GETOPT_LONG */

#ifdef HAS_TROFF

#  ifdef HAS_GETOPT_LONG
    {"troff", no_argument, 		0, 't'},
    {"troff-device", optional_argument,	0, 'T'},
    {0, 0, 0, 0}
};
#  endif /* HAS_GETOPT_LONG */

static char args[] = "lM:P:S:adfhkVm:p:tT:w";

#else /* HAS_TROFF */

#  ifdef HAS_GETOPT_LONG

    {0, 0, 0, 0}
};
#  endif /* HAS_GETOPT_LONG */

static char args[] = "lM:P:S:adfhkVm:p:w";

#endif /* HAS_TROFF */

void man_getopt (int argc, char *argv[])
{
	int c;
	char *p;
	char *end;
	char **mp;
	extern char *optarg;

#ifdef HAS_GETOPT_LONG
	int option_index; /* not used, but required by getopt_long() */

	while ((c = getopt_long (argc, argv, args,
				 long_options, &option_index)) != EOF){
#else /* HAS_GETOPT_LONG */

	while ((c = getopt (argc, argv, args)) != EOF){

#endif /* HAS_GETOPT_LONG */

		switch (c){

			case 'l':
				man_file = 1;
				break;
			case 'M':
				manp = strdup (optarg);
				db--;
				break;
		    	case 'P':
				pager = strdup (optarg);
				break;
		    	case 'S':
				colon_sep_section_list = strdup (optarg);
				db--;
				break;
			case 'V':
				ver();
				exit(0);
		    	case 'a':
				findall++;
				break;
			case 'c':
				catman++;
				break;
		    	case 'd':
				debug++;
				break;
		    	case 'f':
				if (troff)
					gripe_incompatible ("-f and -t");
				if (apropos)
					gripe_incompatible ("-f and -k");
				if (print_where)
					gripe_incompatible ("-f and -w");
				whatis++;
				break;
		    	case 'k':
				if (troff)
					gripe_incompatible ("-k and -t");
				if (whatis)
					gripe_incompatible ("-k and -f");
				if (print_where)
					gripe_incompatible ("-k and -w");
				apropos++;
				break;
		    	case 'm':
				alt_system_name = strdup (optarg);
				dups = 0;
				break;
		    	case 'p':
				roff_directive = strdup (optarg);
				break;
#ifdef HAS_TROFF
		    	case 't':
				if (apropos)
					gripe_incompatible ("-t and -k");
				if (whatis)
					gripe_incompatible ("-t and -f");
				if (print_where)
					gripe_incompatible ("-t and -w");
				troff++;
				break;

			case 'T':
				if (apropos)
					gripe_incompatible ("-t and -k");
				if (whatis)
					gripe_incompatible ("-t and -f");
				if (print_where)
					gripe_incompatible ("-t and -w");
				if (!troff)
					gripe_incompatible ("-T before -t, or -t not selected");
				troff_device = optarg;
				break;
#endif
		    	case 'w':
				if (apropos)
					gripe_incompatible ("-w and -k");
				if (whatis)
					gripe_incompatible ("-w and -f");
				if (troff)
					gripe_incompatible ("-w and -t");
				print_where++;
				break;
		    	case 'h':
		    		usage();
		    		exit(0);
		    	default:
				usage();
				exit(1);
			}
		}

	if (pager == NULL || *pager == '\0' || *pager == '-')
		if ((pager = getenv ("PAGER")) == NULL)
			pager = strdup (PAGER);

	if (debug)
		fprintf (stderr, "\nusing %s as pager\n", pager);

	/* If we are going to format a file, don't bother looking
	   for manpath: it may cause unwanted error messages */
	if (man_file)
		return;

	if (manp == NULL){
		if ((manp = manpath (0, alt_system_name)) == NULL)
			gripe_manpath ();

		if (debug)
			fprintf (stderr,
				"\nsearch path for pages determined by manpath is\n%s\n\n",
				manp);
	}

	/*
	 * Expand the manpath into a list for easier handling.
	 */

	mp = manpathlist;
	for (p = manp;; p = end + 1){
		if ((end = strchr (p, ':')) != NULL)
			*end = '\0';

		mp = add_dir_to_mpath_list (mp, p);
		if (end == NULL)
			break;

		*end = ':';
	}
	*mp = NULL;
}

/*
 * Check to see if the argument is a valid section number.  If the
 * first character of name is a numeral, or the name matches one of
 * the sections listed in section_list, we'll assume that it's a section.
 * The list of sections in config.h simply allows us to specify oddly
 * named directories like .../man3f.  Yuk.
 */
char *is_section (char *name)
{
	char **vs;

	for (vs = section_list; *vs != NULL; vs++)
		if ((strcmp (*vs, name) == 0) || (isdigit (name[0])
						  && !isdigit (name[1])))
			return strdup (name);

	return NULL;
}

/*
 * Try to find the man page corresponding to the given name.  The
 * reason we do this with globbing is because some systems have man
 * page directories named man3 which contain files with names like
 * XtPopup.3Xt.  Rather than requiring that this program know about
 * all those possible names, we simply try to match things like
 * .../man[sect]/name[sect]*.  This is *much* easier.
 *
 * Note that globbing is only done when the section is unspecified.
 */
char **glob_for_file (char *path, char *section, char *name, int cat)
{
	char pathname[BUFSIZ];
	char **gf;

	if (cat)
		sprintf (pathname, "%s/cat%s/%s.%s*", CATPATH (path), section, name, section);
	else
		sprintf (pathname, "%s/man%s/%s.%s*", path, section, name, section);

	if (debug)
		fprintf (stderr, "globbing %s\n", pathname);

	gf = glob_filename (pathname);

	if ((gf == (char **) -1 || *gf == NULL) && isdigit (*section)){
		if (cat)
			sprintf (pathname, "%s/cat%s/%s.%c*", CATPATH (path), section, name, *section);
		else
			sprintf (pathname, "%s/man%s/%s.%c*", path, section, name, *section);

		gf = glob_filename (pathname);
	}

	/*
  	 * If we're given a section that looks like `3f', we may want to try
  	 * file names like .../man3/foo.3f as well.  This seems a bit
  	 * kludgey to me, but what the hey...
  	 */

	if (section[1] != '\0'){
		if (cat)
			sprintf (pathname, "%s/cat%c/%s.%s", CATPATH (path), section[0], name, section);
		else
			sprintf (pathname, "%s/man%c/%s.%s", path, section[0], name, section);

		gf = glob_filename (pathname);
	}
	return gf;
}

/*
 * Return an un-globbed name in the same form as if we were doing
 * globbing.
 */
char **make_name (char *path, char *section, char *name, int cat)
{
	int i = 0;
	static char *names[3];
	char buf[BUFSIZ];

	if (cat)
		sprintf (buf, "%s/cat%s/%s.%s", CATPATH (path), section, name, section);
	else
		sprintf (buf, "%s/man%s/%s.%s", path, section, name, section);

	if (access (buf, R_OK) == 0)
		names[i++] = strdup (buf);

	/*
  	 * If we're given a section that looks like `3f', we may want to try
  	 * file names like .../man3/foo.3f as well.  This seems a bit
  	 * kludgey to me, but what the hey...
  	 */

	if (section[1] != '\0'){
		if (cat)
			sprintf (buf, "%s/cat%c/%s.%s", CATPATH (path), section[0], name, section);
		else
			sprintf (buf, "%s/man%c/%s.%s", path, section[0], name, section);

		if (access (buf, R_OK) == 0)
			names[i++] = strdup (buf);
	}
	names[i] = NULL;
	return &names[0];
}

/*
 * Simply display the preformatted page.
 */
int display_cat_file (char *file)
{
	char command[BUFSIZ];

	if (access (file, R_OK) == 0){

#ifdef COMPRESS
		sprintf (command, DECOMPRESSOR " %s | %s", file, pager);
#else
		sprintf (command, "%s %s", pager, file);
#endif
		do_system_command_drop_privs (command);

		/* 
		 * Even if the system command fails, 
		 * the page is accessable. If we return 0, 
		 * it'll get deleted from the db ! 
		 */

		return 1; 
	}
	return 0; /* file is not world readable */
}

void add_directive (int *first, char *d, char *file, char *buf)
{
	if (strcmp (d, "") != 0){
		if (*first){
			*first = 0;
			strcpy (buf, d);
			strcat (buf, " ");
			strcat (buf, file);
		}
		else {
			strcat (buf, " | ");
			strcat (buf, d);
		}
	}
}

int parse_roff_directive (char *cp, char *file, char *buf)
{
	char c;
	int first = 1;
	int tbl_found = 0;
	char *eqn;

	while ( !isspace(c = *cp++) && c != '\0'){
		switch (c){
		  
		case 'e':

			if (debug)
				fprintf (stderr, "found eqn(1) directive\n");
#ifdef HAS_TROFF
			if (troff) {
				if (troff_device) {
					eqn = (char *) 
					  malloc (sizeof EQN + strlen(troff_device) + 3);
					strcpy(eqn, EQN " -T");
					strcat(eqn, troff_device);
					add_directive (&first, eqn, file, buf);
					free(eqn);
				} else {
					add_directive (&first, EQN, file, buf);
				}
			} else
				add_directive (&first, NEQN, file, buf);
#endif
			break;

		case 'g':

			if (debug)
				fprintf (stderr, "found grap(1) directive\n");

			add_directive (&first, GRAP, file, buf);

			break;

		case 'p':

			if (debug)
				fprintf (stderr, "found pic(1) directive\n");

			add_directive (&first, PIC, file, buf);

			break;

		case 't':

			if (debug)
				fprintf (stderr, "found tbl(1) directive\n");

			tbl_found++;
			add_directive (&first, TBL, file, buf);
			break;

		case 'v':

			if (debug)
				fprintf (stderr, "found vgrind(1) directive\n");

			add_directive (&first, VGRIND, file, buf);
			break;

		case 'r':

			if (debug)
				fprintf (stderr, "found refer(1) directive\n");

			add_directive (&first, REFER, file, buf);
			break;

		default:

			return -1;
		}
	}

	if (first)
		return 1;

#ifdef HAS_TROFF
	if (troff){
		strcat (buf, " | " TROFF);
		if (troff_device) {
			strcat (buf, " -T");
			strcat (buf, troff_device);
		}
	}
	else
#endif
	{
		strcat (buf, " | ");
		strcat (buf, NROFF);
	}

	if (tbl_found && !troff && strcmp (COL, "") != 0){
		strcat (buf, " | ");
		strcat (buf, COL);
	}
	return 0;
}

char *make_roff_command (char *file)
{
	FILE *fp;
	char line[BUFSIZ];
	static char buf[BUFSIZ];
	int status;
	char *cp;

	if (roff_directive != NULL){
		if (debug)
			fprintf (stderr, "parsing directive from command line\n");

		status = parse_roff_directive (roff_directive, file, buf);

		if (status == 0)
			return buf;

		if (status == -1)
			gripe_roff_command_from_command_line (file);
	}

	if ((fp = fopen (file, "r")) != NULL){
		cp = &line[0];
		fgets (line, 100, fp);
		if (*cp++ == '\'' && *cp++ == '\\' && *cp++ == '"' && *cp++ == ' '){

			if (debug)
				fprintf (stderr, "parsing directive from file\n");

			status = parse_roff_directive (cp, file, buf);

			fclose (fp);

			if (status == 0) {
#ifdef SECURE_MAN_UID
				/* the command is considered secure in this case*/
				roff_command_secure = 1;
#endif
				return buf;
			}
			if (status == -1)
				gripe_roff_command_from_file (file);
		}
	}
#ifndef SECURE_MAN_UID
	else {
		/*
      		 * Is there really any point in continuing to look for
      		 * preprocessor options if we can't even read the man page source?
		 * Yes in case man is setuid and I want to do a:
		 *	$ man -l man.1
		 * on the following file:
		 * -rw-------    normaluser   other    man.1
      		 */
		  gripe_reading_man_file (file);
		  return NULL;
	}
#endif

	if ((cp = getenv ("MANROFFSEQ")) != NULL){
	
		if (debug)
			fprintf (stderr, "parsing directive from environment\n");

		status = parse_roff_directive (cp, file, buf);

		if (status == 0)
			return buf;

		if (status == -1)
			gripe_roff_command_from_env ();
	}
	if (debug)
		fprintf (stderr, "using default preprocessor sequence\n");

#ifdef SECURE_MAN_UID
	/* the command is considered secure */
	roff_command_secure = 1;
#endif
#ifdef HAS_TROFF
	if (troff){
		if (strcmp (TBL, "") != 0){
			strcpy (buf, TBL);
			strcat (buf, " ");
			strcat (buf, file);
			strcat (buf, " | " TROFF);
			if (troff_device) {
				strcat (buf, " -T");
				strcat (buf, troff_device);
			}
		}
		else {
			strcpy (buf, TROFF);
			if (troff_device) {
				strcat (buf, " -T");
				strcat (buf, troff_device);
			}
			strcat (buf, " ");
			strcat (buf, file);
		}
	}
	else
#endif
	{
		if (strcmp (TBL, "") != 0){
			strcpy (buf, TBL);
			strcat (buf, " ");
			strcat (buf, file);
			strcat (buf, " | ");
			strcat (buf, NROFF);
		}
		else {
			strcpy (buf, NROFF);
			strcat (buf, " ");
			strcat (buf, file);
		}

		if (strcmp (COL, "") != 0){
			strcat (buf, " | ");
			strcat (buf, COL);
		}
	}
	return buf;
}

/*
 * Try to format the man page and create a new formatted file.  Return
 * 1 for success and 0 for failure.
 */
int make_cat_file (char *path, char *man_file, char *cat_file)
{
	FILE *fp;
	struct stat buf;
	char *roff_command;
	char command[BUFSIZ];
	char *tmpfile;

	if ( (tmpfile = tempnam(TMPDIR, "catpg")) == NULL)
		return 0;
				
	if ((fp = fopen (cat_file, "w")) != NULL) {
		fclose (fp);
		remove (cat_file);

#ifdef SECURE_MAN_UID
		roff_command_secure = 0;
#endif
		roff_command = make_roff_command (man_file);
#ifdef SECURE_MAN_UID
		if (roff_command == NULL || roff_command_secure == 0)
#else
		if (roff_command == NULL)
#endif
			return 0;
		else
#ifdef COMPRESS
			sprintf (command, "umask 022 ; cd %s ; %s | %s > %s", 
				path, roff_command, COMPRESSOR, tmpfile);
#else
			sprintf (command, "umask 022 ; cd %s ; %s > %s",
				path, roff_command, tmpfile);
#endif

		if (!catman)
			fputs ("Formatting page.  Wait...", stderr);

		if (do_system_command_drop_privs (command)) {

			/* check file before copying */

			int status = 0;

			if (!debug) {
				if (stat(tmpfile, &buf) == -1) {
					fprintf(stderr, "make_cat_file: stat: ");
					perror(tmpfile);
					exit(1);
				} else if ( !S_ISREG(buf.st_mode) ) {
					fprintf(stderr, "%s not a regular file.\n", 
					  tmpfile);
					remove(tmpfile);
					exit(1);
				}
			}

			if (debug) {
				fprintf(stderr, CP " %s %s\n", tmpfile, cat_file);
			} else {
				pid_t child;
	
				child = vfork ();
				if (child < 0) {
					perror ("can't fork");
					status = 1;
				} else if (child == 0) {
					umask ((S_IRWXU|S_IRWXG|S_IRWXO) ^ CATMODE);
					execle(CP, "cp", tmpfile, cat_file, NULL, NULL);
					perror ("can't exec "CP);
					exit (1);
				}
				wait(&status);
			}

			/* tmpfile is owned by the normal user and as such, must 
			   be remove()'d by her */
			drop_effective_privs();
			remove (tmpfile);
			regain_effective_privs();

			if (status) {
				fprintf(stderr, 
					"Could not cp temp file to new location.\n");	
				remove (cat_file);
				exit(1);
			}
			if (!debug && chmod (cat_file, CATMODE)) {
				fprintf(stderr, "make_cat_file: chmod: ");
				perror(cat_file);
				remove(cat_file);
				exit(1);
			} else if (debug) {
				fprintf (stderr, "mode of %s is now %o\n", 
				  cat_file, CATMODE);
			}
#ifdef SECURE_MAN_UID
			if (!ruid) { /* are we root ? */
				if (!debug && chown (cat_file, euid, (uid_t) -1)) {
					fprintf(stderr, "make_cat_file: chown: ");
					perror(cat_file);
					remove(cat_file);
					exit(1);
				} else if (debug) {
					fprintf (stderr, "owner of %s is now (%d)\n",
					  cat_file, euid);
				}
			}
#endif /* SECURE_MAN_UID */
		} else {
			if (!catman)
				fputs(" aborted (sorry)\n", stderr);
			drop_effective_privs();
			remove(tmpfile);
			regain_effective_privs();
			exit(0);
		}

		if (!catman)
			fputs(" done\n", stderr);

		return 1;
	}
	else {
		if (debug)
			fprintf (stderr, 
			  "Couldn't open %s for writing with EUID = %d.\n", 
			  cat_file, geteuid());

		return 0;
	}
}

#ifdef COMP_SRC
/*
 * check raw_manfile to see if it's compressed. 
 * decompress it and return new name, if necessary.
 */
char *expanded_name(char *raw_manfile)
{
	char command[BUFSIZ];
	char *man_file;
	extern struct compress_ext decompress[];
	struct compress_ext *comp;
	
	if ( (man_file = strrchr(raw_manfile, '.')) == NULL)
		return NULL;

	for (comp = decompress; comp->ext; comp++)
		if (strcmp(man_file, comp->ext) == 0)
			break;

	if (comp->prog) {
		if ( (man_file = tempnam(TMPDIR, "manpg")) == NULL)
			return NULL;
		sprintf(command, "%s > %s", comp->prog, man_file);
		if (!do_system_command_drop_privs(command)) {
			remove(man_file);
			exit(0);
		}
	} else
		man_file = raw_manfile;
	
	return man_file;
}                       
#endif

/*
 * Try to format the man page source and save it, then display it.  If
 * that's not possible, try to format the man page source and display
 * it directly.
 *
 * Note that we've already been handed the name of the ultimate source
 * file at this point.
 */
int format_and_display (char *path, char *man_file)
{
	int status;
	register int found;
	char *roff_command;
	char command[BUFSIZ];
	char *cat_file;

	found = 0;

	if (access (man_file, R_OK) != 0)
		return 0;

	if (troff){ /* special case */
		roff_command = make_roff_command (man_file);
		if (roff_command == NULL)
			return 0;
		else
			sprintf (command, "cd %s ; %s", path, roff_command);
 
		if (do_system_command_drop_privs (command))
			found++;
		else {
			fputs("Aborted.\n", stderr);
			exit(0);
		}
	}
	else { /* normal case */

		/*
		 * Here we need to check against both possible (FSSTND) 
		 * cat locations for the file
		 */

		cat_file = convert_name(man_file, 0); /* /usr fs */
		if (debug)
			fprintf(stderr, "Checking status of %s\n", cat_file);
		status = is_newer (man_file, cat_file);
#ifdef FSSTND
		if ( (status == 1 || status == -2) &&
		  global_catpath(man_file) != NULL) { 
			free(cat_file);
			cat_file = convert_name(man_file, 1); /* /var fs */
			if (debug)
				fprintf(stderr, "Checking status of %s\n", cat_file);
			status = is_newer (man_file, cat_file);
		}
#endif
		/* check to see if we have a bogus place-holder */
		/* don't bother right now */
		
		if (status == 1 || status == -2 || (status & 4) ){
			/*
	  		 * Cat file is out of date.  Try to format and save it.
	  		 */

			if (print_where){
				printf ("%s\n", man_file);
				found++;
			}
			else if (! do_prompt (man_file)) {
				found = make_cat_file (path, man_file, cat_file);

				if (found){
					/*
		  			 * Creating the cat file worked.  Now just display it.
		  			 */

		  			if (catman)
		  				return found;
					(void) display_cat_file (cat_file);
				}
				else {
					/*
		  			 * Couldn't create cat file.  Just format it and
		  			 * display it through the pager.
		  			 */

		  			if (catman)
		  				return ++found;
					roff_command = make_roff_command (man_file);
					if (roff_command == NULL)
						return 0;
					else
						sprintf (command, "cd %s ; %s | %s", path,
						  roff_command, pager);
					if ( !(found = do_system_command_drop_privs (command)) ) {
						fputs("Aborted\n", stderr);
						exit(0);
					}
				}
			} else
				found = 1; /* we didn't show it, but it's here */
		}
		else if (access (cat_file, R_OK) == 0) {
			/*
	  		 * Formatting not necessary.  Cat file is newer than source
	  		 * file, or source file is not present but cat file is.
	  		 */
			if (print_where){
				printf ("%s (source: %s)\n", cat_file, 
				  status & 2 ? "NONE - straycat." : man_file);
				found++;
			}
			else {
				if (catman)
					return ++found;
				if (!do_prompt (man_file))
					found = display_cat_file (cat_file);
				else
					found = 1;
			}
		}
		free(cat_file);
	}
	return found;
}

/* The return code is < 0 on error, >= 0 otherwise (like a syscall!).
   FIXME -- This is nearly the same as above. It would be nice to make
   format_and_display() call format_and_display_nosave(). */
int format_and_display_nosave (char *filename)
{
	char *roff_command;

	if (access (filename, R_OK) != 0) {
		perror (filename);
		return -1;
	}
	if (troff){ /* special case */
		roff_command = make_roff_command (filename);
		if (roff_command == NULL)
			return -1;
 
		if (! do_system_command_drop_privs (roff_command)) {
			fputs("Aborted.\n", stderr);
			exit(0);
		}
	} else { /* normal case */
		char command[BUFSIZ];
		roff_command = make_roff_command (filename);
		if (roff_command == NULL)
			return -1;
		else
			sprintf (command, "%s | %s", roff_command, pager);

		if (! do_prompt (filename))
			if (! do_system_command_drop_privs (command)) {
				fputs("Aborted\n", stderr);
				exit(0);
			}
	}
	return 0;
}

/*
 * See if the preformatted man page or the source exists in the given
 * section.
 */
int try_section (char *path, char *section, char *name, int glob)
{
	int found = 0;
	int cat;
	char **names;
	char **np;

	if (debug){
		if (glob)
			fprintf (stderr, "trying section %s with globbing\n", section);
		else
			fprintf (stderr, "trying section %s without globbing\n", section);
	}

#ifndef NROFF_MISSING /* #ifdef NROFF */
	/*
  	 * Look for man page source files.
  	 */

	cat = 0;
	if (glob)
		names = glob_for_file (path, section, name, cat);
	else
		names = make_name (path, section, name, cat);

	if (names == (char **) -1 || *names == NULL)
		/*
    		 * No files match.  
    		 * See if there's a preformatted page around that
    		 * we can display.
    		 */
#endif /* NROFF_MISSING */
	{
		if (catman)
			return ++found;
		if (!troff){
			cat = 1;
			if (glob)
				names = glob_for_file (path, section, name, cat);
			else
				names = make_name (path, section, name, cat);

			if (names != (char **) -1 && *names != NULL){
				for (np = names; *np != NULL; np++){
					if (print_where){
						printf ("%s\n", *np);
						found++;
					}
					else {
						found += display_cat_file (*np);
					}
				}
			}
		}
	}
#ifndef NROFF_MISSING
	else {
		for (np = names; *np != NULL; np++){
			char *man_file;

			if ( (man_file = ult_src (*np, path)) == NULL)
				return 0;

			if (debug)
				fprintf (stderr, "found ultimate source file %s\n", man_file);

			found += format_and_display (path, man_file);
			free(man_file);
		}
	}
#endif /* NROFF_MISSING */
	return found;
}

/*
 * Search for manual pages.
 *
 * If preformatted manual pages are supported, look for the formatted
 * file first, then the man page source file.  If they both exist and
 * the man page source file is newer, or only the source file exists,
 * try to reformat it and write the results in the cat directory.  If
 * it is not possible to write the cat file, simply format and display
 * the man file.
 *
 * If preformatted pages are not supported, or the troff option is
 * being used, only look for the man page source file.
 *
 */
int man (char *name)
{
	int found = 0;
	int glob = 0;
	char **mp;
	char **sp;

	fflush (stdout);

	/* first of all check the database */

	if (db == 1) {
		if ((dbf = MYDBM_REOPEN (database)) != NULL) {

  /*
   *  Good, means we get a chance to do things properly.
   *
   *  First we try to find the page within the db. If it's found, great, we
   *  look it up. If there are multiple entries, we can show them all as usual
   */
			datum key, content;
	
			if (debug)
				  fprintf (stderr, "\nopened database file %s\n", database);

			if ( (dbver(dbf)) != 0 ) {
				MYDBM_CLOSE(dbf);
				return 0;
			}

			key.dptr = name;
			key.dsize = strlen (key.dptr) + 1;
	
			content = MYDBM_FETCH (dbf, key);
	
			MYDBM_CLOSE(dbf); 	/* for sanity's sake */
	
			/* if closing the db frees up content.dptr, we've had it here */
	
			if (content.dptr != NULL){
			
				/* 
				 * The db has an entry for the manpage, 
				 * which may be several
				 * pages concattenated with ':'s
				 */
	
				char *c1;
	
				if (debug)
					fprintf (stderr, "found man page(s) in db at %s\n",
					  content.dptr);
	
				while ((c1 = strrchr(content.dptr, ':')) != NULL){
					*c1 = '\0';
	
					found += try_db_manpage(++c1, key, section);
	
					if (found && !findall)  
						return found;
			 	}
	
				found += try_db_manpage(content.dptr, key, section);
	
				if (found && !findall)
					return found;
				
				MYDBM_FREE (content.dptr);
			}
			else {	
	
				/*
       	   		 	 * name not found in db, 
       	   		 	 * if we find it by globbing,
       	   		 	 * we should add it
       	   		 	 */
	
       	   		if (debug)
       	   			fprintf(stderr, "man page not found in %s\n", database);
			}
			return found;
		}
		else 
			/* db not found */
	
			if (debug){
				fputs("couldn't open db, even in READ_ONLY mode.\n", stderr);
		}

		/*
		 * if this is the global db, bummer, we've had it - otherwise
		 * return found and get on w/ it
		 */
		if (strcmp(database, GLOBAL_DB) != 0)
			return found;
	}
	else
		if (debug)
			fputs("Not using any databases.\n\n", stderr);
	if (section != NULL){
		for (mp = manpathlist; *mp != NULL; mp++){
			if (debug)
				fprintf (stderr, "\nsearching in %s\n", *mp);

			found += try_section (*mp, section, name, glob);

			if (found && !findall)	/* i.e. only do this section... */
				return found;
		}
	}
	else {
		for (sp = section_list; *sp != NULL; sp++){
			for (mp = manpathlist; *mp != NULL; mp++){
				if (debug)
					fprintf (stderr, "\nsearching in %s\n", *mp);

				glob = 1;

				found += try_section (*mp, *sp, name, glob);

				if (found && !findall)	/* i.e. only do this section... */
					return found;
			}
		}
	}
	return found;
}

char **get_section_list (void)
{
	int i;
	char *p;
	char *end;
	static char *tmp_section_list[100];

	if (colon_sep_section_list == NULL){

		  if ((p = getenv ("MANSECT")) == NULL)
			return std_sections;
		  else
			colon_sep_section_list = strdup (p);
	}

	i = 0;
	for (p = colon_sep_section_list;; p = end + 1){
		  if ((end = strchr (p, ':')) != NULL)
			*end = '\0';

		  tmp_section_list[i++] = strdup (p);

		  if (end == NULL)
			break;
	}

	tmp_section_list[i] = NULL;
	return tmp_section_list;
}

/* Between two man pages, this gives the user a chance to quit.
   Exits if user presses 'q', return 1 (means skip this page) if user presses
   's', return 0 on any other keypress*/
int do_prompt (char *name)
{
	static int pause = 0;	/* Don't pause on first man page */
	struct termios tsave, tset;
	char c;
	if (isatty (STDOUT_FILENO) && pause) {
		printf ("--Man--(Next: %s)\n", name);
		fflush (stdout);
		if (tcgetattr (STDIN_FILENO, &tsave) < 0)
			return 0;
		tset = tsave;
		tset.c_lflag &= ~(ICANON | ECHO);
		tset.c_cc[VMIN] = 1;
		tset.c_cc[VTIME] = 0;
		if (tcsetattr (STDIN_FILENO, TCSADRAIN, &tset) < 0)
			return 0;
		read (STDIN_FILENO, &c, 1);
		tcsetattr (STDIN_FILENO, TCSADRAIN, &tsave);
		switch (tolower (c)) {
		case 'q':
			exit (EXIT_FAILURE);
		case 's':	/* skip */
		case 'n':	/* no */
			return 1;
		default:
			return 0;
		}
	} else
		pause = 1;
	return 0;
}
