/*
 * manpath.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Wed May  4 15:44:47 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk): changes
 * to get_dirlist() and manpath().
 */

#define MANPATH_MAIN

#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <stdlib.h>

#include "config.h"
#include "manpath.h"
#include "manp.h"
#include "util.h"
#include "gripes.h"

DIRLIST list[MAXDIRS];
char *tmplist[MAXDIRS];

extern int debug;

char *cat_manpath(int quiet, char *alt_system)
{
	DIRLIST *manlist;
	size_t len = 0;
	char *catp, *path, *manp;

	manp = manpath(quiet, alt_system);
	catp = strdup(manp);

	for (path = strtok(catp, ":"); path; path = strtok(NULL, ":")) {
		for (manlist = list; manlist->mandir[0] != '\0'; manlist++)
			if (manlist->mandatory == MANDB_MAP
			  && strcmp(path, manlist->mandir) == 0) {
				len += strlen (manlist->bin) + 1;
				break;
			}
		if (manlist->mandir[0] == '\0')
			len += strlen(path) + 1;
	}

	free(catp);
	catp = (char *) malloc (len + 1);
	if (catp == NULL)
		gripe_alloc(len + 1, "catpath: catp");

	*catp = '\0';

	for (path = strtok(manp, ":"); path; path = strtok(NULL, ":")) {
		strcat(catp, ":");
		for (manlist = list; manlist->mandir[0] != '\0'; manlist++)
			if (manlist->mandatory == MANDB_MAP
			  && strcmp(path, manlist->mandir) == 0) {
				strcat(catp, manlist->bin);
				break;
			}
		if (manlist->mandir[0] == '\0') 
			strcat(catp, path);
	}
	
	return ++catp;
}		

/* Add to the manpathlist system dependent subdirs */
char *add_system_manpath (const char *systems, char *manpathlist)
{
	const char *system;
	char *path;
	char *manpath = NULL;
	size_t manpathsize = 0;

	if (systems == NULL)
		if ((systems = getenv ("SYSTEM")) == NULL)
			return manpathlist;
	if (*systems == '\0')
		return manpathlist;

	/* For each manpathlist[] component */
	
	for (path = strtok (manpathlist, ":"); path;
	     path = strtok (NULL, ":")) {
		char *next;

		/* For each systems component */
		
		for (system = systems; system;
		     system = next) {
			size_t syslen;
			char *newdir;
			size_t newdirlen;
			int status;
			next = strpbrk (system, ",:" );
			syslen = next ? next++ - system : strlen (system);

			/* Check if system is 'man' (native) */

			if (strncmp(system, "man", syslen) == 0)
				continue;
			newdirlen = strlen (path) + syslen + 2;
			newdir = (char *) malloc (newdirlen);

			/* add it to the manpath component */

			strcpy (newdir, path);
			strcat (newdir, "/");
			strncat (newdir, system, syslen);

			/* and see if it exists */

			status = is_directory (newdir);
			if (status == 0)
				fprintf (stderr,
					 "Warning: %s isn't a directory!\n",
					 newdir);
			else if (status == 1) {
				if (debug)
					fprintf (stderr,
						 "adding %s to manpathlist\n",
						 newdir);
				manpath = (char *) realloc (manpath,
						   manpathsize + newdirlen);
				if (manpath == NULL) {
					perror ("FATAL realloc failed");
					exit (EXIT_FAILURE);
				}
				/* Add to manpath */
				if (manpathsize)
					strcat (manpath, ":");
				else
					*manpath = '\0';
				strcat (manpath, newdir);
				manpathsize += newdirlen;
			} else if (debug) {
				fputs ("can't stat ", stderr);
				perror (newdir);
			}
			free (newdir);
		}
		/* Add base directory */
		manpath = (char *) realloc (manpath, manpathsize + strlen (path) + 1);
		if (manpathsize)
			strcat (manpath, ":");
		else
			*manpath = '\0';
		strcat (manpath, path);
		manpathsize += strlen (path) + 1;
	}
	return manpath;
}

/*
 * If the environment variable MANPATH is set, return it.
 * If the environment variable PATH is set and has a nonzero length,
 * try to determine the corresponding manpath, otherwise, return the
 * default manpath.
 *
 * The manpath.config file is used to map system wide /bin directories
 * to top level man page directories.
 *
 * For directories which are in the user's path but not in the
 * manpath.config file, see if there is a subdirectory `man' or `MAN'.
 * If so, add that directory to the path.  Example:  user has
 * $HOME/bin in his path and the directory $HOME/bin/man exists -- the
 * directory $HOME/bin/man will be added to the manpath.
 */
char *manpath (int perrs, const char *systems)
{
	int len;
	char *manpathlist;
	char *path;

	/* need to read config file even if MANPATH set, for mandb(8) */
	
	if (get_dirlist ()) {
		gripe_reading_mp_config (CONFIG_FILE);
	}

	if ((manpathlist = getenv ("MANPATH")) != NULL) {
		/*
    		 * This must be it.
    		 */
		if (perrs)
			fprintf (stderr,
			  "(Warning: MANPATH environment variable set,"
			  " ignoring " CONFIG_FILE ")\n");
		return add_system_manpath (systems, strdup (manpathlist));
	}

	if ((path = getenv ("PATH")) == NULL) {
		/*
		 * Things aren't going to work well, but hey...
		 */
		if (perrs)
			fputs ("Warning: path not set\n", stderr);
		return add_system_manpath (systems, def_path (perrs, MANDATORY));
	} else {
		if ((len = strlen (path)) == 0) {
			/*
			 * Things aren't going to work well here either...
			 */
			if (perrs)
				fputs ("Warning: path set but has zero length\n", stderr);
			return add_system_manpath (systems, def_path (perrs, MANDATORY));
		}
		return add_system_manpath (systems, get_manpath (perrs, path));
	}
}

/*
 * Get the list of bin directories and the corresponding man
 * directories from the manpath.config file.
 *
 * This is ugly.
 * Not quite so ugly anymore :-) Wilf. Wed Apr 27 00:20:32 BST 1994
 *
 */
int get_dirlist (void)
{
	char *bp;
	char buf[BUFSIZ];
	DIRLIST *dlp = list;
	FILE *config;
	int converted;

	if ((config = fopen (CONFIG_FILE, "r")) == NULL)
		gripe_getting_mp_config (CONFIG_FILE);

	while ((bp = fgets (buf, BUFSIZ, config)) != NULL) {

		while ( isspace(*bp) )
			bp++;
			
		if (*bp == '#' || *bp == '\0')
			continue;

		else if (sscanf(bp, "MANBIN %*s") == 1)
			continue;
 		
		else if (sscanf(bp, "MANDATORY_MANPATH %s", dlp->mandir) == 1) {
			dlp->mandatory = MANDATORY;
			if (debug)
				fprintf (stderr, "found mandatory man directory %s\n",
				  dlp->mandir); 
			dlp++;
		} else if (sscanf(bp, "MANPATH_MAP %s %s", dlp->bin, 
		  dlp->mandir) == 2) {
			dlp->mandatory = MANPATH_MAP;
			if (debug)
				fprintf (stderr, "found manpath map %s --> %s\n",
				  dlp->bin, dlp->mandir);
			dlp++;
		} else if ( (converted = 
		  sscanf(bp, "MANDB_MAP %s %s", dlp->mandir, dlp->bin)) > 0) {
		  	dlp->mandatory = MANDB_MAP;
		  	if (!is_directory(dlp->mandir))
		  		continue;
		  	if (converted == 1)
		  		strcpy(dlp->bin, dlp->mandir);
		  	if (debug)
		  		fprintf (stderr, 
		  		  "found global db mandir %s mapped to %s\n",
		  		  dlp->mandir, dlp->bin);
		  	dlp++;
		} else 
			gripe_reading_mp_config (CONFIG_FILE);
		
	}

	dlp->bin[0] = dlp->mandir[0] = '\0';
	dlp->mandatory = 0;
	fclose(config);

	return 0;
}

/*
 * Construct the default manpath.  This picks up mandatory manpaths
 * only.
 */
char *def_path (int perrs, int flag)
{
	int len = 0;
	char *manpathlist; 
	DIRLIST *dlp;

	dlp = list;
	while (dlp->mandatory == flag){
		  len += strlen (dlp->mandir) + 1;
		  dlp++;
	}

	manpathlist = (char *) malloc (len);
	if (manpathlist == NULL)
		gripe_alloc (len, "manpathlist");

	*manpathlist = '\0';

	dlp = list;
	while (dlp->mandatory == flag){
	
		int status;

		status = is_directory (dlp->mandir);

		if (status < 0 && perrs){
			fprintf (stderr, "Warning: couldn't stat file %s!\n", dlp->mandir);
		}
		else if (status == 0 && perrs){
			fprintf (stderr, "Warning: standard directory %s doesn't exist!\n",
			  dlp->mandir);
		}
		else if (status == 1){
			strcat(manpathlist, dlp->mandir);
			strcat(manpathlist, ":"); 
			dlp++;
		}
	}
	return manpathlist;
}

/*
 * For each directory in the user's path, see if it is one of the
 * directories listed in the manpath.config file.  If so, and it is
 * not already in the manpath, add it.  If the directory is not listed
 * in the manpath.config file, see if there is a subdirectory `man' or
 * `MAN'.  If so, and it is not already in the manpath, add it.
 * Example:  user has $HOME/bin in his path and the directory
 * $HOME/bin/man exists -- the directory $HOME/bin/man will be added
 * to the manpath.
 */
char *get_manpath (int perrs, char *path)
{
	int len;
	char *tmppath;
	char *t;
	char *p;
	char **lp;
	char *end;
	char *manpathlist;
	DIRLIST *dlp;
	void add_dir_to_list ();
	char *has_subdirs ();

	tmppath = strdup (path);

	for (p = tmppath;; p = end + 1){
	
		if ((end = strchr (p, ':')))
			*end = '\0';

		if (debug)
			fprintf (stderr, "\npath directory %s ", p);

		/*
      		 * The directory we're working on is in the config file.
      		 * If we haven't added it to the list yet, do.
      		 */

		for (dlp = list; dlp->mandir[0] != '\0'; dlp++)
			if (dlp->bin[0] != '\0' && !strcmp (p, dlp->bin)){
				if (debug)
					fprintf (stderr, "is in the config file\n");

				 add_dir_to_list (tmplist, dlp->mandir, perrs);
				 goto found;
			}

		/*
      		 * The directory we're working on isn't in the config file.  See
      		 * if it has man or MAN subdirectories.  If so, and it hasn't
      		 * been added to the list, do.
      		 */

		 if (debug)
			fprintf (stderr, "is not in the config file\n");

		 t = has_subdirs (p);
		 if (t != NULL){
			if (debug)
				fprintf (stderr, "but it does have a man or MAN subdirectory\n");

			 add_dir_to_list (tmplist, t, perrs);
			 free (t);
		  }
		  else {
			if (debug)
				fprintf (stderr, "and doesn't have man or MAN subdirectories\n");
		  }

	found:

		  if (!end)
			break;
	}

	if (debug)
		fprintf (stderr, "\nadding mandatory man directories\n\n");

	dlp = list;
	while (dlp->mandatory == MANDATORY){
		add_dir_to_list (tmplist, dlp->mandir, perrs);
		dlp++;
	}

	len = 0;
	lp = tmplist;
	while (*lp != NULL){
		len += strlen (*lp) + 1;
		lp++;
	}

	manpathlist = (char *) malloc (len);
	if (manpathlist == NULL)
		gripe_alloc (len, "manpathlist");

	*manpathlist = '\0';

	lp = tmplist;
	p = manpathlist;
	while (*lp != NULL){
		len = strlen (*lp);
		memcpy (p, *lp, len);
		p += len;
		*p++ = ':';
		lp++;
	}

	p[-1] = '\0';

	return manpathlist;
}

/*
 * Add a directory to the manpath list if it isn't already there.
 */
void add_dir_to_list (char **lp, char *dir, int perrs)
{
	int status;

	while (*lp != NULL){

		if (!strcmp (*lp, dir)){
			if (debug)
				fprintf (stderr, "%s is already in the manpath\n", dir);
			return;
		}
		lp++;
	}

	/*
  	 * Not found -- add it.
  	 */
  	 
	status = is_directory (dir);

	if (status < 0 && perrs){
		fprintf (stderr, "Warning: couldn't stat file %s!\n", dir);
	}
	else if (status == 0 && perrs){
		fprintf (stderr, "Warning: %s isn't a directory!\n", dir);
	}
	else if (status == 1){
		if (debug)
			fprintf (stderr, "adding %s to manpath\n", dir);

		*lp = strdup (dir);
	}
}

/*
 * Check to see if the current directory has man or MAN
 * subdirectories.
 */
char *has_subdirs (char *p)
{
	int len;
	char *t;

	len = strlen (p);

	if ((t = (char *) malloc (len + 5)) == NULL)
		gripe_alloc (len + 5, "p\n");

	strcpy(t, p);
	strcpy(t + len, "/man");

	if (is_directory (t) == 1)
		return t;

	strcpy (t + len, "/MAN");

	if (is_directory (t) == 1)
		return t;

	free (t);
	return NULL;
}
