/*
 * util.c
 *
 * Copyright (c) 1990, 1991, John W. Eaton.
 *
 * You may distribute under the terms of the GNU General Public
 * License as specified in the file COPYING that comes with the man
 * distribution.
 *
 * John W. Eaton
 * jwe@che.utexas.edu
 * Department of Chemical Engineering
 * The University of Texas at Austin
 * Austin, Texas  78712
 *
 * Wed May  4 15:44:47 BST 1994 Wilf. (G.Wilford@ee.surrey.ac.uk): slight
 * changes to all routines, mainly cosmetic.
 *
 */

#define MANPATH_MAIN    /* to not define *std_sections[] */

#include <errno.h>
#include <stdio.h>
#include <string.h>
#include <ctype.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <sys/wait.h>
#include <unistd.h>

#ifdef STDC_HEADERS
#include <stdlib.h>
#else
extern int fprintf ();
extern int tolower ();
#endif

/* extern char *strdup ();
extern int system (); */

#include "config.h"
#include "gripes.h"

uid_t ruid;			/* initial real user id */
uid_t euid;			/* initial effective user id */

/*
 * Extract last element of a name like /foo/bar/baz.
 */
char *mkprogname (char *s)
{
	char *t;

	s = ( (t = strrchr (s, '/')) == NULL) ? s : ++t;

	return strdup (s);
}

/*
 * Is file a newer than file b?
 *
 * case:
 *
 *   a is man_page, b is cat_page
 *
 *   a newer than b         returns    1/3  (ret & 1) == 1
 *   a older than b         returns    0/2  !(ret & 1) == 1
 *   a is zero in length    returns    + 2 (for Wilf. and his stray cats)
 *   b is zero in length    returns    + 4
 *   stat on a fails        returns   -1
 *   stat on b fails        returns   -2
 *   stat on a and b fails  returns   -3
 */
int is_newer (char *fa, char *fb)
{
	struct stat fa_sb;
	struct stat fb_sb;
	int fa_stat;
	int fb_stat;
	int status = 0;
	extern int debug;

	if (debug)
		fprintf(stderr, "is_newer: a=%s, b=%s", fa, fb);

	fa_stat = stat (fa, &fa_sb);
	if (fa_stat != 0)
		status = 1;

	fb_stat = stat (fb, &fb_sb);
	if (fb_stat != 0)
		status |= 2;

	if (status != 0) {
		if (debug)
			fprintf(stderr, " (%d)\n", -status);
		return -status;
	}

	if (fa_sb.st_size == 0)
		status |= 2;		

	if (fb_sb.st_size == 0)
		status |= 4;
	if (debug)
		fprintf(stderr, " (%d)\n", status | (fa_sb.st_mtime > fb_sb.st_mtime));
	return (status | (fa_sb.st_mtime > fb_sb.st_mtime));
}

/*
 * Is path a directory?
 */
int is_directory (char *path)
{
	struct stat sb;
	int status;

	status = stat (path, &sb);

	if (status != 0)
		return status;

	return S_ISDIR(sb.st_mode);
}

/*
 * Attempt a system () call.  Return 1 for success and 0 for failure
 * (handy for counting successes :-).
 */
int do_system_command (char *command)
{
	int status = 0;
	extern int debug;

  	/*
  	 * If we're debugging, don't really execute the command -- you never
  	 * know what might be in that mangled string :-O.
  	 */
  	 
	if (debug)
		fprintf (stderr, "\ntrying command: %s\n", command);
	else
		status = system (command);

	/*
  	 * Ultrix returns 127 for failure.  Is this normal?
  	 *
  	 * 127 is usually returned if unsuccessful to fire up a shell.
  	 * Most OS's return != 0 for failure, use that instead, Wilf.
  	 */
  	 
	if (status != 0){
/*		gripe_system_command (status); */
		return 0;
	}
	else
		return 1;
}

/* 
 * If you want to try to understand the following routines, go make 
 * a coffee, you'll need it! (a copy of your kernel sources may also 
 * be handy :-)
 */
 
/* 
 * function to gain user privs by either (a) dropping effective privs 
 * completely (saved ids) or (b) reversing euid w/ uid.
 * Ignore if superuser.
 */
void drop_effective_privs (void)
{
#ifdef SECURE_MAN_UID
	if (!ruid)
		return;
#  if defined (POSIX_SAVED_IDS)
	if (seteuid (ruid)) {
		fputs ("Problems setting effective uid.\n", stderr);
		exit (1);
	}
#  elif defined(SYSV_SAVED_IDS)
	if (setuid (ruid)) {
		fputs ("Problems setting effective uid.\n", stderr);
		exit (1);
	}
#  elif defined(HAVE_SETREUID) /* fallback (could just be #else ) */
	if (setreuid (euid, ruid)) {
		fputs ("Problems swapping real & effective uid.\n", stderr);
		exit (1);
	}
#  endif
#else /* SECURE_MAN_UID */
	/* nothing to do */
#endif
}

/* 
 * function to (re)gain setuid privs by (a) setting euid from suid or (b)
 * (re)reversing uid w/ euid. Ignore if superuser.
 */
void regain_effective_privs (void)
{
#ifdef SECURE_MAN_UID
	if (!ruid)
		return;
#  if defined(POSIX_SAVED_IDS)
	if (seteuid (euid)) {
		fputs ("Problems setting effective uid.\n", stderr);
		exit (1);
	}
#  elif defined(SYSV_SAVED_IDS)
	if (setuid (euid)) {
		fputs ("Problems setting effective uid.\n", stderr);
		exit (1);
	}
#  elif defined(HAVE_SETREUID) /* fallback (could just be #else ) */
	if (setreuid (ruid, euid)) {
		fputs ("Problems swapping real & effective uid.\n", stderr);
		exit (1);
	}
#  endif
#endif /* SECURE_MAN_UID */
	/* nothing to do */
}

/* 
 * If we want to execute a system command with no effective priveledges
 * we have to either
 * 	(a) Use saved id's (if available) to completely drop effective 
 * 	    priveledges and re-engage them after the call.
 *	(b) fork() and then drop effective privs in the child. Do the 
 * 	    system() command from the child and wait for it to die.
 * (b) does not need saved ids as, once dropped, the effective privs are 
 * not required in the child again. (a) does not require a fork() as the
 * system()'d processes will not have suid=MAN_OWNER and will be unable 
 * to gain any man derived priveledges.
 *
 * Obviously (a) is favoured, but there are many implementations...
 * some broken :-(
 */
int do_system_command_drop_privs (char *command)
{
#ifdef SECURE_MAN_UID
	extern int debug;
	int status;
#  if defined(POSIX_SAVED_IDS) || defined(SYSV_SAVED_IDS)

	/* if root or we already dropped privs, just do it */
	if (!ruid || geteuid() == ruid) {
		return do_system_command(command);
	} else {
		drop_effective_privs();
		if (debug)
			system("id");
		status = do_system_command(command);
		regain_effective_privs();
		return status;
	}
#  elif defined(HAVE_SETREUID) /* fallback (could just be #else ) */
	pid_t child;

	/* if root, just do it */
	if (!ruid) {
		return do_system_command(command);
	} else {
		child = vfork ();
		if (child < 0) {
			perror ("can't fork");
			status = 0;
		} else if (child == 0) {
			if (setreuid (ruid, ruid)) { 
				fputs ("Problems dropping real and effective privs",
				  stderr);
				exit(1);
			}
			if (debug)
				system("id");
			exit (do_system_command (command));
		} else
			wait (&status);
		return status;
	}
#  endif
#else 
	return do_system_command(command);
#endif /* SECURE_MAN_UID */
}
