/*
 * jccoefct.c
 *
 * Copyright (C) 1994, Thomas G. Lane.
 * This file is part of the Independent JPEG Group's software.
 * For conditions of distribution and use, see the accompanying README file.
 *
 * This file contains the coefficient buffer controller for compression.
 * This controller is the top level of the JPEG compressor proper.
 * The coefficient buffer lies between forward-DCT and entropy encoding steps.
 */

#define JPEG_INTERNALS
#include "jinclude.h"
#include "jpeglib.h"


/* Private buffer controller object */

typedef struct {
  struct jpeg_c_coef_controller pub; /* public fields */

  JDIMENSION MCU_row_num;	/* keep track of MCU row # within image */

  /* When not doing entropy optimization, it's sufficient to buffer just
   * one MCU (although this may prove a bit slow in practice).  We allocate a
   * workspace of MAX_BLOCKS_IN_MCU coefficient blocks, and reuse it for each
   * MCU constructed and sent.  On 80x86, the workspace is FAR even though
   * it's not really very big; this is to keep the module interfaces unchanged
   * when a large coefficient buffer is necessary.
   */
  JBLOCKROW MCU_buffer[MAX_BLOCKS_IN_MCU];
} my_coef_controller;

typedef my_coef_controller * my_coef_ptr;


/*
 * Initialize for a processing pass.
 */

METHODDEF void
start_pass (j_compress_ptr cinfo, J_BUF_MODE pass_mode)
{
  my_coef_ptr coef = (my_coef_ptr) cinfo->coef;

  switch (pass_mode) {
  case JBUF_PASS_THRU:
    break;
  default:
    ERREXIT(cinfo, JERR_BAD_BUFFER_MODE);
    break;
  }
  coef->MCU_row_num = 0;
}


/*
 * Process some data.
 *
 * NB: input_buf contains a plane for each component in scan.
 */

METHODDEF void
compress_data (j_compress_ptr cinfo,
	       JSAMPIMAGE input_buf, JDIMENSION *in_mcu_ctr)
{
  my_coef_ptr coef = (my_coef_ptr) cinfo->coef;
  JDIMENSION MCU_col_num;	/* index of current MCU within row */
  JDIMENSION last_MCU_col = cinfo->MCUs_per_row - 1;
  JDIMENSION last_MCU_row = cinfo->MCU_rows_in_scan - 1;
  int blkn, bi, ci, yindex, blockcnt;
  JDIMENSION ypos, xpos;
  jpeg_component_info *compptr;

  /* Loop to write as much as one whole MCU row */

  for (MCU_col_num = *in_mcu_ctr; MCU_col_num <= last_MCU_col; MCU_col_num++) {
    /* Determine where data comes from in input_buf and do the DCT thing.
     * Each call on forward_DCT processes a horizontal row of DCT blocks
     * as wide as an MCU; we rely on having allocated the MCU_buffer[] blocks
     * sequentially.  Dummy blocks at the right or bottom edge are filled in
     * specially.  The data in them does not matter for image reconstruction,
     * so we fill them with values that will encode to the smallest amount of
     * data, viz: all zeroes in the AC entries, DC entries equal to previous
     * block's DC value.  (Thanks to Thomas Kinsman for this idea.)
     */
    blkn = 0;
    for (ci = 0; ci < cinfo->comps_in_scan; ci++) {
      compptr = cinfo->cur_comp_info[ci];
      if (MCU_col_num < last_MCU_col)
	blockcnt = compptr->MCU_width;
      else
	blockcnt = compptr->last_col_width;
      xpos = MCU_col_num * compptr->MCU_sample_width;
      ypos = 0;
      for (yindex = 0; yindex < compptr->MCU_height; yindex++) {
	if (coef->MCU_row_num < last_MCU_row ||
	    yindex < compptr->last_row_height) {
	  (*cinfo->fdct->forward_DCT) (cinfo, compptr,
				       input_buf[ci], coef->MCU_buffer[blkn],
				       ypos, xpos, (JDIMENSION) blockcnt);
	  if (blockcnt < compptr->MCU_width) {
	    /* Create some dummy blocks at the right edge of the image. */
	    jzero_far((void FAR *) coef->MCU_buffer[blkn + blockcnt],
		      (compptr->MCU_width - blockcnt) * SIZEOF(JBLOCK));
	    for (bi = blockcnt; bi < compptr->MCU_width; bi++) {
	      coef->MCU_buffer[blkn+bi][0][0] = coef->MCU_buffer[blkn+bi-1][0][0];
	    }
	  }
	} else {
	  /* Create a whole row of dummy blocks at the bottom of the image. */
	  jzero_far((void FAR *) coef->MCU_buffer[blkn],
		    compptr->MCU_width * SIZEOF(JBLOCK));
	  for (bi = 0; bi < compptr->MCU_width; bi++) {
	    coef->MCU_buffer[blkn+bi][0][0] = coef->MCU_buffer[blkn-1][0][0];
	  }
	}
	blkn += compptr->MCU_width;
	ypos += DCTSIZE;
      }
    }
    /* Try to write the MCU.  In event of a suspension failure, we will
     * re-DCT the MCU on restart (a bit inefficient, could be fixed...)
     */
    if (! (*cinfo->entropy->encode_mcu) (cinfo, coef->MCU_buffer))
      break;			/* suspension forced; exit loop */
  }
  if (MCU_col_num > last_MCU_col)
    coef->MCU_row_num++;	/* advance if we finished the row */
  *in_mcu_ctr = MCU_col_num;
}


/*
 * Initialize coefficient buffer controller.
 */

GLOBAL void
jinit_c_coef_controller (j_compress_ptr cinfo, boolean need_full_buffer)
{
  my_coef_ptr coef;
  JBLOCKROW buffer;
  int i;

  coef = (my_coef_ptr)
    (*cinfo->mem->alloc_small) ((j_common_ptr) cinfo, JPOOL_IMAGE,
				SIZEOF(my_coef_controller));
  cinfo->coef = (struct jpeg_c_coef_controller *) coef;
  coef->pub.start_pass = start_pass;
  coef->pub.compress_data = compress_data;

  /* Create the coefficient buffer. */
  if (need_full_buffer) {
    ERREXIT(cinfo, JERR_BAD_BUFFER_MODE);
  } else {
    buffer = (JBLOCKROW)
      (*cinfo->mem->alloc_large) ((j_common_ptr) cinfo, JPOOL_IMAGE,
				  MAX_BLOCKS_IN_MCU * SIZEOF(JBLOCK));
    for (i = 0; i < MAX_BLOCKS_IN_MCU; i++) {
      coef->MCU_buffer[i] = buffer + i;
    }
  }
}
