/*
 * top.c		- show top CPU processes
 *
 * Copyright (c) 1992 Branko Lankester
 * Copyright (c) 1992 Roger Binns
 *
 * Snarfed and HEAVILY modified for the YAPPS (yet another /proc ps)
 * by Michael K. Johnson, johnsonm@sunsite.unc.edu.  What is used is what
 * is required to have a common interface.
 *
 * Modified Michael K Johnson's ps to make it a top program.
 * Also borrowed elements of Roger Binns kmem based top program.
 * Changes made by Robert J. Nation (nation@rocket.sanders.lockheed.com)
 * 1/93
 *
 * Modified by Michael K. Johnson to be more efficient in cpu use
 * 2/21/93
 *
 * Changed top line to use uptime for the load average.  Also
 * added SIGTSTP handling.  J. Cowley, 19 Mar 1993.
 *
 * Fixed kill buglet brought to my attention by Rob Hooft.
 * Problem was mixing of stdio and read()/write().  Added
 * getnum() to solve problem.
 * 12/30/93 Michael K. Johnson
 *
 * Added toggling output of idle processes via 'I' key.
 * 3/29/94 Gregory K. Nickonov
 *
 * Fixed buglet where rawmode wasn't getting restored.
 * Added defaults for signal to send and nice value to use.
 * 5/4/94 Jon Tombs.
 */

#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <time.h>
#include <sys/ioctl.h>
#include <pwd.h>
#include <linux/sched.h>
#include <linux/tty.h>
#include <termcap.h>
#include <termios.h>
#include <signal.h>
#include <sys/time.h>
#include <sys/resource.h>
#include <ctype.h>
#include <setjmp.h>
#include "sysinfo.h"
#include "ps.h"
#include "whattime.h"

/********************************************************************
 * this structure stores some critical information from one frame to
 * the next
 ********************************************************************/
struct save_hist {
  int ticks;
  int pid;
  int pcpu;
  int utime;
  int stime;
};
struct save_hist new_save_hist[NR_TASKS];

/********************************************************************
 * The header printed at the topo of the process list
 ********************************************************************/
char *hdr="  PID USER     PRI  NI SIZE  RES SHRD STAT %CPU %MEM  TIME COMMAND";

int showIdle	= 1;

/********************************************************************
 * Misc function declarations
 ********************************************************************/
void do_setup();
void end();
void stop();
void window_size();
void clear_screen();
void show_procs(struct ps_proc_head *ph, unsigned int maxcmd);
float get_elapsed_time();
unsigned int show_meminfo();
void do_stats(struct ps_proc_head *ph, float elapsed_time,int pass);
void show_task(struct ps_proc *this, unsigned int main_mem, int pcpu);
void do_key(char c);



/********************************************************************
 * Data used to control screen formatting
 ********************************************************************/
struct termio savetty;
struct termio rawtty;
char *cl, *clrtobot,*clrtoeol, *ho;
int lines,cols,maxlines;
int display_procs;
unsigned int maxcmd;
jmp_buf redraw_jmp;

/********************************************************************
 * Controls how long we sleep between screen updates
 ********************************************************************/
float sleeptime = 5;

/* Our pid */

int ourpid ;
   

/********************************************************************
 * Main procedure. Contains loop which displays tasks, and then
 * sleeps/waits for input
 ********************************************************************/
int main(int argc, char **argv)
{
  struct ps_proc_head *ph;
  struct timeval tv;
  fd_set in;
  char c;
  float elapsed_time;
  char *p;

  ourpid = getpid();
  /* get command line args */
  if (argc > 1) {
    for (p = argv[1]; *p; ++p) {
      switch (*p) {
      case 'q':
	if(getuid() == 0) {
	  setpriority(PRIO_PROCESS,ourpid,-15);
	}
	sleeptime = 0.0;
	break;
	  case 'i':
	   showIdle = 1 - showIdle;
	   break;
      }
    }
  }


  /* get termcap info, set screen into raw mode */
  do_setup();
  
  /* find out the window size, set hooks to catch window resize events */
  window_size();


  /* first time through, just collect process stats */
  ph = take_snapshot(1, 1, 1, 1, 0, 0, 0);
  elapsed_time = get_elapsed_time();
  do_stats(ph,elapsed_time,0);
  sleep(1);

  /* loop, collecting process info and sleeping */
  while(1)
    {
      if (setjmp(redraw_jmp) != 0)
	clear_screen();

      /* display the tasks */
      show_procs(ph, maxcmd);
      
      /* sleep & wait for keyboard input */
      tv.tv_sec = sleeptime;
      tv.tv_usec = (sleeptime - (int)sleeptime) * 1000000;
      FD_ZERO(&in);
      FD_SET(0, &in);
      if (select(16, &in, 0, 0, &tv) > 0)
	if (read(0, &c, 1) == 1)
	  do_key(c);
    }
}


/********************************************************************
 * get termcap info for screen clearing, etc.
 * set terminal to raw mode 
 ********************************************************************/
void do_setup()
{
  char *buffer=0;
  char *termtype=getenv("TERM");
  struct termio newtty;

  /* set terminal mode */
  if(!termtype)
    { 
      printf("TERM not set\n"); 
      exit(1);
    }
  close(0);
  if(open("/dev/tty", O_RDONLY))
    printf("stdin is not there\n");
  if(ioctl(0, TCGETA, &savetty) == -1)
    {
      printf("stdin must be a tty\n");
      exit(1);
    }
  signal(SIGHUP, end);
  signal(SIGINT, end);
  signal(SIGTSTP, stop);
  newtty=savetty;
  newtty.c_lflag&=~ICANON;
  newtty.c_lflag&=~ECHO;
  newtty.c_cc[VMIN]=1;
  newtty.c_cc[VTIME]=0;
  if(ioctl(0, TCSETAF, &newtty)==-1)
    {
      printf("cannot put tty into raw mode\n");
      exit(1);
    }
  ioctl(0, TCGETA, &rawtty);

  /* get termcap entries */
  tgetent(buffer, termtype);
  clrtobot=tgetstr("cd", 0);
  cl=tgetstr("cl",0);
  clrtoeol=tgetstr("ce", 0);
  ho=tgetstr("ho",0);}


/********************************************************************
 * This procedure should be called to exit the program.
 * It restores the terminal to its original state 
 ********************************************************************/
void end()
{
  int i;

  ioctl(0, TCSETAF, &savetty);
  printf("%s", ho);
  for (i=0; i< lines-1; i++) 
    printf("\n");
  exit(0);
}

void stop()
{
  int i;
  
  ioctl(0, TCSETAF, &savetty);
  printf("%s", ho);
  for (i=0; i < lines - 3; i++)
    printf("\n");
  fflush(stdout);
  raise(SIGTSTP);
  ioctl(0, TCSETAF, &rawtty);
  signal(SIGTSTP, stop);
  longjmp(redraw_jmp, 1);
}


/********************************************************************
 * reads the window size, clears the window, and sets itself up
 * to catch window resize events
 ********************************************************************/
void window_size()
{
  struct winsize ws;

  cols = lines = 0;
  if (ioctl(1, TIOCGWINSZ, &ws) != -1) 
    {
      cols = ws.ws_col;
      lines = ws.ws_row;
    } 
  if (!cols) {
    cols = tgetnum("co");
  }
  if (!lines) {
    lines = tgetnum("li");
  }
  if(display_procs == 0)
    maxlines=lines-7;
  else
    maxlines = display_procs;
  signal(SIGWINCH, window_size);
  maxcmd = cols - strlen(hdr) + 7;
  clear_screen();
}



/********************************************************************
 * clears the screen
 ********************************************************************/
void clear_screen()
{
  printf("%s",cl);
}


/********************************************************************
 * This is the real program!
 * It reads process info and displays it.
 ********************************************************************/
void show_procs(struct ps_proc_head *ph, unsigned int maxcmd)
{
  struct ps_proc *this,*best;
  int count,top;
  int index,best_index;
  float elapsed_time;
  unsigned int main_mem;

  /* display the load averages */
  printf("%s%s%s\n", ho, sprint_uptime(), clrtoeol);

  /* get the process info */
  ph = refresh_snapshot(ph, 1, 1, 1, 1, 0, 0, 0);
  /* immediately find out the elapsed time for the frame */
  elapsed_time = get_elapsed_time();
  
  /* display the system stats, also calculate percent CPU time */
  do_stats(ph,elapsed_time,1);

  /* display the memory and swap space usage */
  main_mem = show_meminfo();
  printf("%s%s",hdr,clrtoeol);
  
  /* finally! loop through to find the top task, and display it */
  count = 0;
  top = 100;
  while((count < maxlines)&&(top >= 0))
    {
      /* find the top of the remaining processes */
      top=-1;
      this = ph->head;
      best = this;
      best_index = 0;
      index=0;
      while(this !=NULL)
	{
	  if(new_save_hist[index].pcpu>top)
	    {
	      top = new_save_hist[index].pcpu;
	      best = this;
	      best_index = index;
	    }
	  index++;
	  this = this->next;
	}
      count++;
      if(top>=0)
	{
	  /* display the process */
	  show_task(best,main_mem,new_save_hist[best_index].pcpu);
	}
      new_save_hist[best_index].pcpu=-1;
    }
  printf("%s%s\n\n\n\n\n",clrtobot,ho);

  /* make sure that the screen is updated */
  fflush(stdout);
}

/********************************************************************
 * Finds the current time (in microseconds) and finds the time
 * elapsed since the last update. This is essential for computing
 * percent CPU usage.
 ********************************************************************/
float get_elapsed_time()
{
  struct timeval time;
  static struct timeval oldtime;
  struct timezone timez;
  float elapsed_time;

  gettimeofday(&time,&timez);
  elapsed_time = (time.tv_sec - oldtime.tv_sec) +
    (float)(time.tv_usec - oldtime.tv_usec)/1000000.0;
  oldtime.tv_sec = time.tv_sec;
  oldtime.tv_usec= time.tv_usec;
  return elapsed_time;
}



/********************************************************************
 * Reads the memory info and displays it 
 * returns the total memory available for use in percent memory 
 * usage calculations.
 ********************************************************************/
unsigned int show_meminfo()
{
  char memory[1024];
  static int fd;
  unsigned int main_mem, used_mem, free_mem, shared_mem, buf_mem;
  unsigned int swap_mem, used_swap, free_swap;

  fd = open("/proc/meminfo", O_RDONLY, 0);
  if (fd == -1) 
    {
      perror("ps.c:/proc/meminfo");
      end();
    }
  read(fd,memory,sizeof(memory)-1);
  close(fd);
  sscanf(memory, "%*s %*s %*s %*s %*s %*s %u %u %u %u %u %*s %u %u %u",
	 &main_mem, &used_mem, &free_mem, &shared_mem, &buf_mem,
	 &swap_mem, &used_swap, &free_swap);
  printf("Mem:  %5dK av, %5dK used, %5dK free, %5dK shrd, %5dK buff%s\n",
	 main_mem/1024, used_mem/1024, free_mem/1024, 
	 shared_mem/1024, buf_mem/1024,clrtoeol);
  printf("Swap: %5dK av, %5dK used, %5dK free%s\n%s\n",
	 swap_mem/1024, used_swap/1024, free_swap/1024,clrtoeol,
	 clrtoeol);
  return main_mem;
}


/********************************************************************
 * Calculates the number of tasks in each state (running, sleeping, etc.)
 * Calculates the CPU time in each state (system, user, nice, etc)
 * calculates percent cpu usage for each task 
 ********************************************************************/
void do_stats(struct ps_proc_head *ph,float elapsed_time,int pass)
{
  struct ps_proc *this;
  int index,total_time,i;
  int sleeping = 0,stopped = 0,zombie = 0,running = 0;
  int system_ticks = 0,user_ticks = 0,nice_ticks = 0,idle_ticks = 1000;
  static int prev_count=0;
  static struct save_hist save_hist[NR_TASKS];
  int stime, utime;
  /* make sure that there aren't too many tasks */
  if(ph->count >NR_TASKS)
    {
      printf("Help! Too many tasks!");
      end();
    }

  /* make a pass through the data to get stats */
  index=0;
  this = ph->head;
  while(this != NULL)
    {
      if((this->state == 'S')||(this->state == 'D'))
	sleeping++;
      else if(this->state == 'T')
	stopped++;
      else if(this->state == 'Z')
	zombie++;
      else if(this->state == 'R')
	running++;

      /* calculate time in this process */
      /* time is sum of user time (utime) plus system time (stime) */
      total_time = this->utime + this->stime;
      new_save_hist[index].ticks = total_time;
      new_save_hist[index].pid = this->pid;
      stime = this->stime;
      utime = this->utime;
      new_save_hist[index].stime = stime;
      new_save_hist[index].utime = utime;
      /* find matching entry from previous pass*/
      i=0;
      while(i<prev_count)
	{
	  if(save_hist[i].pid == this->pid)
	    {
	      total_time -= save_hist[i].ticks;
	      stime -= save_hist[i].stime;
	      utime -= save_hist[i].utime;

	      i = NR_TASKS;
	    }
	  i++;
	}
      /* calculate percent cpu time for this task */
      new_save_hist[index].pcpu = (total_time * 10) /elapsed_time;
      if (new_save_hist[index].pcpu > 999)
	new_save_hist[index].pcpu = 999;

      /* calculate time in idle, system, user and niced tasks */
      idle_ticks -= new_save_hist[index].pcpu;
      system_ticks += stime;
      user_ticks += utime;
      if(this->priority < PZERO)
	nice_ticks += new_save_hist[index].pcpu;

      index++;
      this = this->next;
    }

  if(idle_ticks < 0)
    idle_ticks = 0;
  system_ticks = (system_ticks * 10) /elapsed_time;      
  user_ticks = (user_ticks * 10) /elapsed_time;

  /* display stats */
  if(pass>0)
    {
      printf("%d processes: %d sleeping, %d running, %d zombie, %d stopped%s\n",
	     ph->count,sleeping,running,zombie,stopped,clrtoeol);
      printf("CPU states: %2d.%d%% user, %2d.%d%% nice,",
	     user_ticks/10, user_ticks%10,
	     nice_ticks/10, nice_ticks%10);
      printf(" %2d.%d%% system, %2d.%d%% idle%s\n",
	     system_ticks/10, system_ticks%10,
	     idle_ticks/10, idle_ticks%10,clrtoeol);
    }

  /* save this frame's information */
  for(i=0;i<ph->count;i++)
    {
      /* copy the relevant info for the next pass */
      save_hist[i].pid = new_save_hist[i].pid;
      save_hist[i].ticks = new_save_hist[i].ticks;
      save_hist[i].stime = new_save_hist[i].stime;
      save_hist[i].utime = new_save_hist[i].utime;
    }
  prev_count = ph->count;
}


/********************************************************************
 * get a number from the user
 ********************************************************************/
int getnum(void) {
  char line[20];
  int i=0;
  int r, j;

  /* must make sure that buffered IO doesn't kill us */
  fflush(stdout);
  fflush(stdin);
  do {
    read(STDIN_FILENO, &line[i], 1);
  } while ((line[i++] != '\n') && (i < 20));

  line[--i]=(char)0; /* make sscanf happy */

  /* now check sanity */
  for (j=0;j<i;j++) {
    if(!isdigit(line[j])) {
      return (-1);
    }
  }

  if(i == 0) return (-1); /* so an empty line is an error */
  sscanf(line, "%d", &r);
  return(r);
}



/********************************************************************
 * displays information relevant to each task
 ********************************************************************/
void show_task(struct ps_proc *this,unsigned int main_mem, int pcpu)
{
  int pmem;
  unsigned int t;
  char *cmdptr;

  cmdptr = status( this );
  
  if( !showIdle && (*cmdptr == 'S' || *cmdptr == 'Z') )
  	 return;
  	 
  /* show task info */
  pmem = this->rss * 1000 / (main_mem / 4096);
  printf("\n%5d %-8s %3d %3d %4d %4d %4d %s %2d.%d %2d.%d", 
	 this->pid,this->user,2*PZERO-this->counter,PZERO-this->priority,
	 this->vsize/1024,this->rss*4,this->statm.share << 2,
	 status(this), pcpu / 10, pcpu % 10, pmem / 10, pmem % 10);

  /* show total CPU time */
  t = (this->utime + this->stime) / HZ;
  printf("%3d:%02d ", t / 60, t % 60);
  
  /* show command line */
  if(strlen(this->cmdline) > 0)
    cmdptr = this->cmdline;
  else
    cmdptr=this->cmd;
  if (strlen(cmdptr) > maxcmd)
    cmdptr[maxcmd - 1] = 0;
  printf("%s%s",cmdptr,clrtoeol);
}


/********************************************************************
 * processes keyboard input
 ********************************************************************/
void do_key(char c)
{
  int pid,signal,val;
  char junk;

  c = toupper(c);

  /* first the commands which don't require a terminal mode switch */
  /* 'Q' or 'q' for quit */
  if(c=='Q')
    end();
  /* ^L to clear and re-draw the screen */
  else if(c==12)
    {
      clear_screen();
      return;
    }

  /* switch the terminal to normal mode */
  ioctl(0,TCSETA,&savetty);

  /* process other inputs */
  switch (c) {
  /* 'N' or 'n' or '#' to change the number of processes displayed */
    case 'N':
    case '#':
      printf("Display how many processes? ");
      if ((display_procs = getnum()) != -1)
        maxlines = display_procs;
      break;
  /* 'i' or 'I' toggles output of idle tasks */
    case 'I':
      showIdle	= 1 - showIdle;
      break;
  /* 's' or 'S' to change the sleep time */
    case 'S':
      printf("Delay between updates: ");
      if ((val = getnum()) != -1)
	sleeptime = (float) val;
      break;
  /* 'k' or 'K' to kill a task */
    case 'K':
      printf("Kill PID: ");
      pid = getnum();
      if (pid == -1) break;
      printf("%s\n\n\n\n\n%sKill PID %d with signal: ",ho,clrtoeol,pid);
      signal = getnum();
      if (signal == -1) signal = 2;
      if(kill(pid,signal))
	{
	  printf("%s\n\n\n\n\n%s\007Kill of PID %d failed: %s",ho,clrtoeol,
		 pid,strerror(errno));
	  fflush(stdout);
	  sleep(3);
	}
      break;
  /* 'r' or 'R' to renice a task */
    case 'R':
      printf("Renice PID: ");
      pid = getnum();
      if (pid == -1) break;
      printf("%s\n\n\n\n\n%sRenice PID %d to value: ",ho,clrtoeol,pid);
      val = getnum();
      if (pid == -1) val = 10;
      if(setpriority(PRIO_PROCESS,pid,val))
	{
	  printf("%s\n\n\n\n\n%s\007Renice of PID %d failed: %s",ho,clrtoeol,
		 pid,strerror(errno));
	  fflush(stdout);
	  sleep(3);
	}
      break;
  /* 'h' or 'H' or '?' to get help */
    case 'H':
    case '?':
      printf("%s\n\t\t\tProc-Top Revision 0\n\n",cl);
      printf("Interactive commands are:\n\n"
	     "? or h   Print this list\n"
	     "i        Toggle output of IDLE processes\n"
	     "k        Kill a task. You will be prompted for the PID and signal\n"
	     "r        Renice a task. You will be prompted for the PID and"
	     " nice level\n"
	     "s        Set the delay in seconds between updates\n"
	     "n        Set the number of process to show\n"
	     "^L       Redraw the screen\n"
	     "q        Quit\n"
	     "\n\n\t\t\tPress RETURN to continue\n");
      scanf("%c",&junk);
      break;
    }
  /* back to raw mode */
  ioctl(0,TCSETA,&rawtty);
  return;
}
